/**
 * @file Platform/Win32/Receiver.h
 *
 * This file declares classes related to receivers.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */
#ifndef __RECEIVER_H__
#define __RECEIVER_H__

#ifndef __ProcessFramework_h__
#error Never include this file directly. Include ProcessFramework.h instead.
#endif

#include "Tools/Streams/InStreams.h"

/**
 * The class is the base class for receivers.
 * A receiver is an object that reads packages from Aperios queues.
 * The class manages a global list of all receivers in an Aperios process.
 */
class ReceiverList
{
  private:
    ReceiverList* next;         /**< The successor of the current receiver. */
    char name[NAME_LENGTH_MAX]; /**< The name of a receiver without the module's name. */
  
  protected:
    PlatformProcess* process;   /**< The process this receiver is associated with. */
    int eventId;                /**< The id of the current receiver in the range [0..30]. */
    bool blocking;              /**< Stores whether this is a blocking receiver. */
    void* package;              /**< A received package. */

    /**
     * The function checks whether a new package has arrived.
     */    
    virtual void checkForPackage() = 0;

  public:
    /**
     * The constructor.
     * @param process The process this receiver is associated with.
     * @param receiverName The Aperios connection name of the receiver without the process name.
     * @param blocking Decides whether this receiver blocks the execution of the next frame
     *                 until it has received a package.
     */
    ReceiverList(PlatformProcess* process,const char* receiverName,bool blocking);

    /**
     * Destructor.
     */
    ~ReceiverList()
    {
      if(package)
        delete [] package;
    }

    /**
     * Returns the begin of the list of all receivers.
     * @return The first receiver in the list, or 0 if the list ist empty.
     */
    ReceiverList*& getFirst();

    /**
     * Returns the next receiver in the list.
     * @return The next receiver in the list, or 0 if this receiver is the last one.
     */
    ReceiverList* getNext() const {return next;}

    /**
     * Returns the Aperios connection name of the receiver.
     * @return The Aperios connection name without the process name ("Receiver.type.O")
     */
    const char* getName() const {return name;}

    /**
     * Returns whether a new package was received in the current frame.
     * This is always true if this is a blocking receiver.
     * @return Has a new package been received?
     */
    bool receivedNew() const;

    /**
     * The function must be called to finish the current frame.
     */
    void finishFrame();

    /**
     * The function checks for all receivers whether a new package has arrived.
     */    
    void checkAllForPackages();

    /**
     * The function sets the package.
     * @param p The package.
     */
    void setPackage(void* p)
    {
      package = p;
    }

    /**
     * The function determines whether the receiver has a pending package.
     * @return Is there still an unprocessed package?
     */
    bool hasPendingPackage() const {return package != 0;}

    /**
     * The function searches for a receiver with the given name.
     * @param processName The name of the receiver's process.
     * @param receiverName The name of the requested receiver.
     * @return If the receiver is found, a pointer to it is returned.
     *         Otherwise, the funtion returns 0.
     */ 
    ReceiverList* lookup(const char* processName,const char* receiverName);
};

/**
 * The class implements a receiver.
 * A receiver is an object that reads packages from an Aperios queue.
 */
template<class T> class Receiver : public ReceiverList, public T
{
  private:
    /**
     * The function checks whether a new package has arrived.
     */    
    virtual void checkForPackage()
    {
      if(package && !receivedNew())
      {
        T& data = *static_cast<T*>(this);
        InBinaryMemory memory(package);
        memory >> data;
        process->setEventId(eventId);
      }
    }

  public:
    /**
     * The constructor.
     * @param process The process this receiver is associated with.
     * @param receiverName The Aperios connection name of the receiver without the process name.
     * @param blocking Decides whether this receiver blocks the execution of the next frame
     *                 until it has received a package.
     */
    Receiver(PlatformProcess* process,const char* receiverName,bool blocking)
    : ReceiverList(process,receiverName,blocking)
    {
    }
};


#endif

/*
 * Change log :
 *
 * $Log: Receiver.h,v $
 * Revision 1.1.1.1  2004/05/22 17:24:00  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.1  2002/09/10 15:40:05  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/07/23 13:39:39  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - new design of debugging architecture
 * - exchanged StaticQueue with MessageQueue
 *
 * Revision 1.1.1.1  2002/05/10 12:40:18  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.6  2002/04/21 16:09:02  roefer
 * Anchor removed
 *
 * Revision 1.5  2002/01/19 23:52:30  roefer
 * "Receiving same package several times" fixed
 *
 * Revision 1.4  2002/01/14 22:50:28  loetzsch
 * SimRobot / RobotControl integration started
 *
 *
 * Revision 1.3  2001/12/10 17:47:08  risler
 * change log added
 *
 */
