/**
 * @file Platform/Win32/Oracle2.h
 *
 * This file declares the magic Oracle that knows everything.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#ifndef __Oracle_H__
#define __Oracle_H__

#include "Representations/Cognition/RobotPose.h"
#include "Representations/Cognition/PlayerPoseCollection.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Cognition/ObstaclesModel.h"
#include "Tools/Field.h"

class SimObject;

/**
 * This the magic Oracle that knows everything.
 * Derivations for real world applications will be implemented later.
 */
class Oracle
{
  private:
    static Field field; /**< A model of the field. */

    /**
     * The function determines whether the specified object is blue.
     * @param obj The object that is checked for being blue.
     * @return Is the object red?
     */
    static bool isBlue(SimObject* obj);

    /**
     * The function inserts all robots in a group into a player pose collection.
     * @param ppc The player pose collection.
     * @param me The robot that is watching.
     * @param objs The robot group that shall be added.
     */
    static void addPlayers(PlayerPoseCollection& ppc, SimObject* me, SimObject* objs);

    /**
     * The function inserts a robot into a player pose collection.
     * @param ppc The player pose collection.
     * @param me The robot that is watching.
     * @param obj The robot that shall be added.
     */
    static void addPlayer(PlayerPoseCollection& ppc, SimObject* me, SimObject* obj);

  public:
    /**
     * The function returns the pose of a robot.
     * @param me The SimRobot object of the robot.
     * @return The robot's current pose.
     */
    static RobotPose getRobotPose(SimObject* me);

    /**
     * The function returns the player pose collection for a robot.
     * @param me The SimRobot object of the robot.
     * @return The player pose collection.
     */
    static PlayerPoseCollection getPlayerPoseCollection(SimObject* me);

    /**
     * The function returns the ball's position for a robot.
     * @param me The SimRobot object of the robot.
     * @return The ball's position.
     */
    static BallModel getBallModel(SimObject* me);

    /**
     * The function returns the obstacle model for a robot.
     * @param pose The pose of the robot.
     * @param ppc The poses of the other players.
     * @return The obstacle model.
     */
    static ObstaclesModel getObstaclesModel(const RobotPose& pose, const PlayerPoseCollection& ppc);

    /**
     * The function returns the camera matrix for a robot.
     * @param sensorData All joint angles.
     * @return The camera matrix.
     */
    static CameraMatrix getCameraMatrix(const SensorData& sensorData);

    /**
     * The function answers any other question.
     * @return The correct answer to any question...
     */
    static int getTheAnswerToEverythingElse() {return 42;}
};

#endif


/*
 * Change log :
 * 
 * $Log: Oracle2.h,v $
 * Revision 1.3  2004/05/27 09:55:38  roefer
 * Oracled camera matrix and camera info
 *
 * Revision 1.2  2004/05/22 22:52:04  juengel
 * Renamed ballP_osition to ballModel.
 *
 * Revision 1.1.1.1  2004/05/22 17:23:56  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.4  2004/02/03 13:19:49  spranger
 * renamed all references to  class BallP_osition to BallModel
 *
 * Revision 1.3  2003/10/31 08:36:18  dueffert
 * doxygen bugs fixed
 *
 * Revision 1.2  2003/10/23 15:41:40  roefer
 * Oracled obstacle model added
 *
 * Revision 1.1  2003/10/21 15:05:51  roefer
 * Added the oracle
 *
 */
