/**
 * @file Platform/Win32/Oracle.cpp
 *
 * This file implements the magic Oracle that knows everything.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#include "RoboCupCtrl.h"
#include "Tools/Player.h"
#include "Oracle.h"

Field Oracle::field;

RobotPose Oracle::getRobotPose(unsigned currentFrameNumber)
{
  RoboCupCtrl& ctrl = *RoboCupCtrl::getController();
  SIM3DOBJECT me = ctrl.getSimRobotObject();
  RobotPose rp;
  rp.setPose(ctrl.getPose2D(me));
  if(getPlayer().getTeamColor() == Player::blue)
    rp.setPose(Pose2D(double(pi)) + rp.getPose());
  rp.setValidity(1);
  rp.frameNumber = currentFrameNumber;
  return rp;
}

PlayerPoseCollection Oracle::getPlayerPoseCollection(unsigned currentFrameNumber)
{
  RoboCupCtrl& ctrl = *RoboCupCtrl::getController();
  SIM3DOBJECT me = ctrl.getSimRobotObject();
  PlayerPoseCollection ppc;

  // search active robots
  SIM3DOBJECT obj = ((SIM3DGroup*) ctrl.GetObject("robots"))->FirstSubObject();
  if(obj)
  {
    SIM3DOBJECT objFirst = obj;
    do
    {
      addPlayer(ppc,me,obj,currentFrameNumber);
      obj = obj->Next;
    }
    while(obj != objFirst);
  }
  // search passive robots
  obj = ((SIM3DGroup*) ctrl.GetObject("extras"))->FirstSubObject();
  if(obj)
  {
    SIM3DOBJECT objFirst = obj;
    do
    {
      addPlayer(ppc,me,obj,currentFrameNumber);
      obj = obj->Next;
    }
    while(obj != objFirst);
  }
  return ppc;
}

BallModel Oracle::getBallModel(unsigned currentFrameNumber)
{
  RoboCupCtrl& ctrl = *RoboCupCtrl::getController();
  SIM3DOBJECT me = ctrl.getSimRobotObject();
  BallModel bp;
  SIM3DOBJECT obj = ((SIM3DGroup*) ctrl.GetObject("balls"))->FirstSubObject();
  if(obj)
  {
    SIM3DOBJECT objFirst = obj;
    do
    {
      Pose2D pose = RoboCupCtrl::getController()->getPose2D(obj);
      if(getPlayer().getTeamColor() == Player::blue)
        pose = (Pose2D(pi) + pose);
      (Vector2<double>&) bp.seen = pose.translation;
      bp.seen.timeWhenLastSeen = SystemCall::getCurrentSystemTime();
      (Vector2<double>&) bp.communicated = pose.translation;
      bp.communicated.timeWhenLastObserved = bp.seen.timeWhenLastSeen;
      bp.validity = 1.0;
      bp.frameNumber = currentFrameNumber;
      bp.seen.timeUntilSeenConsecutively = SystemCall::getCurrentSystemTime();
      bp.seen.timeWhenFirstSeenConsecutively = 0;
      obj = obj->Next;
    }
    while(obj != objFirst);
  }
  return bp;
}

bool Oracle::isBlue(SIM3DElement* obj)
{
  obj = obj->FirstSubObject();
  return obj->Color != 9;
}

void Oracle::addPlayer(PlayerPoseCollection& ppc,SIM3DElement* me,SIM3DElement* obj,unsigned currentFrameNumber)
{
  if(obj != me)
  {
    PlayerPose pp;
    pp.setPose(RoboCupCtrl::getController()->getPose2D(obj));
    if(getPlayer().getTeamColor() == Player::blue)
      pp.setPose(Pose2D(double(pi)) + pp.getPose());
    pp.setValidity(1);
     pp.setFrameNumber(currentFrameNumber);

    if(isBlue(obj) == (getPlayer().getTeamColor() == Player::blue))
    {
      if(ppc.numberOfOwnPlayers < 3)
        ppc.setOwnPlayerPose(ppc.numberOfOwnPlayers++,pp);
    }
    else if(ppc.numberOfOpponentPlayers < 4)
      ppc.setOpponentPlayerPose(ppc.numberOfOpponentPlayers++,pp);
  }
}

ObstaclesModel Oracle::getObstaclesModel(const RobotPose& pose, const PlayerPoseCollection& ppc)
{
  field.placePlayers(ppc);
  ObstaclesModel obstaclesModel;
  int i;

  // set sector distances
  for(i = 0; i < ObstaclesModel::numOfSectors; i++)
  {
    double distance = field.getObstacleDistance(pose + Pose2D(ObstaclesModel::getAngleOfSector(i)), obstaclesModel.obstacleType[i]);
    if(distance > 0)
      obstaclesModel.distance[i] = (int)distance;
    else
    {
      obstaclesModel.distance[i] = ObstaclesModel::maxDistance;
      obstaclesModel.obstacleType[i] = ObstaclesPercept::unknown;
    }
  }

  // calculate corridor
  double closest = ObstaclesModel::maxDistance; 
  for(i = ObstaclesModel::getSectorFromAngle(-pi_2); i < ObstaclesModel::getSectorFromAngle(pi_2); i++)
  {
    Vector2<double> v = (pose + Pose2D(ObstaclesModel::getAngleOfSector(i)) + 
                         Pose2D(Vector2<double>(obstaclesModel.distance[i], 0))).translation;
    if(v.x < closest && v.x > 10 && v.y < 90 && v.y > -90)
      closest = v.x;
  }
  obstaclesModel.corridorInFront = (int) closest;
  obstaclesModel.frameNumber = pose.frameNumber;
  return obstaclesModel;  
}


/*
 * Change log :
 * 
 * $Log: Oracle.cpp,v $
 * Revision 1.2  2004/05/22 22:52:03  juengel
 * Renamed ballP_osition to ballModel.
 *
 * Revision 1.1.1.1  2004/05/22 17:23:56  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.8  2004/03/29 18:40:21  juengel
 * Added timeUntilSeenConsecutively and timeWhenFirstSeenConsecutively to ballModel
 *
 * Revision 1.7  2004/03/01 11:46:31  juengel
 * Moved enum ObstacleType to class ObstaclesPercept.
 *
 * Revision 1.6  2004/02/28 14:00:43  juengel
 * Added ObstacleType to ObstaclesModel.
 *
 * Revision 1.5  2004/02/03 13:20:48  spranger
 * renamed all references to  class BallP_osition to BallModel (possibly changed include files)
 *
 * Revision 1.4  2003/11/18 16:38:39  goehring
 * PlayerPose setFrameNumber
 *
 * Revision 1.3  2003/11/18 15:44:31  dueffert
 * Oracle now cares about frame numbers
 *
 * Revision 1.2  2003/10/23 15:41:40  roefer
 * Oracled obstacle model added
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.11  2003/05/01 19:32:04  roefer
 * Ball position assignment issues solved
 *
 * Revision 1.10  2003/05/01 18:10:03  loetzsch
 * renamed CommunicatedBallPosition::timeWhenLastReceived to CommunicatedBallP_osition::timeWhenLastObserved
 *
 * Revision 1.9  2003/05/01 17:09:08  loetzsch
 * Redesign of ball modeling:
 * - Modularized class BallP_osition
 * - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
 *   and "TeamBallLocator" for modelling communicated positions
 * - Removed solution JumpingBallLocator
 * - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
 * - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
 *
 * Revision 1.8  2003/04/15 15:52:09  risler
 * DDD GO 2003 code integrated
 *
 * Revision 1.8  2003/03/27 14:33:37  max
 * oracle sets ball validity
 *
 * Revision 1.7  2003/02/11 14:30:12  thomas
 * bug fix: ball.timesincelastknow was not set to current time by oracle
 *
 * Revision 1.6  2003/01/30 13:16:49  loetzsch
 * Redesign of class BallP_osition
 *
 * Revision 1.5  2002/11/19 15:43:03  dueffert
 * doxygen comments corrected
 *
 * Revision 1.4  2002/10/10 14:59:21  roefer
 * getBallP_osition corrected
 *
 * Revision 1.3  2002/09/25 12:05:44  loetzsch
 * removed BarPercept and PatternPercept
 *
 * Revision 1.2  2002/09/22 18:40:55  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.1  2002/09/10 15:40:04  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.7  2002/06/28 10:26:20  roefer
 * OUTPUT is possible in constructors
 *
 * Revision 1.6  2002/06/22 02:59:17  risler
 * no message
 *
 * Revision 1.5  2002/06/17 19:24:14  mkunz
 * BarPerceptor searches for own end only now
 *
 * Revision 1.4  2002/06/13 12:20:21  risler
 * missing rotate in GetBarPosition
 *
 * Revision 1.3  2002/06/13 11:43:15  risler
 * bar oracle
 *
 * Revision 1.2  2002/06/07 09:29:05  kosen
 * Oracle paints now multipleBalls.
 *
 * Revision 1.1.1.1  2002/05/10 12:40:18  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.10  2002/04/25 20:29:58  roefer
 * New BallPercept and BallP_osition, GTMath errors in SimGT2002 fixed
 *
 * Revision 1.9  2002/04/20 15:52:20  roefer
 * Project simpified, WATCH and WATCH_PART added
 *
 * Revision 1.8  2002/02/11 11:13:20  roefer
 * BallPerceptor and BallLocator inserted
 *
 * Revision 1.7  2002/02/06 18:57:58  roefer
 * Simplifications for oracle and debug drawings
 *
 * Revision 1.6  2002/02/05 14:04:30  roefer
 * Oracle has optional parameter for referred robot
 *
 * Revision 1.5  2002/02/05 04:12:40  loetzsch
 * replaced direct member access in WorldState
 * and PerceptCollection members by
 * inline const VALUE& get...() const   and
 * inline void set...(const Value&) methods.
 *
 * Revision 1.4  2002/02/02 17:30:55  roefer
 * replaced intpi by pi
 *
 * Revision 1.3  2002/01/27 22:14:54  roefer
 * BallP_osition is now a pair of two vectors
 *
 * Revision 1.2  2002/01/25 16:10:48  roefer
 * Comments inserted
 *
 * Revision 1.3  2001/12/10 17:47:07  risler
 * change log added
 *
 */
