/**
* @file Xabsl2View.cpp
*
* Implementation of class Xabsl2View
*
* @author <a href="mailto:roefer.tzi.de">Thomas Rfer</a>
*/

#include "Xabsl2View.h"

double Xabsl2View::leftOfColumnPercent[10] =
{
  0, 0.25, 0.5, 0.7, 0.05, 0.2, 0.35, 0.6, 0.8, 0.85
};

Xabsl2View::Xabsl2View(RobotConsole& c, const Xabsl2Info& i)
: console(c),
  info(i)
{
  normalFont.CreateFont(14,0,0,0,FW_NORMAL,FALSE,FALSE,0,ANSI_CHARSET,
    OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,
    DEFAULT_PITCH | FF_SWISS,"Arial");

  boldFont.CreateFont(14,0,0,0,FW_BOLD,FALSE,FALSE,0,ANSI_CHARSET,
    OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,
    DEFAULT_PITCH | FF_SWISS,"Arial");
}

Xabsl2View::~Xabsl2View()
{
  boldFont.DeleteObject();
  normalFont.DeleteObject();
}

void Xabsl2View::draw(CDC& dc)
{
  this->dc = &dc;
  line = 0;
  CFont* oldFont = dc.SelectObject(&normalFont);
  if(GetDeviceCaps(dc.m_hDC,TECHNOLOGY) != DT_METAFILE)
  {
    width = dc.GetViewportExt().cx;
    TEXTMETRIC tm;
    dc.GetTextMetrics(&tm);
    lineHeight = tm.tmHeight;
  }
  else
  {
    width = 400;
    lineHeight = 14;
  }
  dc.SetMapMode(MM_TEXT);
  dc.SetBkMode(TRANSPARENT);
  SYNC_WITH(console);
  print("Agent:",0,true);
  print(info.selectedBehaviorControlName + " - " + info.selectedAgentName,1);
  newLine(true);

  print("Option Activation Path:",0,true);
  newLine(true);
  for(List<Xabsl2Info::ActiveOption>::Pos i = info.activeOptions.getFirst(); i; ++i)
  {
    const Xabsl2Info::ActiveOption& ao = info.activeOptions[i];
    print(ao.name,0);
    print(ao.durationOfActivation + " s",6);
    print(ao.activeState,2);
    print(ao.durationOfStateActivation + " s",9);
    for(List<Xabsl2Info::NameValue>::Pos j = ao.parameters.getFirst(); j; ++j)
    {
      const Xabsl2Info::NameValue& aop = ao.parameters[j];
      newLine();
      if(aop.name.compare(0,ao.name.length(),ao.name))
        print(aop.name,4);
      else
        print(aop.name.substr(ao.name.length() + 1),4);
      print(aop.value,3);
    }
    newLine(true);
  }

  print("Active Basic Behavior:",0,true);
  print(info.activeBasicBehaviorName,2);
  for(List<Xabsl2Info::NameValue>::Pos j = info.activeBasicBehaviors.getFirst(); j; ++j)
  {
    const Xabsl2Info::NameValue& abb = info.activeBasicBehaviors[j];
    newLine();
    if(abb.name.compare(0,info.activeBasicBehaviorName.length(),info.activeBasicBehaviorName))
      print(abb.name,4);
    else
      print(abb.name.substr(info.activeBasicBehaviorName.length() + 1),4);
    print(abb.value,3);
  }
  newLine(true);
  
  print("Motion Request:",0,true);
  print(info.executedMotionRequest,2);
  newLine(true);

  print("Output Symbols:",0,true);
  for(List<Xabsl2Info::OutputSymbol>::Pos k = info.outputSymbols.getFirst(); k; ++k)
  {
    const Xabsl2Info::OutputSymbol& os = info.outputSymbols[k];
    if(os.show)
    {
      newLine();
      print(os.name,10);
      if(os.value.compare(0,os.name.length(),os.name))
        print(os.value,2);
      else
        print(os.value.substr(os.name.length() + 1),2);
    }
  }
  newLine(true);

  print("Input Symbols:",0,true);
  for(List<Xabsl2Info::InputSymbol>::Pos l = info.inputSymbols.getFirst(); l; ++l)
  {
    const Xabsl2Info::InputSymbol& is = info.inputSymbols[l];
    if(is.show)
    {
      newLine();
      print(is.name,10);
      print(is.value,2);
    }
  }

  dc.SelectObject(oldFont);
}

void Xabsl2View::print(const std::string& s, int column, bool bold)
{
  dc->SelectObject(bold ? &boldFont : &normalFont);
  if(column < 10)
    dc->TextOut((int)(5 + (width - 10) * leftOfColumnPercent[column]),
                line, s.c_str());
  else
    dc->TextOut(5 + column, line, s.c_str());
}

void Xabsl2View::newLine(bool drawLine)
{
  line += lineHeight;
  if(drawLine)
  {
    dc->MoveTo(0,line);
    dc->LineTo(width,line);
  }
}

/*
 * Change log :
 * 
 * $Log: Xabsl2View.cpp,v $
 * Revision 1.2  2004/06/19 08:04:24  roefer
 * Removed crash if Xabsl view is too large
 *
 * Revision 1.1.1.1  2004/05/22 17:24:45  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2004/04/12 10:07:03  roefer
 * Removed repetition of names of options, basic behaviors and output symbols in parameters/values
 *
 * Revision 1.2  2003/10/20 14:05:53  roefer
 * Views and buttons
 *
 * Revision 1.1  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 *
 * Revision 1.1  2003/10/07 10:07:00  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1  2003/09/26 11:34:08  juengel
 * no message
 *
 * Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.1  2003/04/03 21:13:25  roefer
 * Xabsl2 view added
 *
 *
 */


