/**
* @file Xabsl2Info.h
*
* Declaration of class Xabsl2Info
*
* @author <a href="mailto:roefer.tzi.de">Thomas Rfer</a>
*/

#ifndef __Xabsl2Info_h_
#define __Xabsl2Info_h_

#include "Tools/List.h"
#include <string>
#include "Tools/MessageQueue/InMessage.h"

class Xabsl2Info;

/**
 * Streaming operator that reads a Xabsl2Info from a stream.
 * @param stream The stream from which is read.
 * @param info The Xabsl2Info object.
 * @return The stream.
 */ 
In& operator>>(In& stream,Xabsl2Info& info);
 
/**
 * Streaming operator that writes a Xabsl2 debug request to a stream.
 * @param stream The stream from which is read.
 * @param info The Xabsl2Info object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream,const Xabsl2Info& info);

/**
* @class Xabsl2Info 
* 
* A class to represent information about a Xabsl2 behavior.
*
* @author <a href="mailto:roefer.tzi.de">Thomas Rfer</a>
*/
class Xabsl2Info
{
  public:
    /** The type of an input symbol. */
    enum InputSymbolType {decimal, boolean, enumerated};

    /** A pair of a name and a value. */
    struct NameValue
    {
      std::string name,
                  value;
    };

    /** An input symbol. */
    struct InputSymbol : public NameValue
    {
      bool show; /**< Show this symbol? */
      InputSymbolType type; /**< The type of the symbol. */
    };

    /** An output symbol. */
    struct OutputSymbol : public NameValue
    {
      bool show; /**< Show this symbol? */
      List<std::string> alternatives; /**< The alternatives to set this symbol to. */
      List<std::string>::Pos selectedAlternative; /**< The currently selected alternative. */
    };

    /** An option. */
    struct Option
    {
      std::string name; /**< The name of the option. */
      List<std::string> parameters; /**< The names of the parameters. */
    };

    /** A basic behavior. */
    struct BasicBehavior
    {
      std::string name; /**< The name of the basic behavior. */
      List<std::string> parameters; /**< The names of the parameters. */
    };

    /** An active option. */
    struct ActiveOption
    {
      std::string name, /**< The name of the active option. */
                  durationOfActivation, /**< The duration this option is active. */
                  activeState, /**< The active state. */
                  durationOfStateActivation; /**< The duration the state is active. */
      List<NameValue> parameters; /**< The parameters of the option. */
    };

    std::string selectedBehaviorControlName, /**< The name of the behavior. */
                selectedAgentName, /**< The name of the agent. */
                activeBasicBehaviorName, /**< The name of the current basic behavior. */
                executedMotionRequest; /**< The motion request currently executed. */
    List<InputSymbol> inputSymbols; /**< The input symbols. */
    List<OutputSymbol> outputSymbols; /**< The output symbols. */
    List<Option> options; /**< The options. */
    List<BasicBehavior> basicBehaviors; /**< The basic behaviors. */
    List<ActiveOption> activeOptions; /**< The active options. */
    List<NameValue> activeBasicBehaviors; /**< The active basic behaviors. */
    List<Option>::Pos selectedOption; /**< The option selected by the user. */
    List<BasicBehavior>::Pos selectedBasicBehavior; /**< The basic behavior selected by the user. */
    double parameters[10]; /**< The parameters of the basic behavior selected. */

    /**
    * The function handles a Xabsl2 debug message.
    * @param message The message.
    * @return Is the message compatible with the behavior loaded?
    */
    bool handleMessage(InMessage& message);

    /**
    * The function empties the Xabsl2 info object.
    */
    void reset();

  private:
    /**
    * The function finds an input symbol by its name.
    * Note: the function throws an exception if the symbol is not found.
    * @param search The name of the symbol to search for.
    * @return A reference to the symbol found.
    */
    InputSymbol& findInputSymbol(const std::string& search);

    /**
    * The function finds an output symbol by its name.
    * Note: the function throws an exception if the symbol is not found.
    * @param search The name of the symbol to search for.
    * @return A reference to the symbol found.
    */
    OutputSymbol& findOutputSymbol(const std::string& search);

  friend In& operator>>(In& stream,Xabsl2Info& info);
  friend Out& operator<<(Out& stream,const Xabsl2Info& info);
};

#endif //__Xabsl2Info_h_

/*
 * Change log :
 * 
 * $Log: Xabsl2Info.h,v $
 * Revision 1.2  2004/06/18 13:45:10  roefer
 * 10 parameters for xbb and xo instead of 5
 *
 * Revision 1.1.1.1  2004/05/22 17:24:45  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.1  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 *
 */
