/**
* @file Platform/Win32/ForSimRobXP/Xabsl2Info.cpp
*
* Implementation of class Xabsl2Info
*
* @author <a href="mailto:roefer.tzi.de">Thomas Rfer</a>
*/

#include "Xabsl2Info.h"
#include "Tools/Module/SolutionRequest.h"

void Xabsl2Info::reset()
{
  selectedBehaviorControlName = "";
  selectedAgentName = "";
  activeBasicBehaviorName = "";
  executedMotionRequest = "";
  selectedBasicBehavior = List<BasicBehavior>::Pos();
  selectedOption = List<Option>::Pos();
  for(int i = 0; i < 10; ++i)
    parameters[i] = 0.0;
  inputSymbols.clear();
  outputSymbols.clear();
  options.clear();
  basicBehaviors.clear();
  activeOptions.clear();
  activeBasicBehaviors.clear();
}

In& operator>>(In& stream,Xabsl2Info& info)
{
  info.reset();

  char buffer[200];
  int i,
      j,
      count,
      count2;

  stream >> count;
  for(i = 0; i < count; ++i)
  {
    Xabsl2Info::InputSymbol* is = new Xabsl2Info::InputSymbol;
    stream >> buffer;
    switch(buffer[0])
    {
      case 'd':
        is->type = Xabsl2Info::decimal;
        break;
      case 'b':
        is->type = Xabsl2Info::boolean;
        break;
      case 'e':
        is->type = Xabsl2Info::enumerated;
        break;
    }
    stream >> buffer;
    is->name = buffer;
    is->value = "0";
    is->show = false;
    info.inputSymbols.insert(is);
  }

  stream >> count;
  for(i = 0; i < count; ++i)
  {
    Xabsl2Info::OutputSymbol* os = new Xabsl2Info::OutputSymbol;
    stream >> buffer;
    os->name = buffer;
    os->value = "0";
    os->show = false;
    stream >> count2;
    for(j = 0; j < count2; ++j)
    {
      stream >> buffer;
      os->alternatives.insert(new std::string(buffer));
    }
    info.outputSymbols.insert(os);
  }

  stream >> count;
  for(i = 0; i < count; ++i)
  {
    Xabsl2Info::Option* o = new Xabsl2Info::Option;
    stream >> buffer;
    o->name = buffer;
    stream >> count2;
    for(j = 0; j < count2; ++j)
    {
      stream >> buffer;
      o->parameters.insert(new std::string(buffer));
    }
    info.options.insert(o);
  }

  stream >> count;
  for(i = 0; i < count; i++)
  {
    Xabsl2Info::BasicBehavior* bb = new Xabsl2Info::BasicBehavior;
    stream >> buffer;
    bb->name = buffer;
    stream >> count2;
    for(j = 0; j < count2; ++j)
    {
      stream >> buffer;
      bb->parameters.insert(new std::string(buffer));
    }
    info.basicBehaviors.insert(bb);
  }
  
  return stream;
}

bool Xabsl2Info::handleMessage(InMessage& message)
{
  char buffer[200],
       buffer2[200];
  int count,
      i;
  double valueDouble;
  char valueChar;
  unsigned long valueUnsignedLong;

  message.bin >> buffer[0];
  if(selectedBehaviorControlName == SolutionRequest::getXabsl2EngineIDName(SolutionRequest::xabsl2EngineID(buffer[0])))
  {
    activeOptions.clear();
    activeBasicBehaviors.clear();

    message.bin >> buffer >> count;
    selectedAgentName = buffer;
    try
    {
      for(i = 0; i < count; ++i)
      {
        message.bin >> buffer >> valueDouble;
        sprintf(buffer2,"%.2f",valueDouble);
        findInputSymbol(buffer).value = buffer2;
      }

      message.bin >> count; 
      for(i = 0; i < count; ++i)
      {
        message.bin >> buffer >> valueChar;
        findInputSymbol(buffer).value = valueChar ? "true" : "false";
      }

      message.bin >> count; 
      for(i = 0; i < count; ++i)
      {
        message.bin >> buffer >> buffer2;
        findInputSymbol(buffer).value = buffer2;
      }

      message.bin >> count; 
      for(i = 0; i < count; ++i)
      {
        message.bin >> buffer >> buffer2;
        findOutputSymbol(buffer).value = buffer2;
      }
    }
    catch(std::string&) {}
    message.bin >> count;
    for(i = 0; i < count; ++i)
    { 
      ActiveOption* ao = new ActiveOption;
      message.bin >> buffer >> valueUnsignedLong;
      ao->name = buffer;
      sprintf(buffer,"%.1f",valueUnsignedLong / 1000.0);
      ao->durationOfActivation = buffer;
  
      message.bin >> buffer >> valueUnsignedLong;
      ao->activeState = buffer;
      sprintf(buffer,"%.1f",valueUnsignedLong / 1000.0);
      ao->durationOfStateActivation = buffer;

      message.bin >> valueChar;
      for(int j = 0; j < valueChar; ++j)
      {
        NameValue* nv = new NameValue;
        message.bin >> buffer >> valueDouble;
        nv->name = buffer;
        sprintf(buffer,"%.2f",valueDouble);
        nv->value = buffer;
        ao->parameters.insert(nv);
      }
      activeOptions.insert(ao);
    }

    message.bin >> buffer >> valueChar;
    activeBasicBehaviorName = buffer;
    for(i = 0; i < valueChar; ++i)
    {
      NameValue* nv = new NameValue;
      message.bin >> buffer >> valueDouble;
      nv->name = buffer;
      sprintf(buffer,"%.2f",valueDouble);
      nv->value = buffer;
      activeBasicBehaviors.insert(nv);
    }

    message.bin >> buffer;
    executedMotionRequest = buffer;
    return true;
  }
  else
    return false;
}

Xabsl2Info::InputSymbol& Xabsl2Info::findInputSymbol(const std::string& search)
{
  for(List<InputSymbol>::Pos i = inputSymbols.getFirst(); i; ++i)
  {
    InputSymbol& is = inputSymbols[i];
    if(is.name == search)
      return is;
  }
  throw std::string(search) + " not found";
}

Xabsl2Info::OutputSymbol& Xabsl2Info::findOutputSymbol(const std::string& search)
{
  for(List<OutputSymbol>::Pos i = outputSymbols.getFirst(); i; ++i)
  {
    OutputSymbol& os = outputSymbols[i];
    if(os.name == search)
      return os;
  }
  throw std::string(search) + " not found";
}

Out& operator<<(Out& stream,const Xabsl2Info& info)
{
  int k;
  for(k = 0; k < SolutionRequest::undefined && 
      info.selectedBehaviorControlName != SolutionRequest::getXabsl2EngineIDName(SolutionRequest::xabsl2EngineID(k)); ++k) 
    ;
  stream << (char) k;
  int count = 0;
  List<Xabsl2Info::InputSymbol>::Pos i;
  for(i = info.inputSymbols.getFirst(); i; ++i)
    if(info.inputSymbols[i].show)
      ++count;
  stream << count;
  for(i = info.inputSymbols.getFirst(); i; ++i)
  {
    const Xabsl2Info::InputSymbol& is = info.inputSymbols[i];
    if(is.show)
    {
      switch(is.type)
      {
        case Xabsl2Info::decimal: 
          stream << 'd'; 
          break;
        case Xabsl2Info::boolean:
          stream << 'b';
          break;
        case Xabsl2Info::enumerated:
          stream << 'e';
          break;
      }
      stream << is.name.c_str();
    }
  }

  count = 0;
  List<Xabsl2Info::OutputSymbol>::Pos j;
  for(j = info.outputSymbols.getFirst(); j; ++j)
    if(info.outputSymbols[j].show)
      ++count;
  stream << count;
  for(j = info.outputSymbols.getFirst(); j; ++j)
  {
    const Xabsl2Info::OutputSymbol& os = info.outputSymbols[j];
    if(os.show)
      stream << os.name.c_str();
  }

  if(info.selectedBasicBehavior)
  {
    const Xabsl2Info::BasicBehavior& bb = info.basicBehaviors[info.selectedBasicBehavior];
    stream << char('b') << bb.name.c_str() << char(bb.parameters.getSize());
    int j = 0;
    for(List<std::string>::Pos i = bb.parameters.getFirst(); i; ++i)
      stream << bb.parameters[i].c_str() << info.parameters[j++];
  }
  else if(info.selectedOption)
  {
    const Xabsl2Info::Option& o = info.options[info.selectedOption];
    stream << char('o') << o.name.c_str() << char(o.parameters.getSize());
    int j = 0;
    for(List<std::string>::Pos i = o.parameters.getFirst(); i; ++i)
      stream << o.parameters[i].c_str() << info.parameters[j++];
  }
  else
    stream << char('d');

  count = 0;
  for(j = info.outputSymbols.getFirst(); j; ++j)
    if(info.outputSymbols[j].selectedAlternative)
      ++count;

  stream << char(count);
  for(j = info.outputSymbols.getFirst(); j; ++j)
  {
    const Xabsl2Info::OutputSymbol& os = info.outputSymbols[j];
    if(os.selectedAlternative)
      stream << os.name.c_str() << os.alternatives[os.selectedAlternative].c_str();
  }

  return stream;
}

/*
 * Change log :
 * 
 * $Log: Xabsl2Info.cpp,v $
 * Revision 1.3  2004/06/18 13:45:10  roefer
 * 10 parameters for xbb and xo instead of 5
 *
 * Revision 1.2  2004/05/26 11:07:18  roefer
 * Behavior data is only removed if message from correct engine is received
 *
 * Revision 1.1.1.1  2004/05/22 17:24:45  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.4  2004/05/18 09:14:34  roefer
 * Compatibiliy with new Xabsl2 request format
 *
 * Revision 1.3  2004/05/14 22:56:32  roefer
 * Adapted to new Xabsl engine
 *
 * Revision 1.2  2003/10/16 15:30:34  goehring
 * Doxygen
 *
 * Revision 1.1  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 *
 */
