/**
 * @file Platform/Win32/ForSimRobXP/View.h
 *
 * This file declares the view classes and macros.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */
#ifndef __View_H__
#define __View_H__

#include "Tools/List.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Debugging/DebugImages.h"

/**
 * The class is a base class for image views.
 */
class ImageViewBase
{
  public:
    static List<ImageViewBase*> list; /**< The list of all views. */

    /**
     * The function returns information on the view to create.
     * @param name Returns the name of the view.
     * @param layers Returns a list of layer ids.
     * @param numOfLayers Returns the length of the list.
     */
    virtual void getInfo(const char*& name,
                         Drawings::ImageDrawing*& layers,
                         int& numOfLayers,
                         Images::ImageID& id) = 0;

    /**
     * Constructor.
     */
    ImageViewBase() {list.insert(this);}
};

/**
 * The class is a base class for field views.
 */
class FieldViewBase
{
  public:
    static List<FieldViewBase*> list; /**< The list of all views. */

    /**
     * The function returns information on the view to create.
     * @param name Returns the name of the view.
     * @param layers Returns a list of layer ids.
     * @param numOfLayers Returns the length of the list.
     */
    virtual void getInfo(const char*& name,
                         Drawings::FieldDrawing*& layers,
                         int& numOfLayers) = 0;

    /**
     * Constructor.
     */
    FieldViewBase() {list.insert(this);}
};

/**
 * The macro begins the definition of an image view.
 * It must be followed by a list of Drawings::ImageDrawing constants.
 * These constants define the layers that will be drawn in the view.
 * The list must be finished by the END_VIEW macro.
 * @param name The name of the view.
 */
#define IMAGE_VIEW(name) \
  class _View##name : public ImageViewBase, public DebugDrawing\
  { \
    public: \
      static const char* viewName; \
      static Drawings::ImageDrawing layers[]; \
      static int numOfLayers; \
    public: \
    virtual void getInfo(const char*& s,Drawings::ImageDrawing*& l,int& n,Images::ImageID& id) \
        {s = viewName; l = layers + 1; n = numOfLayers; id = Images::name;} \
  } _view##name; \
  const char* _View##name::viewName = #name; \
  Drawings::ImageDrawing _View##name::layers[] = {Drawings::numberOfImageDrawings,

/**
 * The macro begins the definition of a field view.
 * It must be followed by a list of Drawings::FieldDrawing constants.
 * These constants define the layers that will be drawn on the view.
 * The list must be finished by the END_VIEW macro.
 * @param name The name of the view.
 */
#define FIELD_VIEW(name) \
  class _View##name : public FieldViewBase, public DebugDrawing\
  { \
    public: \
      static const char* viewName; \
      static Drawings::FieldDrawing layers[]; \
      static int numOfLayers; \
    public: \
      virtual void getInfo(const char*& s,Drawings::FieldDrawing*& l,int& n) \
        {s = viewName; l = layers + 1; n = numOfLayers;} \
  } _view##name; \
  const char* _View##name::viewName = #name; \
  Drawings::FieldDrawing _View##name::layers[] = {Drawings::numberOfFieldDrawings,

/**
 * The macro finishes the definition of a view.
 * @param name The name of the view. It must be the same as 
 *             used at the beginning of the list.
 */
#define END_VIEW(name) \
  }; \
  int _View##name::numOfLayers = sizeof(_View##name::layers) / sizeof(_View##name::layers[0]) - 1;

#endif

/*
 * Changelog:
 * 
 * $Log: View.h,v $
 * Revision 1.1.1.1  2004/05/22 17:24:34  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.5  2004/04/27 19:39:37  roefer
 * Support for debug images added to simulator
 *
 * Revision 1.4  2003/10/17 15:38:59  roefer
 * Comments for old revisions removed
 *
 * Revision 1.2  2003/10/16 15:30:34  goehring
 * Doxygen
 *
 * Revision 1.1  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 */
