/**
* @file TimeInfo.h
*
* Declaration of class TimeInfo
*
* @author <a href="mailto:roefer.tzi.de">Thomas Rfer</a>
*/

#ifndef __TimeInfo_h_
#define __TimeInfo_h_

#include "Tools/RingBuffer.h"
#include "Tools/Debugging/Stopwatch.h"
#include "Tools/MessageQueue/InMessage.h"

/**
* @class TimeInfo 
* 
* A class to represent information about the timing of modules.
*
* @author <a href="mailto:roefer.tzi.de">Thomas Rfer</a>
*/
class TimeInfo
{
  private:
    enum {ringBufferSize = 100};
    RingBuffer<unsigned long, ringBufferSize> startTime[Stopwatch::numberOfStopwatchEventIDs];
    RingBuffer<unsigned long, ringBufferSize> endTime[Stopwatch::numberOfStopwatchEventIDs];
    RingBuffer<unsigned long, ringBufferSize> counter[Stopwatch::numberOfStopwatchEventIDs];
    unsigned long lastReceived[Stopwatch::numberOfStopwatchEventIDs];

  public:
    TimeInfo()
    {
      reset();
    }

    /**
    * The function handles a stop watch message.
    * @param message The message.
    * @return Was it a stop watch message?
    */
    bool handleMessage(InMessage& message);

    /**
    * The function empties the time info object.
    */
    void reset();

    /**
    * The function empties the time info of a certain stop watch.
    * @param id The id of the stop watch to reset.
    */
    void reset(Stopwatch::StopwatchEventID id);

    /**
    * The function returns statistics about a certain stop watch.
    * @param id The id of the stop watch.
    * @param minTime The shortest measurement is returned to this variable.
    * @param maxTime The longest measurement is returned to this variable.
    * @param avgTime The average measurement is returned to this variable.
    * @param freq The frequency is returned to this variable.
    * @return There were enough values to calculate the data.
    */ 
    bool getStatistics(Stopwatch::StopwatchEventID id, double& minTime, double& maxTime, double& avgTime, double& freq) const;

};

#endif //__TimeInfo_h_

/*
 * Change log :
 * 
 * $Log: TimeInfo.h,v $
 * Revision 1.1.1.1  2004/05/22 17:24:34  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2004/04/12 14:18:47  dueffert
 * case of header file corrected
 *
 * Revision 1.1  2004/04/09 18:55:16  roefer
 * Better structure for views
 * cp command added
 * Timing view added
 *
 * Revision 1.1  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 *
 */
