/**
* @file SensorView.cpp
*
* Implementation of class SensorView
*
* @author <a href="mailto:roefer.tzi.de">Thomas Rfer</a>
*/

#include "SensorView.h"

double SensorView::leftOfColumnPercent[3] =
{
  0, 0.8, 0.8
};

const SensorView::ViewInfo SensorView::viewInfoERS210[] =
{
  {SensorData::neckTilt, SensorView::angle, false},
  {SensorData::headPan, SensorView::angle, false},
  {SensorData::headTilt, SensorView::angle, false},
  {SensorData::mouth, SensorView::angle, true},
  {SensorData::legFL1, SensorView::angle, false},
  {SensorData::legFL2, SensorView::angle, false},
  {SensorData::legFL3, SensorView::angle, false},
  {SensorData::pawFL, SensorView::button, true},
  {SensorData::legHL1, SensorView::angle, false},
  {SensorData::legHL2, SensorView::angle, false},
  {SensorData::legHL3, SensorView::angle, false},
  {SensorData::pawHL, SensorView::button, true},
  {SensorData::legFR1, SensorView::angle, false},
  {SensorData::legFR2, SensorView::angle, false},
  {SensorData::legFR3, SensorView::angle, false},
  {SensorData::pawFR, SensorView::button, true},
  {SensorData::legHR1, SensorView::angle, false},
  {SensorData::legHR2, SensorView::angle, false},
  {SensorData::legHR3, SensorView::angle, false},
  {SensorData::pawHR, SensorView::button, true},
  {SensorData::tailPan, SensorView::angle, false},
  {SensorData::tailTilt, SensorView::angle, true},
  {SensorData::headBack, SensorView::pressure, false},
  {SensorData::headFront, SensorView::pressure, false},
  {SensorData::chin, SensorView::button, false},
  {SensorData::back, SensorView::button, true},
  {SensorData::psd, SensorView::distance, false},
  {SensorData::thermo, SensorView::temperature, true},
  {SensorData::accelerationX, SensorView::acceleration, false},
  {SensorData::accelerationY, SensorView::acceleration, false},
  {SensorData::accelerationZ, SensorView::acceleration, false}
};

const SensorView::ViewInfo SensorView::viewInfoERS7[] =
{
  {SensorData::neckTilt, SensorView::angle, false},
  {SensorData::headPan, SensorView::angle, false},
  {SensorData::neckTilt, SensorView::angle, false},
  {SensorData::mouth, SensorView::angle, true},
  {SensorData::legFL1, SensorView::angle, false},
  {SensorData::legFL2, SensorView::angle, false},
  {SensorData::legFL3, SensorView::angle, false},
  {SensorData::pawFL, SensorView::button, true},
  {SensorData::legHL1, SensorView::angle, false},
  {SensorData::legHL2, SensorView::angle, false},
  {SensorData::legHL3, SensorView::angle, false},
  {SensorData::pawHL, SensorView::button, true},
  {SensorData::legFR1, SensorView::angle, false},
  {SensorData::legFR2, SensorView::angle, false},
  {SensorData::legFR3, SensorView::angle, false},
  {SensorData::pawFR, SensorView::button, true},
  {SensorData::legHR1, SensorView::angle, false},
  {SensorData::legHR2, SensorView::angle, false},
  {SensorData::legHR3, SensorView::angle, false},
  {SensorData::pawHR, SensorView::button, true},
  {SensorData::tailPan, SensorView::angle, false},
  {SensorData::tailTilt, SensorView::angle, true},
  {SensorData::head, SensorView::button, false},
  {SensorData::chin, SensorView::button, false},
  {SensorData::backF, SensorView::button, false},
  {SensorData::backM, SensorView::button, false},
  {SensorData::backR, SensorView::button, false},
  {SensorData::wlan, SensorView::button, true},
  {SensorData::headPsdNear, SensorView::distance, false},
  {SensorData::headPsdFar, SensorView::distance, false},
  {SensorData::bodyPsd, SensorView::distance, true},
  {SensorData::accelerationX, SensorView::acceleration, false},
  {SensorData::accelerationY, SensorView::acceleration, false},
  {SensorData::accelerationZ, SensorView::acceleration, false}
};

SensorView::SensorView(RobotConsole& c, const SensorData& s)
: console(c),
  sensorData(s)
{
  normalFont.CreateFont(14,0,0,0,FW_NORMAL,FALSE,FALSE,0,ANSI_CHARSET,
    OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,
    DEFAULT_PITCH | FF_SWISS,"Arial");

  boldFont.CreateFont(14,0,0,0,FW_BOLD,FALSE,FALSE,0,ANSI_CHARSET,
    OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,
    DEFAULT_PITCH | FF_SWISS,"Arial");
}

SensorView::~SensorView()
{
  boldFont.DeleteObject();
  normalFont.DeleteObject();
}

void SensorView::draw(CDC& dc)
{
  this->dc = &dc;
  line = 0;
  CFont* oldFont = dc.SelectObject(&normalFont);
  if(GetDeviceCaps(dc.m_hDC,TECHNOLOGY) != DT_METAFILE)
  {
    width = dc.GetViewportExt().cx;
    lineHeight = dc.GetTextExtent("M").cy;
  }
  else
  {
    width = 400;
    lineHeight = 14;
  }
  dc.SetMapMode(MM_TEXT);
  dc.SetBkMode(TRANSPARENT);
  SYNC_WITH(console);
  print("Sensor",0,true);
  print("Value",1,true);
  newLine(true);
  if(sensorData.data[SensorData::bodyPsd])
    for(int i = 0 ; i < SensorData::numOfSensor_ERS7; ++i)
    {
      const ViewInfo& vi = viewInfoERS7[i];
      print(SensorData::getSensorNameERS7(vi.sensor), 0);
      printSensorData(vi);
    }
  else
    for(int i = 0 ; i < SensorData::numOfSensor_ERS210; ++i)
    {
      const ViewInfo& vi = viewInfoERS210[i];
      print(SensorData::getSensorNameERS210(vi.sensor), 0);
      printSensorData(vi);
    }
  dc.SelectObject(oldFont);
}

void SensorView::printSensorData(const ViewInfo& vi)
{
  char value[30];
  switch(vi.type)
  {
  case angle:
    sprintf(value, "%.1f", toDegrees(fromMicroRad(sensorData.data[vi.sensor])));
    print("", 2);
    break;
  case distance:
    sprintf(value, "%.1f", sensorData.data[vi.sensor] / 1000.0);
    print(" mm", 2);
    break;
  case pressure:
    sprintf(value, "%.1f", sensorData.data[vi.sensor] / 1000.0);
    print(" mN", 2);
    break;
  case button:
    sprintf(value, "%d", sensorData.data[vi.sensor]);
    break;
  case temperature:
    sprintf(value, "%.1f", sensorData.data[vi.sensor] / 1000000.0);
    print(" C", 2);
    break;
  case acceleration:
    sprintf(value, "%.1f", sensorData.data[vi.sensor] / 1000.0);
    print(" mm/s", 2);
    break;
  }
  print(value, 1);
  newLine(vi.endOfSection);
}

void SensorView::print(const std::string& s, int column, bool bold)
{
  dc->SelectObject(bold ? &boldFont : &normalFont);
  if(column != 1)
    dc->TextOut((int)(5 + (width - 10) * leftOfColumnPercent[column]),
                line, s.c_str());
  else
    dc->TextOut((int)(5 + (width - 10) * leftOfColumnPercent[column] - dc->GetTextExtent(s.c_str()).cx),
                line, s.c_str());
}

void SensorView::newLine(bool drawLine)
{
  line += lineHeight;
  if(drawLine)
  {
    dc->MoveTo(0,line);
    dc->LineTo(width,line);
  }
}

/*
 * Change log :
 * 
 * $Log: SensorView.cpp,v $
 * Revision 1.2  2004/05/27 17:13:37  jhoffman
 * - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
 * - clipping included for setJoints
 * - removed some microrad/rad-bugs
 * - bodyPosture constructor and "=" operator fixed
 *
 * Revision 1.1.1.1  2004/05/22 17:24:33  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2004/04/14 15:52:35  roefer
 * Temperatur seems to be measured in C, so display  C
 *
 * Revision 1.1  2004/04/12 18:00:45  roefer
 * SensorView added
 *
 */
