/** 
* @file Platform/Win32/ForSimRobXP/RobotConsole.h
*
* Declaration of RobotConsole.
*
* @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
*/

#ifndef __RobotConsole_H__
#define __RobotConsole_H__

#include "Controller/Controller.h"
#include "Tools/Process.h"
#include "Platform/Sensors.h"
#include <fstream>
#include "Representations/Motion/OdometryData.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "Representations/Perception/BallPercept.h"
#include "Representations/Perception/PlayersPercept.h"
#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Perception/PSDPercept.h"
#include "Representations/Perception/LinesPercept.h"
#include "Representations/Perception/BodyPercept.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/SpecialPercept.h"
#include "Representations/Perception/EdgesPercept.h"
#include "Representations/Perception/ColorTable64.h"
#include "Representations/Perception/CollisionPercept.h"
#include "Representations/Cognition/RobotPose.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Cognition/PlayerPoseCollection.h"
#include "Representations/Cognition/ObstaclesModel.h"
#include "Representations/Cognition/RobotState.h"
#include "Representations/RoboCup/GameControlData.h"
#include "Representations/Motion/MotionRequest.h"

#include "Tools/Player.h"
#include "Tools/MessageQueue/MessageQueue.h"
#include "Tools/MessageQueue/LogPlayer.h"
#include "Tools/Debugging/DebugImages.h"
#include "RobotControl/Visualization/DebugDrawing.h"
#include "Xabsl2Info.h"
#include "TimeInfo.h"

class ConsoleRoboCupCtrl;

/**
* @class RobotConsole
*
* A process that simulates the Aperios RobotConsole.
*/
class RobotConsole : public Process
{
public:
  DECLARE_SYNC; /**< Make this object synchronizable. */
  
protected:
  ConsoleRoboCupCtrl* ctrl; /** A pointer to the controller object. */
  Player::teamColor teamColor; /**< The color of the robots team. */
  bool printMessages, /**< Decides whether to output text messages in the console window. */
       calculateImage, /**< Decides whether images are calculated by the simulator. */
       sendOracle; /**< Decides whether an oracled world state is sent to the processes. */
  LogPlayer logPlayer; /**< The log player to record and replay log files. */
  
  MessageQueue& debugIn, /**< The incoming debug queue. */
              & debugOut; /**< The outgoing debug queue. */
private:
  std::string robotName; /**< The name of the robot. */
  Image* images[Images::numberOfImageIDs]; /** All debug images (created on demand). */
  ColorTable64 colorTable; /**< A color table for classifying images. */
  RobotPose robotPose; /**< A buffer for a robot pose from the debug queue */
  RobotPose oracledRobotPose; /**< A buffer for a oracled robot pose from the debug queue */
  BallModel ballModel; /**< A buffer for a ball model from the debug queue */
  BallModel oracledBallModel; /**< A buffer for a oracled ball position from the debug queue */
  RobotState robotState;/**< A buffer for a robot state from the debug queue */
  RobotState oracledRobotState;/**< A buffer for a oracled robot state from the debug queue */
  PlayerPoseCollection playerPoseCollection; /**< A buffer for a player pose collection from the debug queue */
  ObstaclesModel obstaclesModel; /**< A buffer for a obstacles model from the debug queue */
  PlayerPoseCollection oracledPlayerPoseCollection; /**< A buffer for a oracled player pose collection from the debug queue */
  CameraMatrix cameraMatrix; /**< A buffer for a camera matrix from the debug queue */
  LandmarksPercept landmarksPercept; /**< A buffer for a landmarks percept from the debug queue */
  BallPercept ballPercept; /**< A buffer for a ball percept from the debug queue */
  PlayersPercept playersPercept; /**< A buffer for a players percept from the debug queue */
  ObstaclesPercept obstaclesPercept; /**< A buffer for a obstacles percept from the debug queue */
  PSDPercept psdPercept; /**< A buffer for a PSD percept from the debug queue */
  CollisionPercept collisionPercept; /**< A buffer for a joint state percept from the debug queue */
  LinesPercept linesPercept; /**< A buffer for a lines percept from the debug queue */
  BodyPercept bodyPercept; /**< A buffer for a body percept from the debug queue */
  SpecialPercept specialPercept; /**< A buffer for a special percept from the debug queue */
  EdgesPercept edgesPercept; /**< A buffer for an percept of edges from the debug queue */

  DebugDrawing imageDrawings[Drawings::numberOfImageDrawings], /**< Buffers for image drawings from the debug queue. */
    fieldDrawings[Drawings::numberOfFieldDrawings]; /**< Buffers for field drawings from the debug queue. */

  DebugDrawing incompleteDrawings[Drawings::numberOfFieldDrawings + Drawings::numberOfImageDrawings]; /**< Buffers incomplete drawings from the debug queue. */

  SolutionRequest sr; /**< The current state of all solution requests. */
  GameControlData gameControlData; /**< The current state of game control data. */
  MotionRequest motionRequest; /**< The current state of the motion request. */
  Xabsl2Info xabsl2Infos[2]; /**< Xabsl symbols. */
  bool xabsl2Error, /**< A flag stating that the loaded behavior is different from the one on the robot. */
       destructed; /**< A flag stating that this object has already been destructed. */
  std::fstream* logMessages; /** The file messages from the robot are written to. */
  TimeInfo timeInfo; /**< Information about the timing of modules. */
  SensorData sensorData; /**< The most current sensor data sent. */

protected:
  DebugKeyTable dkt; /**< The current state of the debug key table. */
    
public:
  /**
  * Constructor.
  */
  RobotConsole(MessageQueue& in,MessageQueue& out);
  
  /**
  * Destructor.
  */
  ~RobotConsole();

  /**
  * The function adds all views.
  */
  void addViews();

  /** 
  * The function is called for every incoming debug message.
  */
  virtual bool handleMessage(InMessage& message);
  
  /**
  * The function is called when a console command has been entered.
  * @param stream A text stream containing the console command.
  */
  void handleConsole(InConfigMemory& stream);
  
private:
  /**
  * The function prints a text as part of a list to the console if it contains a required subtext.
  * @param text The text to print. On the console, it will be followed by a space.
  * @param required The subtext that is search for.
  */
  void list(const char* text, const char* required);

  /**
  * The function returns the number of the Xabsl info to use.
  * @param stream The stream the command is read from. The function always reads one string
  *               in advance.
  * @param buffer A buffer strings can be read in. It will contain the next string when
  *               the function returns.
  * @return 0 for the behavior info and 1 for the head control info.
  */
  int getXabslIndex(InConfigMemory& stream, char* buffer);
  
  //!@name Handler for different console commands
  //!@{
  bool msg(InConfigMemory&);
  bool calcImage(InConfigMemory&);
  bool cameraParameters(InConfigMemory&);
  bool debugKey(InConfigMemory&);
  bool gameControl(InConfigMemory& stream);
  bool headControlMode(InConfigMemory&);
  bool headMotionRequest(InConfigMemory&);
  bool log(InConfigMemory& stream);
  bool sendMotionRequest(InConfigMemory&);
  bool penalizeRobot(InConfigMemory& stream);
  bool queueFillRequest(InConfigMemory& stream);
  bool sendGeneric(InConfigMemory& stream);
  bool sendOracledWorldStates(InConfigMemory& stream);
  bool solutionRequest(InConfigMemory&);
  bool sendTailRequest(InConfigMemory& stream);
  bool xabslLoadBehavior(InConfigMemory& stream);
  bool xabslInputSymbol(InConfigMemory& stream);
  bool xabslOutputSymbol(InConfigMemory& stream);
  bool xabslOption(InConfigMemory& stream);
  bool xabslBasicBehavior(InConfigMemory& stream);
  //!@}
};

#endif // __RobotConsole_H__


/*
* Change log :
*
* $Log: RobotConsole.h,v $
* Revision 1.4  2004/06/15 17:12:27  roefer
* Edge percept added
*
* Revision 1.3  2004/05/25 18:55:11  tim
* repaired SimRobot
*
* Revision 1.2  2004/05/22 22:52:04  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.1.1.1  2004/05/22 17:24:33  cvsadm
* created new repository GT2004_WM
*
* Revision 1.10  2004/05/18 11:04:54  roefer
* Separate view and optional parameter hc for xabsl head control
*
* Revision 1.9  2004/04/27 19:39:37  roefer
* Support for debug images added to simulator
*
* Revision 1.8  2004/04/20 13:14:53  roefer
* All console commands now also work outside the start script
*
* Revision 1.7  2004/04/12 18:00:45  roefer
* SensorView added
*
* Revision 1.6  2004/04/09 18:55:16  roefer
* Better structure for views
* cp command added
* Timing view added
*
* Revision 1.5  2004/02/03 13:19:50  spranger
* renamed all references to  class BallP_osition to BallModel
*
* Revision 1.4  2003/12/29 16:06:34  roefer
* Ability to log text messages sent from robot added
*
* Revision 1.3  2003/11/20 17:28:32  roefer
* xy ? command has optional pattern
*
* Revision 1.2  2003/10/16 15:30:34  goehring
* Doxygen
*
* Revision 1.1  2003/10/14 07:34:16  roefer
* Support files for SimRobXP added, not finished yet
*
*/
