/** 
 * @file Platform/Win32/ForSimRobXP/OVirtualRobotComm.h
 *
 * Declaration of OVirtualRobotComm.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#ifndef __OVirtualRobotComm_H__
#define __OVirtualRobotComm_H__

#include "RobotConsole.h"

/**
 * @class OVirtualRobotComm
 *
 * A process that simulates the Aperios OVirtualRobotComm.
 */
class OVirtualRobotComm : public RobotConsole
{
  private:
    DEBUGGING;
    RECEIVER(MotorCommands);
    RECEIVER(OdometryData);

    SENDER(SensorDataBuffer);
    SENDER(Image);

    bool playBack; /**< Decides whether this robot uses the simulation or replays a log file. */
    unsigned frameNumber, /**< The frame counter. */
             lastFrameNumber; /**< The last frame already processed. */
    MotorCommands motorCommands; /**< A buffer for the motor commands. */
    SimObject* objRobot; /**< A pointer to the associated SimRobot robot. */
    int spCamera, /**< The sensor port of the robot's camera. */
        spPsd, /**< The sensor port of the robot's psd sender. */
        spPsdFar, /**< The sensor port of the robot's far psd sender. */
        spBodyPsd, /**< The sensor port of the robot's body psd sender. */
        apJoint[JointData::numOfJoint], /**< Actuator ports of all joints. */
        spJoint[JointData::numOfJoint], /**< Sensor ports of all joints. */
        spSwitch[5]; /**< Sensor ports of all switches. */
    SimObject* objLed[LEDValue::numOfLED_ERS7]; /**< Pointers to all leds. */
    OdometryData odometryData, /**< A buffer for the odometry data. */
                 lastOdometry; /**< The last odometry pose. */
    double lastHeight, /**< The previous height of the body of the robot. */
           lastSpeed;  /**< The previous vertical speed of the body of the robot. */
    static const char* jointName[JointData::numOfJoint]; /**< A table of the SimRobot names of the joints. */
    static const char* ledNameERS210[LEDValue::numOfLED_ERS210]; /**< A table of the SimRobot names of the ERS-210 leds. */
    static const char* ledNameERS7[LEDValue::numOfLED_ERS7]; /**< A table of the SimRobot names of the ERS-7 leds. */
    static const char* switchNameERS210[3]; /**< A table of the SimRobot names of the ERS-210 switches. */
    static const char* switchNameERS7[5]; /**< A table of the SimRobot names of the ERS-7 switches. */

    /**
     * The function returns the angle of a joint.
     * @param j The index of the joint.
     * @return The angle in radians.
     */
    double getJointAngle(int j) const;

    /**
     * The function sets the angle of a joint.
     * @param j The index of the joint.
     * @param angle The angle in radians.
     */
    void setJointAngle(int j,double angle);

    /**
     * The function sets the on/off state of an led.
     * @param l The index of the led.
     * @param state The new state of all leds.
     * @param off The surface used for the deactivated led.
     * @param on The surface used for the activated led.
     */
    void setLedState(LEDValue::LED l,int state,const char* off, const char* on);

    /**
     * The function sets the on/off state of double leds.
     * @param l The index of the led.
     * @param state The new state of all leds.
     * @param off The surface used for the deactivated led.
     * @param first The surface used if the first led is activated.
     * @param second The surface used if the second led is activated.
     * @param both The surface used if both leds are activated.
     */
    void setLedState(LEDValue::LED l,int state,const char* off, const char* first,
                                               const char* second, const char* both);

    /**
     * The function sets the on/off state of triple leds.
     * @param l The index of the led.
     * @param state The new state of all leds.
     * @param off The surface used for the deactivated led.
     * @param first The surface used if the first led is activated.
     * @param second The surface used if the second led is activated.
     * @param both The surface used if both leds are activated.
     * @param third The surface used for the third led.
     * @param andFirst The surface used if the first and third leds are activated.
     * @param andSecond The surface used if the second and third leds are activated.
     * @param all The surface used if all leds are activated.
     */
    void setLedState(LEDValue::LED l,int state,const char* off, const char* first,
                                               const char* second, const char* both,
                                               const char* third, const char* andFirst,
                                               const char* andSecond, const char* all);

    /**
     * The function returns the angle of a joint.
     * @param j The index of the joint.
     * @return The angle in radians.
     */
    bool getSwitch(int s) const;

  public:
    /**
     * Constructor.
     */
    OVirtualRobotComm();
  
    /**
     * The function is called from the framework once in every frame 
     */
    virtual int main();
  
    /**
     * The function must be called to exchange data with SimRobot.
     * It sends the motor commands to SimRobot and acquires new sensor data.
     */
    void update();
};

#endif // __OVirtualRobotComm_H__


/*
 * Change log :
 *
 * $Log: OVirtualRobotComm.h,v $
 * Revision 1.1.1.1  2004/05/22 17:24:29  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.8  2004/05/22 09:05:50  roefer
 * All ERS-7 LEDs but face LEDs added
 *
 * Revision 1.7  2004/05/20 17:37:23  roefer
 * Support for ERS-7 switches
 *
 * Revision 1.6  2004/05/13 06:56:23  roefer
 * Simulation of PSDs added
 * Special intrisic parameters for simulated images
 *
 * Revision 1.5  2003/12/31 14:29:20  roefer
 * Joints and LEDs for ERS-7
 *
 * Revision 1.4  2003/12/06 21:03:16  roefer
 * Improved simulation of z-acceleration
 *
 * Revision 1.3  2003/10/20 14:05:53  roefer
 * Views and buttons
 *
 * Revision 1.2  2003/10/16 15:30:34  goehring
 * Doxygen
 *
 * Revision 1.1  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 *
 */
