/** 
* @file Platform/Win32/ForSimRobXP/OVirtualRobotComm.cpp
*
* Implementation of OVirtualRobotComm.
*
* @author <A href="mailto:roefer@tzi.de">Thomas Rfer</A>
*/

#include "OVirtualRobotComm.h"
#include "ConsoleRoboCupCtrl.h"
#include "../Oracle2.h"
#include "SimRobotCore/DrawableObject.h"
#include "Modules/ImageProcessor/ImageProcessorTools/ColorCorrector.h"

const char* OVirtualRobotComm::jointName[JointData::numOfJoint] =
{
  ".head",                         // headTilt
  ".head.pan",                     // headPan
  ".head.pan.roll",                // headRoll
  ".head.pan.roll.head.mouth",     // mouth
  ".head.pan.roll.head.left ear",  // earL
  ".head.pan.roll.head.right ear", // earR
  ".right arm",                    // legFR1
  ".right arm.x",                  // legFR2
  ".right arm.x.y",                // legFR3
  ".left arm",                     // legFL1
  ".left arm.x",                   // legFL2
  ".left arm.x.y",                 // legFL3
  ".right leg",                    // legHR1
  ".right leg.x",                  // legHR2
  ".right leg.x.y",                // legHR3
  ".left leg",                     // legHL1
  ".left leg.x",                   // legHL2
  ".left leg.x.y",                 // legHL3
  ".tail",                         // tailPan
  ".tail.y"                        // tailTilt
};

const char* OVirtualRobotComm::ledNameERS210[LEDValue::numOfLED_ERS210] =
{
  ".head.pan.roll.head.head.bottom left red", // redLB,
  ".head.pan.roll.head.head.left green",      // greenL,
  ".head.pan.roll.head.head.top left red",    // redLT,
  ".head.pan.roll.head.head.bottom right red",// redRB,
  ".head.pan.roll.head.head.right green",     // greenR,
  ".head.pan.roll.head.head.top right red",   // redRT,
  ".head.pan.roll.head.head.top green",       // greenT,
  ".tail.y.tail.LED",                         // tailBlue
  ".tail.y.tail.LED"                          // tailRed
};

const char* OVirtualRobotComm::ledNameERS7[LEDValue::numOfLED_ERS7] =
{
  ".head.pan.roll.head.head04.orange", // headOrange
  ".head.pan.roll.head.head04.orange", // headWhite
  ".head.pan.roll.head.head04.ear", // modeRed
  ".head.pan.roll.head.head04.ear", // modeGreen
  ".head.pan.roll.head.head04.ear", // modeBlue
  ".head.pan.roll.head.head04.blue", // wireless
  "", // face1
  "", // face2
  "", // face3
  "", // face4
  "", // face5
  "", // face6
  "", // face7
  "", // face8
  "", // face9
  "", // face10
  "", // face11
  "", // face12
  "", // face13
  "", // face14
  ".body01.body04.blue", // backFrontBlue
  ".body01.body04.blue", // backFrontWhite
  ".body01.body04.orange", // backMiddleOrange
  ".body01.body04.orange", // backMiddleWhite
  ".body01.body04.red", // backRearRed
  ".body01.body04.red" // backRearWhite
};

const char* OVirtualRobotComm::switchNameERS210[3] =
{
  ".body.backSwitch",
  ".head.pan.roll.head.head.backSwitch",
  ".head.pan.roll.head.head.frontSwitch"
};

const char* OVirtualRobotComm::switchNameERS7[5] =
{
  ".body01.backSwitch",
  ".body01.middleSwitch",
  ".body01.frontSwitch",
  ".head.pan.roll.head.head03.switch",
  ".head.pan.roll.head.mouth.chops.switch"
};

OVirtualRobotComm::OVirtualRobotComm()
: RobotConsole(theDebugReceiver,theDebugSender),
  theDebugReceiver(this,"Receiver.MessageQueue.O",false),
  theDebugSender(this,"Sender.MessageQueue.S",false),
  theMotorCommandsReceiver(this,"Effector.OCommandVectorData.O",false),
  INIT_RECEIVER(OdometryData,false),

  theSensorDataBufferSender(this,"Sensor.OSensorFrameVectorData.S",false),
  theImageSender(this,"FbkImageSensor.OFbkImageVectorData.S",false)
{
  theDebugReceiver.setSize(300000);
  theDebugSender.setSize(1000000);

  std::string robotName = RoboCupCtrl::getRobotFullName();
  std::string logFile = ConsoleRoboCupCtrl::getLogFile();
  playBack = logFile.size() > 0;
  dkt.set(DebugKeyTable::sendImage,DebugKey::always);
  dkt.set(DebugKeyTable::sendPercepts,DebugKey::always);
  dkt.set(DebugKeyTable::sendWorldState,DebugKey::always);
  debugOut.out.bin << dkt;
  debugOut.out.finishMessage(idDebugKeyTable);

  addViews();

  if(playBack)
    logPlayer.open(logFile.c_str());
  else
  {
    VERIFY(objRobot = ctrl->getObjectReference(robotName));
  
    spCamera = ctrl->getSensorportId(robotName + ".head.pan.roll.head.camera.image");
    if(getRobotConfiguration().getRobotDesign() == RobotDesign::ERS210)
    {
      spPsd = ctrl->getSensorportId(robotName + ".head.pan.roll.head.PSD.distance");
      int i;
      for(i = 0; i < LEDValue::numOfLED_ERS210; ++i)
        VERIFY(objLed[i] = ctrl->getObjectReference(robotName + ledNameERS210[i]));
      for(i = 0; i < 3; ++i)
        spSwitch[i] = ctrl->getSensorportId(robotName + switchNameERS210[i] +  ".click");
    }
    else
    {
      spPsd = ctrl->getSensorportId(robotName + ".head.pan.roll.head.nearPSD.distance");
      spPsdFar = ctrl->getSensorportId(robotName + ".head.pan.roll.head.farPSD.distance");
      spBodyPsd = ctrl->getSensorportId(robotName + ".bodyPSD.distance");
      int i;
      for(i = 0; i < LEDValue::numOfLED_ERS7; ++i)
        if(*ledNameERS7[i])
          VERIFY(objLed[i] = ctrl->getObjectReference(robotName + ledNameERS7[i]));
      for(i = 0; i < 5; ++i)
        spSwitch[i] = ctrl->getSensorportId(robotName + switchNameERS7[i] +  ".click");
    }
    for(int i = 0; i < JointData::numOfJoint; ++i)
    {
      apJoint[i] = ctrl->getActuatorportId(robotName + jointName[i] +  ".value");
      spJoint[i] = ctrl->getSensorportId(robotName + jointName[i] +  ".valueSensor");
    }
    frameNumber = 1000; // approx. 8 seconds after startup
    lastFrameNumber = frameNumber;
    lastHeight = objRobot->getPosition().v[2];
    lastSpeed = 0;
  }
}

int OVirtualRobotComm::main() 
{
  // Only one thread can access *this now.
  SYNC;

  if(playBack)
    theDebugSender.send();
  else
  {
    // determine teamColor here
    teamColor = getPlayer().getTeamColor();
    // Receivers cannot be accessed in a synchronized way.
    // Therefore, copy them to member variables.
    motorCommands = theMotorCommandsReceiver;
    odometryData = theOdometryDataReceiver;
  
    if(frameNumber != lastFrameNumber)
    {
      lastFrameNumber = frameNumber;
    
      if(frameNumber % 5 == 0)
      {
        //calculate an oracled world state
        RobotPose oracledRobotPose = Oracle::getRobotPose(objRobot);
        BallModel oracledBallModel = Oracle::getBallModel(objRobot);
        PlayerPoseCollection oracledPlayerPoseCollection = Oracle::getPlayerPoseCollection(objRobot);
        ObstaclesModel obstaclesModel = Oracle::getObstaclesModel(oracledRobotPose, oracledPlayerPoseCollection);
        RobotState robotState;
        CameraMatrix cameraMatrix = Oracle::getCameraMatrix(theSensorDataBufferSender.lastFrame());
        CameraInfo cameraInfo(getRobotConfiguration().getRobotDesign());
      
        if(sendOracle)
        {
          getDebugOut().bin << SEND_WORLD_STATE(oracledRobotPose,
            oracledBallModel,oracledPlayerPoseCollection,obstaclesModel,robotState, cameraMatrix, cameraInfo);
          getDebugOut().finishMessage(idWorldState);
        }
      
        // send the same world state as idOracledWorldState. 
        // That message is then immediately sent back 
        getDebugOut().bin << SEND_WORLD_STATE(oracledRobotPose,
          oracledBallModel,oracledPlayerPoseCollection,obstaclesModel,robotState, cameraMatrix, cameraInfo);
        getDebugOut().finishMessage(idOracledWorldState);
      }
    
      theDebugSender.send();
    
    if(frameNumber % 5 == 0 && getRobotConfiguration().getRobotDesign() == RobotDesign::ERS210 ||
       frameNumber % 4 == 0 && getRobotConfiguration().getRobotDesign() == RobotDesign::ERS7)
        theImageSender.send();
      if(frameNumber % 4 == 0)
        theSensorDataBufferSender.send();
    }
  }  
  return 8;
}

void OVirtualRobotComm::update()
{
  // Only one thread can access *this now.
  SYNC;

  if(playBack)
  {
    if(logPlayer.getState() != LogPlayer::playing)
      logPlayer.play();
    logPlayer.onIdle();
  }
  else
  {
    // Set joint data
    for(int i = 0; i < JointData::numOfJoint; ++i)
      if(motorCommands.jointDataBuffer.frame[0].data[i] != jointDataInvalidValue)
        setJointAngle(i,motorCommands.jointDataBuffer.frame[0].data[i] / 1.0e6);
  
    if(getRobotConfiguration().getRobotDesign() == RobotDesign::ERS210)
    {
      long value = motorCommands.ledValue.data[0];
      setLedState(LEDValue::redLB, value, "dark red led", "red led");
      setLedState(LEDValue::redLT, value, "dark red led", "red led");
      setLedState(LEDValue::redRB, value, "dark red led", "red led");
      setLedState(LEDValue::redRT, value, "dark red led", "red led");
      setLedState(LEDValue::greenT, value, "dark green led", "green led");
      setLedState(LEDValue::greenL, value, "dark green led", "green led");
      setLedState(LEDValue::greenR, value, "dark green led", "green led");
      setLedState(LEDValue::tailBlue, value, "dark tail led", "blue led", "orange led", "pink led");
    }
    else
    {
      long value = motorCommands.ledValue.data[0];
      setLedState(LEDValue::headOrange, value, "dark orange led", "orange back led", "white back led", "orange white back led");
      setLedState(LEDValue::modeRed, value, "dark back led", "red back led", "green head led", "yellow head led", 
                                            "blue back led", "pink head led", "head turquoise led", "white back led");
      setLedState(LEDValue::wireless, value, "dark blue led", "blue back led");
      setLedState(LEDValue::backFrontBlue, value, "dark back led", "blue back led", "white back led", "blue white back led");
      setLedState(LEDValue::backMiddleOrange, value, "dark back led", "orange back led", "white back led", "orange white back led");
      setLedState(LEDValue::backRearRed, value, "dark back led", "red back led", "white back led", "red white back led");
    }

    if((frameNumber % 5 == 0 && getRobotConfiguration().getRobotDesign() == RobotDesign::ERS210 ||
        frameNumber % 4 == 0 && getRobotConfiguration().getRobotDesign() == RobotDesign::ERS7) && calculateImage)
    {
      theImageSender.cameraInfo = CameraInfo(getRobotConfiguration().getRobotDesign());
      unsigned char* imageBuffer;
      ctrl->getSensorValue(spCamera, imageBuffer);
      int r, g, b,
          y, u, v,
          row;
      for(row = 0; row < theImageSender.cameraInfo.resolutionHeight; ++row)
        for(int x = 0; x < theImageSender.cameraInfo.resolutionWidth; ++x)
        {
          b = *imageBuffer++;
          g = *imageBuffer++;
          r = *imageBuffer++;
          y =          306 * r + 601 * g + 117 * b;
          u = 130048 + 512 * r - 429 * g -  83 * b;
          v = 130048 - 173 * r - 339 * g + 512 * b;

          if(y < 0) y = 0; else if(y > 261120) y = 261120;
          if(u < 0) u = 0; else if(u > 261120) u = 261120;
          if(v < 0) v = 0; else if(v > 261120) v = 261120;

          theImageSender.image[row][0][x] = (unsigned char)(y >> 10);
          theImageSender.image[row][1][x] = (unsigned char)(u >> 10);
          theImageSender.image[row][2][x] = (unsigned char)(v >> 10);
        }
      if(theImageSender.cameraInfo.resolutionWidth == cameraResolutionWidth_ERS7)
        ColorCorrector::distort(theImageSender);
      theImageSender.frameNumber = frameNumber + 0x10000000;
    }
    // Setup sensor data
    if(theSensorDataBufferSender.numOfFrames == 4)
      for(int i = 0; i < 3; ++i)
        theSensorDataBufferSender.frame[i] = theSensorDataBufferSender.frame[i + 1];
    else
      ++theSensorDataBufferSender.numOfFrames;

    SensorData& s = theSensorDataBufferSender.frame[theSensorDataBufferSender.numOfFrames - 1];
    s.data[SensorData::neckTilt] = s.refValue[SensorData::neckTilt] = long(getJointAngle(JointData::neckTilt) * 1e6);
    s.data[SensorData::headPan] = s.refValue[SensorData::headPan] = long(getJointAngle(JointData::headPan) * 1e6);
    s.data[SensorData::headTilt] = s.refValue[SensorData::headTilt] = long(getJointAngle(JointData::headTilt) * 1e6);
    if(getRobotConfiguration().getRobotDesign() == RobotDesign::ERS210)
    {
      s.data[SensorData::headBack] = s.refValue[SensorData::headBack] = getSwitch(1) ? 400000 : 0;
      s.data[SensorData::headFront] = s.refValue[SensorData::headFront] = getSwitch(2) ? 400000 : 0;
      s.data[SensorData::back] = s.refValue[SensorData::back] = getSwitch(0) ? 1 : 0;
      s.data[SensorData::chin] = s.refValue[SensorData::chin] = 0;
      double psd;
      ctrl->getSensorValue(spPsd, psd);
      psd += 0.1;
      s.data[SensorData::psd] = s.refValue[SensorData::psd] = long(psd * 1000000);
    }
    else
    {
      s.data[SensorData::backR] = s.refValue[SensorData::backR] = getSwitch(0) ? 31 : 0;
      s.data[SensorData::backM] = s.refValue[SensorData::backM] = getSwitch(1) ? 31 : 0;
      s.data[SensorData::backF] = s.refValue[SensorData::backF] = getSwitch(2) ? 31 : 0;
      s.data[SensorData::head] = s.refValue[SensorData::head] = getSwitch(3) ? 31 : 0;
      s.data[SensorData::chin] = s.refValue[SensorData::chin] = getSwitch(4) ? 1 : 0;
      double psd;
      ctrl->getSensorValue(spPsd, psd);
      psd += 0.02;
      if(psd < 0.05)
        psd = 0.05;
      s.data[SensorData::headPsdNear] = s.refValue[SensorData::headPsdNear] = long(psd * 1000000);
      ctrl->getSensorValue(spPsdFar, psd);
      psd += 0.02;
      if(psd < 0.2)
        psd = 0.2;
      s.data[SensorData::headPsdFar] = s.refValue[SensorData::headPsdFar] = long(psd * 1000000);
      ctrl->getSensorValue(spBodyPsd, psd);
      psd += 0.02;
      if(psd < 0.1)
        psd = 0.1;
      s.data[SensorData::bodyPsd] = s.refValue[SensorData::bodyPsd] = long(psd * 1000000);
    }
    s.data[SensorData::mouth] = s.refValue[SensorData::mouth] = long(getJointAngle(JointData::mouth) * 1e6);
    s.data[SensorData::legFL1] = s.refValue[SensorData::legFL1] = long(getJointAngle(JointData::legFL1) * 1e6);
    s.data[SensorData::legFL2] = s.refValue[SensorData::legFL2] = long(getJointAngle(JointData::legFL2) * 1e6);
    s.data[SensorData::legFL3] = s.refValue[SensorData::legFL3] = long(getJointAngle(JointData::legFL3) * 1e6);
    // missing: pawFL
    s.data[SensorData::legHL1] = s.refValue[SensorData::legHL1] = long(getJointAngle(JointData::legHL1) * 1e6);
    s.data[SensorData::legHL2] = s.refValue[SensorData::legHL2] = long(getJointAngle(JointData::legHL2) * 1e6);
    s.data[SensorData::legHL3] = s.refValue[SensorData::legHL3] = long(getJointAngle(JointData::legHL3) * 1e6);
    // missing: pawHL
    s.data[SensorData::legFR1] = s.refValue[SensorData::legFR1] = long(getJointAngle(JointData::legFR1) * 1e6);
    s.data[SensorData::legFR2] = s.refValue[SensorData::legFR2] = long(getJointAngle(JointData::legFR2) * 1e6);
    s.data[SensorData::legFR3] = s.refValue[SensorData::legFR3] = long(getJointAngle(JointData::legFR3) * 1e6);
    // missing: pawFR
    s.data[SensorData::legHR1] = s.refValue[SensorData::legHR1] = long(getJointAngle(JointData::legHR1) * 1e6);
    s.data[SensorData::legHR2] = s.refValue[SensorData::legHR2] = long(getJointAngle(JointData::legHR2) * 1e6);
    s.data[SensorData::legHR3] = s.refValue[SensorData::legHR3] = long(getJointAngle(JointData::legHR3) * 1e6);
    // missing: pawHR
    s.data[SensorData::tailPan] = s.refValue[SensorData::tailPan] = long(getJointAngle(JointData::tailPan) * 1e6);
    s.data[SensorData::tailTilt] = s.refValue[SensorData::tailTilt] = long(getJointAngle(JointData::tailTilt) * 1e6);
    // missing: thermo

    Matrix3d m = objRobot->getRotation();
    double height = objRobot->getPosition().v[2],
           speed = (height - lastHeight) * 125,
           acc = (speed - lastSpeed);
    if(acc < -5)
      acc = -5;
    lastHeight = height;
    lastSpeed = speed;
    m.rotateZ(-m.getZAngle());
    double tilt = m.getYAngle(),
           roll = m.getXAngle();
    s.data[SensorData::accelerationX] = long(9810000 * sin(tilt));
    s.data[SensorData::accelerationY] = long(-9810000 * sin(roll));;
    s.data[SensorData::accelerationZ] = long(-9810000 * cos(tilt) - acc * 1e6);

    s.frameNumber = frameNumber;
  
    Pose2D pose = ctrl->getPose2D(objRobot);
    if(fabs(roll) < 0.7)
    {
      pose.translation *= 1000;
      pose += odometryData - lastOdometry;
      pose.translation *= 0.001;
    }
    lastOdometry = odometryData;
    ctrl->setPose2D(objRobot,pose,&s);

    ++frameNumber;
  }
}

void OVirtualRobotComm::setJointAngle(int j, double angle)
{
  ctrl->setActuatorport(apJoint[j], angle * 180 / M_PI);
}

double OVirtualRobotComm::getJointAngle(int j) const
{
  double value;
  ctrl->getSensorValue(spJoint[j], value);
  return value * M_PI / 180;
}

void OVirtualRobotComm::setLedState(LEDValue::LED l,int state,const char* off, const char* on)
{
  ((DrawableObject*) objLed[l])->setSurface(ctrl->getSurface(state & (1 << l) ? on : off));
}

void OVirtualRobotComm::setLedState(LEDValue::LED l,int state,const char* off, const char* first,
                                                              const char* second, const char* both)
{
  if(state & (1 << (l + 1)))
    setLedState(l, state, second, both);
  else
    setLedState(l, state, off, first);
}

void OVirtualRobotComm::setLedState(LEDValue::LED l,int state,const char* off, const char* first,
                                                              const char* second, const char* both,
                                                              const char* third, const char* andFirst,
                                                              const char* andSecond, const char* all)
{
  if(state & (1 << (l + 2)))
    setLedState(l, state, third, andFirst, andSecond, all);
  else
    setLedState(l, state, off, first, second, both);
}

bool OVirtualRobotComm::getSwitch(int s) const
{
  bool value;
  ctrl->getSensorValue(spSwitch[s], value);
  return value;
}

MAKE_PROCESS(OVirtualRobotComm);

/*
* Change log :
*
* $Log: OVirtualRobotComm.cpp,v $
* Revision 1.4  2004/05/27 17:13:37  jhoffman
* - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
* - clipping included for setJoints
* - removed some microrad/rad-bugs
* - bodyPosture constructor and "=" operator fixed
*
* Revision 1.3  2004/05/27 09:55:38  roefer
* Oracled camera matrix and camera info
*
* Revision 1.2  2004/05/22 22:52:04  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.1.1.1  2004/05/22 17:24:29  cvsadm
* created new repository GT2004_WM
*
* Revision 1.27  2004/05/22 09:05:50  roefer
* All ERS-7 LEDs but face LEDs added
*
* Revision 1.26  2004/05/20 17:37:23  roefer
* Support for ERS-7 switches
*
* Revision 1.25  2004/05/13 06:56:23  roefer
* Simulation of PSDs added
* Special intrisic parameters for simulated images
*
* Revision 1.24  2004/04/22 14:28:46  roefer
* ColorCorrector now supports MSH and DDD color correction
*
* Revision 1.23  2004/04/20 13:14:53  roefer
* All console commands now also work outside the start script
*
* Revision 1.22  2004/04/17 21:50:50  roefer
* New color corrector (BB/DDD mix) used in BB2004ImageProcessor and Simulator
* Displaying corrected images as debug image
*
* Revision 1.21  2004/04/09 11:35:53  roefer
* Bremen Byters German Open check-in
*
* Revision 1.21  2004/03/31 09:51:50  roefer
* BBColorCorrector was loaded in wrong class
*
* Revision 1.20  2004/03/28 12:02:47  roefer
* All drawings can be switched on and off in simulator
*
* Revision 1.19  2004/03/21 19:07:48  roefer
* More realistic camera images for ERS-7
*
* Revision 1.18  2004/03/10 23:55:27  roefer
* ERS7 support for log files
*
* Revision 1.17  2004/02/03 13:20:48  spranger
* renamed all references to  class BallP_osition to BallModel (possibly changed include files)
*
* Revision 1.16  2004/01/28 21:55:49  roefer
* RobotDimensions revised
*
* Revision 1.15  2004/01/23 00:13:23  roefer
* ERS-7 simulation, first draft
*
* Revision 1.14  2004/01/21 00:52:24  roefer
* Include files and first version of simulated ERS-7
*
* Revision 1.13  2004/01/17 19:19:18  roefer
* Simulator calculates robot pose based on class Kinematics now
*
* Revision 1.12  2004/01/10 10:09:14  juengel
* Added CameraInfo to and removed Player from (SEND|RECEIVE)_(PERCEPTS|WORLDSTATE).
*
* Revision 1.11  2003/12/31 14:29:20  roefer
* Joints and LEDs for ERS-7
*
* Revision 1.10  2003/12/15 11:49:07  juengel
* Introduced CameraInfo
*
* Revision 1.9  2003/12/09 21:17:55  roefer
* Platform-dependent code moved to GUI (release code) project
*
* Revision 1.8  2003/12/08 16:48:56  roefer
* Acceleration limited
*
* Revision 1.7  2003/12/06 21:03:16  roefer
* Improved simulation of z-acceleration
*
* Revision 1.6  2003/10/23 15:41:40  roefer
* Oracled obstacle model added
*
* Revision 1.5  2003/10/21 15:05:51  roefer
* Added the oracle
*
* Revision 1.4  2003/10/21 12:18:08  roefer
* Calculation of acceleration sensors corrected
*
* Revision 1.3  2003/10/20 14:05:53  roefer
* Views and buttons
*
* Revision 1.2  2003/10/16 15:30:34  goehring
* Doxygen
*
* Revision 1.1  2003/10/14 07:34:16  roefer
* Support files for SimRobXP added, not finished yet
*
*/
