/**
 * @file Platform/Win32/ForSimRobXP/ConsoleRoboCupCtrl.cpp
 *
 * This file implements the class ConsoleRoboCupCtrl.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */
#include "ConsoleRoboCupCtrl.h"
#include "OVirtualRobotComm.h"
#include "Representations/Motion/HeadControlMode.h"
#include "Tools/Debugging/GenericDebugData.h"
#include <mmsystem.h>

std::string ConsoleRoboCupCtrl::logFile;

CONNECT_CONTROLLER_TO_SCENE(ConsoleRoboCupCtrl,"GT2004");

ConsoleRoboCupCtrl::ConsoleRoboCupCtrl()
{
  xabsl2Infos[0] = xabsl2Infos[1] = 0;
  newLine = true;
  nesting = 0;
  lastTime = 0;
  tilt = pan = roll = 0;
  tiltId = 0;
  panId = 2;
  rollId = 1;
  if(robots.getFirst())
    simulated.insert(robots.getFirst());

  std::string fileName = getSimulationFileName();
  int p = fileName.find_last_of("\\"),
      p2 = fileName.find_last_of(".");
  if(p2 > p)
    fileName = fileName.substr(0, p2);
  executeFile(fileName.c_str(), false);
  createCompletion();
}

ConsoleRoboCupCtrl::~ConsoleRoboCupCtrl()
{
  for(List<PhysicalRobot>::Pos i = physicalRobots.getFirst(); i; ++i)
    physicalRobots[i].announceStop();
}

void ConsoleRoboCupCtrl::execute()
{
  handleJoystick();
  {
    SYNC;
    for(List<std::string>::Pos i = textMessages.getFirst(); i; ++i)
      if(newLine || i != textMessages.getLast())
        RoboCupCtrl::printLn(textMessages[i]);
      else
        RoboCupCtrl::print(textMessages[i]);
    textMessages.clear();
  }
  RoboCupCtrl::execute();
}

void ConsoleRoboCupCtrl::executeFile(std::string name, bool printError)
{
  if(nesting == 10)
    printLn("Nesting Error");
  else
  {
    ++nesting;
    if(name.find_first_of(".") == -1)
      name = name + ".con";
    if(name[0] != '\\' && (name.size() < 2 || name[1] != ':'))
      name = std::string("Scenes\\") + name;
    InBinaryFile stream(name.c_str());
    if(!stream.exists())
    {
      if(printError)
        printLn(name + " not found");
    }
    else
      while(!stream.eof())
      {
        std::string line;
        while(!stream.eof())
        {
          char c[2] = " ";
          stream >> c[0];
          if(c[0] == '\n')
            break;
          else if(c[0] != '\r')
            line = line + c;
        }
        if(line.find_first_not_of(" ") != -1)
          onConsoleCommand(line);
      }
    --nesting;
  }
}

void ConsoleRoboCupCtrl::onSelected(SimObject* obj)
{
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    if(obj == robots[i].getSimRobotObject())
    {
      simulated.clear();
      physical.clear();
      simulated.insert(i);
      printLn(std::string("robot ") + robots[i].getName());
      return;
    }
}

void ConsoleRoboCupCtrl::onConsoleCommand(const std::string& command)
{
  char buffer[80];
  InConfigMemory stream(command.c_str(), command.size());
  stream >> buffer;
  if(!strcmp(buffer,"")) // comment
    return;
  else if(!strcmp(buffer,"help") || !strcmp(buffer,"?"))
    help();
  else if(!strcmp(buffer,"cls"))
    clear();
  else if(!strcmp(buffer,"echo"))
    echo(stream);
  else if(!strcmp(buffer,"call"))
  {
    stream >> buffer;
    executeFile(buffer);
  }
  else if(!strcmp(buffer,"robot"))
  {
    stream >> buffer;
    if(!strcmp("?",buffer))
    {
      for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
        print(std::string(robots[i].getName()) + " ");
      for(List<PhysicalRobot>::Pos j = physicalRobots.getFirst(); j; ++j)
        print(std::string(physicalRobots[j].getName()) + " ");
      printLn("");
      return;
    }
    else if(!strcmp("all",buffer))
    {
      simulated.clear();
      physical.clear();
      for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
        simulated.insert(i);
      for(List<PhysicalRobot>::Pos j = physicalRobots.getFirst(); j; ++j)
        physical.insert(j);
      return;
    }
    else 
    {
      simulated.clear();
      physical.clear();
      for(;;)
      {
        if(!selectRobot(buffer))
          break;
        else if(stream.getEof())
          return;
        stream >> buffer;
      }
    }
    printLn("Syntax Error");
  }
  else if(!strcmp(buffer,"st"))
  {
    stream >> buffer;
    if(!strcmp(buffer,"on"))
    {
      if(!simTime)
      { // simulation time continues at real time
        time = getTime();
        simTime = true;
      }
    }
    else if(!strcmp(buffer,"off"))
    {
      if(simTime)
      { // real time contiues at simulation time
        time = getTime() - getCurrentSystemTime();
        simTime = false;
      }
    }
    else
      printLn("Syntax Error");
  }
  else if(!strcmp(buffer,"gc"))
  {
    for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
      robots[i].getRobotProcess()->handleConsole(InConfigMemory(command.c_str(), command.size()));
    for(List<PhysicalRobot>::Pos j = physicalRobots.getFirst(); j; ++j)
      physicalRobots[j].handleConsole(InConfigMemory(command.c_str(), command.size()));
  }
  else if(!strcmp(buffer,"sc"))
  {
    if(!startConnection(stream))
    {
      simulated.clear();
      physical.clear();
      if(robots.getFirst())
        simulated.insert(robots.getFirst());
    }
  }
  else if(!strcmp(buffer,"sl"))
  {
    if(!startLogFile(stream))
      printLn("Logfile not found!");
  }
  else if(!strcmp(buffer,"jbc"))
    joystickButtonCommand(stream);
  else if(!strcmp(buffer,"jhc"))
    joystickHeadControl(stream);
  else if(!simulated.getSize() && !physical.getSize())
    printLn("No robot selected!");
  else
  {
    for(List<List<Robot>::Pos>::Pos i = simulated.getFirst(); i; ++i)
      robots[simulated[i]].getRobotProcess()->handleConsole(InConfigMemory(command.c_str(), command.size()));
    for(List<List<PhysicalRobot>::Pos>::Pos j = physical.getFirst(); j; ++j)
      physicalRobots[physical[j]].handleConsole(InConfigMemory(command.c_str(), command.size()));
  }
  if(completion.empty())
    createCompletion();
}

bool ConsoleRoboCupCtrl::selectRobot(const char* name)
{
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    if(!strcmp(name,robots[i].getName()))
    {
      simulated.insert(i);
      return true;
    }
  for(List<PhysicalRobot>::Pos j = physicalRobots.getFirst(); j; ++j)
    if(!strcmp(name,physicalRobots[j].getName()))
    {
      physical.insert(j);
      return true;
    }
  return false;
}

void ConsoleRoboCupCtrl::help()
{
  printLn("Initialization commands:");
  printLn("  sc <name> <a.b.c.d> ( ers210 | ers7 ) : Start connection to a real robot.");
  printLn("  sl <name> <file> ( ers210 | ers7 ) : Starts a robot reading its inputs from a log file.");
  printLn("Global commands:");
  printLn("  call <file> : Execute a script file.");
  printLn("  cls : Clear console window.");
  printLn("  echo <text> : Print text into console window. Useful in console.con.");
  printLn("  gc ready [ ( blue | red ) [ <blueScore> <redScore> ] ] | set | playing | finished : Game control.");
  printLn("  help | ? : Display this text.");
  printLn("  jbc <button> <command> : Set joystick button command.");
  printLn("  jhc tilt | pan | roll : Set head axis to be controlled by accelerator lever. The other axes will be controlled by the coolie head.");
  printLn("  robot ? | all | <name> {<name>} : Connect console to a set of active robots. Alternatively, double click one robot.");
  printLn("  st off | on : Switch simulation of time on or off.");
  printLn("  # <text> : Comment.");
  printLn("Robot commands:");
  printLn("  ci off | on : Switch the calculation of images on or off.");
  printLn("  cp ( indoor | outdoor | fluorescent ) ( low | mid | high ) ( slow | mid | fast ) : Set camera parameters.");
  printLn("  dk ? [pattern] | <key> off | on | <number> | every <number> [ms] : Set debug key.");
  printLn("  sg ? [pattern] | <id> {<num>} : Send generic debug data.");
  printLn("  hcm ? [pattern] | <mode> : Set head control mode.");
  printLn("  hmr <tilt> <pan> <roll> <mouth> : Send head motion request.");
  printLn("  log  start | stop | clear | save <file> : Record log file.");
  printLn("  mr ? [pattern] | <type> [<x> <y> <r>] : Send motion request.");
  printLn("  msg off | on | log <file> : Switch output of text messages on or off. Log text messages to a file.");
  printLn("  pr continue | ballHolding | keeperCharged | playerCharged | illegalDefender | illegalDefense | obstruction | pickup : Penalize robot.");
  printLn("  qfr queue | replace | reject | collect <seconds> | save <seconds> : Send queue fill request.");
  printLn("  so off | on : Switch sending of oracled world states on or off.");
  printLn("  sr ? [pattern] | <module> ( ? [pattern] | <solution> | off ) : Send solution request.");
  printLn("  tr ? [pattern] | <type> : Send tail request.");
  printLn("  xbb [hc] ( ? [pattern] | unchanged | <behavior> {<num>} ) : Select Xabsl2 basic behavior.");
  printLn("  xis [hc] ( ? [pattern] | <inputSymbol> ( on | off ) ) : Show Xabsl2 input symbol.");
  printLn("  xlb [hc] ( ? [pattern] | <name> [ update ] ) : Load Xabsl2 behavior and optionally update it on the robot.");
  printLn("  xo [hc] ( ? [pattern] | unchanged | <option> {<num>} ) : Select Xabsl2 option.");
  printLn("  xos [hc] ( ? [pattern] | <outputSymbol> ( on | off | ? [pattern] | unchanged | <value> ) ) : Show or set Xabsl2 output symbol.");
}

void ConsoleRoboCupCtrl::echo(InConfigMemory& stream)
{
  bool first = true;
  while(!stream.eof())
  {
    char text[80];
    stream >> text;
    if(first)
      first = false;
    else
      print(" ");
    print(text);
  }
  printLn("");
}

void ConsoleRoboCupCtrl::joystickButtonCommand(InConfigMemory& stream)
{
  int id = 0;
  stream >> id;
  if(id > 0 && id <= 32)
  {
    buttonCommand[--id] = "";
    bool first = true;
    while(!stream.eof())
    {
      char text[80];
      stream >> text;
      if(first)
        first = false;
      else
        buttonCommand[id] = buttonCommand[id] + " ";
      buttonCommand[id] = buttonCommand[id] + text;
    }
  }
  else
    printLn("Syntax Error");
}

void ConsoleRoboCupCtrl::joystickHeadControl(InConfigMemory& stream)
{
  char axis[80];
  stream >> axis;
  if(!strcmp(axis,"tilt"))
  {
    tiltId = 2;
    panId = 1;
    rollId = 0;
  }
  else if(!strcmp(axis,"pan"))
  {
    tiltId = 0;
    panId = 2;
    rollId = 1;
  }
  else if(!strcmp(axis,"roll"))
  {
    tiltId = 0;
    panId = 1;
    rollId = 2;
  }
  else
    printLn("Syntax Error");
}

bool ConsoleRoboCupCtrl::startConnection(InConfigMemory& stream)
{
  char name[80],
       ip[80],
       type[80];
  stream >> name >> ip >> type;
  robotName = std::string(".") + name;
  currentDesign = strcmp(type, "ers210") ? RobotDesign::ERS7: RobotDesign::ERS210;
  PhysicalRobot* pr = new PhysicalRobot(name, ip);
  if(!pr->isClient())
  {
    delete pr;
    printLn(std::string("No connection to ") + ip + " established!");
    return false;
  }
  else
  {
    simulated.clear();
    physical.clear();
    physicalRobots.insert(pr);
    physical.insert(physicalRobots.getLast());
    pr->addViews();
    pr->start();
    return true;
  }
}

bool ConsoleRoboCupCtrl::startLogFile(InConfigMemory& stream)
{
  char name[80],
       file[80],
       type[80];
  stream >> name >> file >> type;
  std::string fileName = file;
  if(fileName.find_first_of('.') == -1)
    fileName = fileName + ".log";
  if(fileName[0] != '\\' && (fileName.size() < 2 || fileName[1] != ':'))
    fileName = std::string("Logs\\") + fileName;
  {
    InBinaryFile test(fileName.c_str());
    if(!test.exists())
      return false;
  }
  robotName = std::string(".") + name;
  logFile = fileName;
  currentDesign = strcmp(type, "ers210") ? RobotDesign::ERS7 : RobotDesign::ERS210;
  robots.insert(new Robot(name,0,currentDesign));
  logFile = "";
  simulated.clear();
  physical.clear();
  simulated.insert(robots.getLast());
  connect(robots[robots.getLast()]);
  robots[robots.getLast()].start();
  return true;
}

void ConsoleRoboCupCtrl::print(const std::string& text) 
{
  SYNC;
  if(newLine)
    textMessages.insert(text);
  else
    textMessages[textMessages.getLast()] = textMessages[textMessages.getLast()] + text;
  newLine = false;
}

void ConsoleRoboCupCtrl::printLn(const std::string& text) 
{
  SYNC; 
  if(newLine)
    textMessages.insert(text);
  else
    textMessages[textMessages.getLast()] = textMessages[textMessages.getLast()] + text;
  newLine = true;
}
 
void ConsoleRoboCupCtrl::handleJoystick()
{
  JOYINFOEX joy;
  joy.dwSize = sizeof(joy);
  joy.dwFlags = JOY_RETURNALL;
  std::string command[3];
  if(joyGetPosEx(JOYSTICKID1,&joy) == 0)
  {
    command[0] = calcButtonCommand(joy.dwButtons);
    if(command[0] == "") // walk and move head only if no button command is executed
      if(getTime() - lastTime >= 40) 
      { // don't generate too many commands 
        lastTime = getTime();
        JOYCAPS joyCaps;
        if(joyGetDevCaps(JOYSTICKID1,&joyCaps,sizeof(joyCaps)) == 0)
        {
          if(joyCaps.wCaps & JOYCAPS_HASR)
            command[1] = calcWalk(joy.dwYpos,joy.dwXpos,joy.dwRpos);
          else
            command[1] = calcWalk(joy.dwYpos,0x8000,joy.dwXpos);
          if(joyCaps.wCaps & JOYCAPS_HASZ)
            command[2] = calcHead(joy.dwZpos,joy.dwPOV);
          else
            command[2] = calcHead(0x8000,joy.dwPOV);
        }
      }
      else
      { // use old command -> will be ignored
        command[1] = lastCommand[1];
        command[2] = lastCommand[2];
      }
  }
  for(int k = 0; k < 3; ++k)
    if(!(command[k] == lastCommand[k]))
    {
      lastCommand[k] = command[k];
      if(!(command[k] == ""))
        onConsoleCommand(command[k]);
    }
}

std::string ConsoleRoboCupCtrl::calcButtonCommand(unsigned buttons) const
{
  for(int i = 0; i < 7; ++i)
    if(buttons & (1 << i))
      return buttonCommand[i];
  return "";
}

std::string ConsoleRoboCupCtrl::calcWalk(unsigned x,unsigned y,unsigned r) const
{
  double xJoy = convert(x,0.4);
  int xSpeed = int(xJoy < 0 ? xJoy * 166
                            : xJoy * 200),
      ySpeed = int(convert(y,0.4) * 137),
      rSpeed = int(convert(r,0.2) * toDegrees(1.186));
  char command[100];
  sprintf(command,"mr normal %d %d %d",xSpeed,ySpeed,rSpeed);
  return command;
}

std::string ConsoleRoboCupCtrl::calcHead(unsigned acc,unsigned coolie)
{
  coolie /= 4500;
  const int step = 2;

  int dir[2];
  if(coolie == 7 || coolie == 0 || coolie == 1)
    dir[0] = -1;
  else if(coolie >= 3 && coolie <= 5)
    dir[0] = 1;
  else
    dir[0] = 0;

  if(coolie >= 5 && coolie <= 7)
    dir[1] = -1;
  else if(coolie >= 1 && coolie <= 3)
    dir[1] = 1;
  else
    dir[1] = 0;
  
  if(tiltId == 2)
    tilt = int(convert(acc,0) * -67.5 - 22.5);
  else if(dir[tiltId] < 0)
    tilt = tilt - step < -90 ? -90 : tilt - step;
  else if(dir[tiltId] > 0)
    tilt = tilt + step > 45 ? 45 : tilt + step;

  if(panId == 2)
    pan = int(convert(acc,0) * -90);
  else if(dir[panId] > 0)
    pan = pan - step < -90 ? -90 : pan - step;
  else if(dir[panId] < 0)
    pan = pan + step > 90 ? 90 : pan + step;

  if(rollId == 2)
    roll = int(convert(acc,0) * 30);
  else if(dir[rollId] < 0)
    roll = roll - step < -30 ? -30 : roll - step;
  else if(dir[rollId] > 0)
    roll = roll + step > 30 ? 30 : roll + step;

  char command[100];
  sprintf(command,"hmr %d %d %d",tilt,pan,roll);
  return command;
}

double ConsoleRoboCupCtrl::convert(unsigned value,double threshold) const
{
  double d = (32768 - int(value)) / 32767.0;
  if(d < -threshold)
    return (d + threshold) / (1 - threshold);
  else if(d > threshold)
    return (d - threshold) / (1 - threshold);
  else
    return 0;
}

void ConsoleRoboCupCtrl::createCompletion()
{
  const char* commands[] =
  {
    "sc",
    "sl",
    "call",
    "cls",
    "cp fluorescent high fast",
    "cp fluorescent high mid",
    "cp fluorescent high slow",
    "cp fluorescent low fast",
    "cp fluorescent low mid",
    "cp fluorescent low slow",
    "cp fluorescent mid fast",
    "cp fluorescent mid mid",
    "cp fluorescent mid slow",
    "cp indoor high fast",
    "cp indoor high mid",
    "cp indoor high slow",
    "cp indoor low fast",
    "cp indoor low mid",
    "cp indoor low slow",
    "cp indoor mid fast",
    "cp indoor mid mid",
    "cp indoor mid slow",
    "cp outdoor high fast",
    "cp outdoor high mid",
    "cp outdoor high slow",
    "cp outdoor low fast",
    "cp outdoor low mid",
    "cp outdoor low slow",
    "cp outdoor mid fast",
    "cp outdoor mid mid",
    "cp outdoor mid slow",
    "echo",
    "gc ready",
    "gc ready blue",
    "gc ready red",
    "gc set",
    "gc playing",
    "gc finished",
    "help",
    "jbc",
    "jhc tilt",
    "jhc pan",
    "jhc roll",
    "st off",
    "st on",
    "ci off",
    "ci on",
    "hmr",
    "log start",
    "log stop",
    "log save",
    "log clear",
    "msg off",
    "msg on",
    "msg log",
    "pr continue",
    "pr ballHolding",
    "pr keeperCharged",
    "pr playerCharged",
    "pr illegalDefender",
    "pr illegalDefense",
    "pr obstruction",
    "pr pickup",
    "qfr queue",
    "qfr replace",
    "qfr reject",
    "qfr collect",
    "qfr save",
    "so off",
    "so on",
    "robot all"
  };

  completion.clear();
  const int num = sizeof(commands) / sizeof(commands[0]);
  int i;
  for(i = 0; i < num; ++i)
    completion.insert(std::pair<std::string,int>(commands[i], 0));

  for(List<Robot>::Pos j = robots.getFirst(); j; ++j)
    completion.insert(std::pair<std::string,int>(std::string("robot ") + robots[j].getName(), 0));
  for(List<PhysicalRobot>::Pos k = physicalRobots.getFirst(); k; ++k)
    completion.insert(std::pair<std::string,int>(std::string("robot ") + physicalRobots[k].getName(), 0));

  for(i = 0; i < DebugKeyTable::numOfDebugKeys; ++i)
  {
    completion.insert(std::pair<std::string,int>(std::string("dk ") + DebugKeyTable::getDebugKeyName(DebugKeyTable::debugKeyID(i)), 0));
    completion.insert(std::pair<std::string,int>(std::string("dk ") + DebugKeyTable::getDebugKeyName(DebugKeyTable::debugKeyID(i)) + " off", 0));
    completion.insert(std::pair<std::string,int>(std::string("dk ") + DebugKeyTable::getDebugKeyName(DebugKeyTable::debugKeyID(i)) + " on", 0));
    completion.insert(std::pair<std::string,int>(std::string("dk ") + DebugKeyTable::getDebugKeyName(DebugKeyTable::debugKeyID(i)) + " every", 0));
  }

  for(i = 0; i < GenericDebugData::numOfGenericDebugDataIDs; ++i)
    completion.insert(std::pair<std::string,int>(std::string("sg ") + GenericDebugData::getGenericDebugDateName(GenericDebugData::GenericDebugDataID(i)), 0));

  for(i = 0; i < HeadControlMode::numOfHeadControlModes; ++i) 
    completion.insert(std::pair<std::string,int>(std::string("hcm ") + HeadControlMode::getHeadControlModeName(HeadControlMode::HeadControlModes(i)), 0));

  completion.insert(std::pair<std::string,int>(std::string("mr getup"), 0));
  for(i = 0; i < WalkRequest::numOfWalkType; ++i) 
    completion.insert(std::pair<std::string,int>(std::string("mr ") + WalkRequest::getWalkTypeName(WalkRequest::WalkType(i)), 0));
  for(i = 0; i < SpecialActionRequest::numOfSpecialAction; ++i) 
    completion.insert(std::pair<std::string,int>(std::string("mr ") + SpecialActionRequest::getSpecialActionIDName(SpecialActionRequest::SpecialActionID(i)), 0));

  for(i = 0; i < SolutionRequest::numOfModules; ++i)
    for(int j = -1; j < SolutionRequest::getNumOfSolutions(SolutionRequest::ModuleID(i)); ++j)
      completion.insert(std::pair<std::string,int>(std::string("sr ") + SolutionRequest::getModuleName(SolutionRequest::ModuleID(i)) +
                                                   " " + SolutionRequest::getModuleSolutionName(SolutionRequest::ModuleID(i), SolutionRequest::ModuleSolutionID(j)), 0));

  for(i = 0; i < TailRequest::numOfTailRequests; ++i) 
    completion.insert(std::pair<std::string,int>(std::string("tr ") + TailRequest::getTailRequestIDName(TailRequest::TailRequestID(i)), 0));

  for(i = 0; i < SolutionRequest::numOfXabslBehaviors; ++i)
  {
    completion.insert(std::pair<std::string,int>(std::string("xlb ") + SolutionRequest::getXabsl2EngineFileID(SolutionRequest::xabsl2EngineID(i)), 0));
    completion.insert(std::pair<std::string,int>(std::string("xlb ") + SolutionRequest::getXabsl2EngineFileID(SolutionRequest::xabsl2EngineID(i)) + " update", 0));
  }

  for(i = SolutionRequest::numOfXabslBehaviors; i < SolutionRequest::undefined; ++i) 
  {
    std::string file = SolutionRequest::getXabsl2EngineFileID(SolutionRequest::xabsl2EngineID(i));
    int p = file.find_first_of("/");
    if(p >= 0)
      file = file.substr(p + 1);
    completion.insert(std::pair<std::string,int>(std::string("xlb hc ") + file, 0));
    completion.insert(std::pair<std::string,int>(std::string("xlb hc ") + file + " update", 0));
  }

  static const char* engines[2] = {"", "hc "};

  for(int index = 0; index < 2; ++index)
  {
    const Xabsl2Info* xabsl2Info = xabsl2Infos[index];
    std::string engine = engines[index];
    if(xabsl2Info)
    {
      {
        completion.insert(std::pair<std::string,int>(std::string("xbb ") + engine + "unchanged", 0));
        for(List<Xabsl2Info::BasicBehavior>::Pos i = xabsl2Info->basicBehaviors.getFirst(); i; ++i)
          completion.insert(std::pair<std::string,int>(std::string("xbb ") + engine + xabsl2Info->basicBehaviors[i].name, 0));
      }
      {
        for(List<Xabsl2Info::InputSymbol>::Pos i = xabsl2Info->inputSymbols.getFirst(); i; ++i)
        {
          completion.insert(std::pair<std::string,int>(std::string("xis ") + engine + xabsl2Info->inputSymbols[i].name + " off", 0));
          completion.insert(std::pair<std::string,int>(std::string("xis ") + engine + xabsl2Info->inputSymbols[i].name + " on", 0));
        }
      }
      {
        for(List<Xabsl2Info::OutputSymbol>::Pos i = xabsl2Info->outputSymbols.getFirst(); i; ++i)
        {
          const Xabsl2Info::OutputSymbol& os = xabsl2Info->outputSymbols[i];
          completion.insert(std::pair<std::string,int>(std::string("xos ") + engine + os.name + " off", 0));
          completion.insert(std::pair<std::string,int>(std::string("xos ") + engine + os.name + " on", 0));
          completion.insert(std::pair<std::string,int>(std::string("xos ") + engine + os.name + " unchanged", 0));
          for(List<std::string>::Pos i = os.alternatives.getFirst(); i; ++i)
            if(!strncmp((os.name + ".").c_str(), os.alternatives[i].c_str(), os.name.length() + 1))
              completion.insert(std::pair<std::string,int>(std::string("xos ") + engine + os.name + " " + os.alternatives[i].substr(os.name.length() + 1), 0));
            else
              completion.insert(std::pair<std::string,int>(std::string("xos ") + engine + os.name + " " + os.alternatives[i], 0));
        }
      }
      {
        completion.insert(std::pair<std::string,int>(std::string("xo ") + engine + "unchanged", 0));
        for(List<Xabsl2Info::Option>::Pos i = xabsl2Info->options.getFirst(); i; ++i)
          completion.insert(std::pair<std::string,int>(std::string("xo ") + engine + xabsl2Info->options[i].name, 0));
      }
    }
  }
}

void ConsoleRoboCupCtrl::onConsoleCompletion(std::string& command, bool forward)
{
  std::string constantPart;
  int n = command.rfind(' ');
  if(n != -1)
    constantPart = command.substr(0,n);

  if(forward)
  {
    std::map<std::string, int>::const_iterator i = completion.lower_bound(command + " z");
    if(i == completion.end() || constantPart != i->first.substr(0, constantPart.length()))
    {
      i = completion.lower_bound(constantPart + " ");
      if(i == completion.end() || constantPart != i->first.substr(0, constantPart.length()))
        return;
    }
    command = i->first.substr(constantPart.length());
  }
  else
  {
    std::map<std::string, int>::const_iterator i = completion.lower_bound(command);
    --i;
    if(i == completion.begin() || constantPart != i->first.substr(0, constantPart.length()))
    {
      i = completion.lower_bound(constantPart + "zzzzzzzzzzz");
      --i;
      if(i == completion.begin() || constantPart != i->first.substr(0, constantPart.length()))
        return;
    }
    command = i->first.substr(constantPart.length());
  }
  while(command.length() > 0 && command[0] == ' ')
    command = command.substr(1);

  command = (constantPart == "" ? "" : constantPart + " ") + command.substr(0, (command + " ").find(' ')) + " ";
}

/*
 * Change log :
 *
 * $Log: ConsoleRoboCupCtrl.cpp,v $
 * Revision 1.3  2004/06/03 16:27:04  roefer
 * Adapted to changed implementation of motion request
 *
 * Revision 1.2  2004/05/27 22:33:32  roefer
 * Adaptation to new location handling
 *
 * Revision 1.1.1.1  2004/05/22 17:24:22  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.21  2004/05/18 20:41:03  roefer
 * A few additional tab completion strings
 *
 * Revision 1.20  2004/05/18 11:04:53  roefer
 * Separate view and optional parameter hc for xabsl head control
 *
 * Revision 1.19  2004/05/14 22:56:32  roefer
 * Adapted to new Xabsl engine
 *
 * Revision 1.18  2004/04/20 13:14:53  roefer
 * All console commands now also work outside the start script
 *
 * Revision 1.17  2004/04/14 14:47:57  roefer
 * Only wait one second if a robot cannot be connected
 *
 * Revision 1.16  2004/04/09 18:55:15  roefer
 * Better structure for views
 * cp command added
 * Timing view added
 *
 * Revision 1.15  2004/03/27 15:57:21  roefer
 * Reverse tab completion
 *
 * Revision 1.14  2004/03/16 18:52:05  roefer
 * Tab completion for console commands
 *
 * Revision 1.13  2004/03/15 20:39:39  roefer
 * Adaptation for new game controller
 *
 * Revision 1.12  2004/03/10 23:55:27  roefer
 * ERS7 support for log files
 *
 * Revision 1.11  2004/01/27 23:41:12  roefer
 * off as synonym for disabled added to command sr
 *
 * Revision 1.10  2004/01/17 19:19:18  roefer
 * Simulator calculates robot pose based on class Kinematics now
 *
 * Revision 1.9  2004/01/04 16:53:53  roefer
 * Adaptation to new communication scheme
 *
 * Revision 1.8  2003/12/29 20:57:12  roefer
 * Surrounded SimRobot weakness
 *
 * Revision 1.7  2003/12/29 16:06:33  roefer
 * Ability to log text messages sent from robot added
 *
 * Revision 1.6  2003/12/09 19:12:46  roefer
 * Perform 5 simulation steps in 1
 *
 * Revision 1.5  2003/11/20 17:28:32  roefer
 * xy ? command has optional pattern
 *
 * Revision 1.4  2003/10/25 12:56:50  roefer
 * Startscript uses path now
 *
 * Revision 1.3  2003/10/25 11:52:20  roefer
 * Simulator is SimRobXP now
 *
 * Revision 1.2  2003/10/21 15:05:51  roefer
 * Added the oracle
 *
 * Revision 1.1  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 *
 */
