/**
 * This file contains classes related to receivers.
 * @author Thomas Rfer
 */
#ifndef __RECEIVER_H__
#define __RECEIVER_H__

#ifndef __ProcessFramework_h__
#error Never include this file directly. Include ProcessFramework.h instead.
#endif

#include "Tools/Streams/InStreams.h"

/**
 * The class is the base class for receivers.
 * A receiver is an object that reads packages from Aperios queues.
 * The class manages a global list of all receivers in an Aperios process.
 */
class ReceiverList : public OObserver
{
  private:
    PlatformProcess* process;   /**< The process this receiver is associated with. */
    ReceiverList* next;         /**< The successor of the current receiver. */
    char name[NAME_LENGTH_MAX]; /**< The name of a receiver without the module's name. */
  
  protected:
    int eventId;                /**< The id of the current receiver in the range [0..30]. */
    bool blocking;              /**< Stores whether this is a blocking receiver. */

  public:
    /**
     * The constructor.
     * @param process The process this receiver is associated with.
     * @param receiverName The Aperios connection name of the receiver without the process name.
     * @param blocking Decides whether this receiver blocks the execution of the next frame
     *                 until it has received a package.
     */
    ReceiverList(PlatformProcess* process,const char* receiverName,bool blocking);

    /**
     * Returns the begin of the list of all receivers.
     * @return The first receiver in the list, or 0 if the list ist empty.
     */
    ReceiverList*& getFirst();

    /**
     * Returns the next receiver in the list.
     * @return The next receiver in the list, or 0 if this receiver is the last one.
     */
    ReceiverList* getNext() const {return next;}

    /**
     * Returns the Aperios connection name of the receiver.
     * @return The Aperios connection name without the process name ("Receiver.type.O")
     */
    const char* getName() const {return name;}

    /**
     * Returns whether a new package was received in the current frame.
     * This is always true if this is a blocking receiver.
     * @return Has a new package been received?
     */
    bool receivedNew() const;

    /**
     * The function must be called to finish the current frame.
     */
    void finishFrame();

    /**
     * The functions sets or resets a bit in the blocking mask.
     * After a bit is set in the blocking mask for a certain 
     * sender or receiver, a new frame will not be started before
     * this sender or receiver received an event.
     * @param id The id of the sender or receiver.
     * @param block Should it block or not?
     */
    static void setBlockingId(int id,bool block = true);

    /**
     * The function is called when an event was received. 
     * If this was the last event the process was waiting for, the next
     * frame is started, i.e. NextFrame() is called.
     * @param id The id of the sender or receiver that received an event.
     */
    static void setEventId(int id);

    /**
     * The function is called when a new package arrived.
     * @param msg An Open-R message that contains the package.
     */    
    virtual void handleMessage(ONotifyMessage& msg) = 0;
};

/**
 * The class implements a receiver.
 * A receiver is an object that reads packages from an Aperios queue.
 */
template<class T> class Receiver : public ReceiverList, public T
{
  public:
    /**
     * The function is called when a new package arrived.
     * @param msg An Open-R message that contains the package.
     */    
    void handleMessage(ONotifyMessage& msg)
    {
      ONotifyEvent event;
      event.SetIndex(eventId);
      NotifyHandler(msg,&event);
      T& data = *static_cast<T*>(this);
      InBinaryMemory memory(event.RCData(0)->Base(),event.RCData(0)->Size());
      memory >> data;
      ASSERT(memory.eof());
      setEventId(eventId);
    }

    /**
     * The constructor.
     * @param process The process this receiver is associated with.
     * @param receiverName The Aperios connection name of the receiver without the process name.
     * @param blocking Decides whether this receiver blocks the execution of the next frame
     *                 until it has received a package.
     */
    Receiver(PlatformProcess* process,const char* receiverName,bool blocking)
    : ReceiverList(process,receiverName,blocking)
    {
    }
};


#endif

/*
 * Change log :
 * 
 * $Log: Receiver.h,v $
 * Revision 1.1.1.1  2004/05/22 17:23:55  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.2  2003/04/04 21:28:35  roefer
 * Communication protocol changed
 *
 * Revision 1.1  2002/09/10 15:40:04  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.3  2002/08/14 17:11:23  dueffert
 * adapted by Thomas Roefer to OPENR_SDK-1.1.3-r1 and OPENR_SYS-007
 *
 * Revision 1.2  2002/07/23 16:40:11  roefer
 * Router and SimGT2002 adapted to new message queue and streams
 *
 * Revision 1.1.1.1  2002/05/10 12:40:18  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.1  2002/05/02 17:04:08  roefer
 * New router
 *
 * Revision 1.5  2002/04/21 16:09:02  roefer
 * Anchor removed
 *
 * Revision 1.4  2001/12/15 20:32:08  roefer
 * Senders and receivers are now part of the processes
 *
 * Revision 1.3  2001/12/10 17:47:08  risler
 * change log added
 *
 */
