/**
 * This file contains classes related to processes on the Linux platform.
 * @author Thomas Rfer
 */
#ifndef __ProcessFramework_h__
#define __ProcessFramework_h__

#include "GTAssert.h"
#include "SystemCall.h"
#include <signal.h>
#include <string.h>
#include <sys/msg.h>
#include <unistd.h>
#include <MCOOP.h>
#include <OPENR/OObject.h>
#include <OPENR/OSubject.h>
#include <OPENR/OObserver.h>

const int ENTRY_TABLE_MIN = 4,   /**< The default number of entries in an Aperios entry table. */
          ENTRY_TABLE_MAX = 100, /**< The maximum number of entries in an Aperios entry table. */
          NAME_LENGTH_MAX = 100, /**< The maximum length of Aperios connection names. */
          PROCESS_MAX = 1,       /**< The maximum number of processes in this address context. */
          ROBOT_MAX = 1;         /**< The maximum number of robots in this address context. */

class PlatformProcess;

#include "Sender.h"
#include "Receiver.h"

/**
 * This class is the platform dependent base class for all processes.
 */
class PlatformProcess
{
  private:
    SenderList* firstSender;     /**< The begin of the list of all senders of this process. */
    ReceiverList* firstReceiver; /**< The begin of the list of all receivers of this process. */

  public:
    /**
     * Constructor.
     */
    PlatformProcess()
    {
      firstSender = 0;
      firstReceiver = 0;
    }

    /**
     * The function returns the begin of list of all senders.
     * Note that the function returns a reference that can be changed.
     * @return A reference to the address of the first element. 
     */
    SenderList*& getFirstSender() {return firstSender;}

    /**
     * The function returns the begin of list of all receivers.
     * Note that the function returns a reference that can be changed.
     * @return A reference to the address of the first element. 
     */
    ReceiverList*& getFirstReceiver() {return firstReceiver;}
    
    /**
     * The function returns the index of the current process.
     * @return Under Aperios, the function always returns 0.
     */    
    static int getIndex() {return 0;}

    /**
     * The function returns the index of the robot associated to the current process.
     * @return Under Aperios, the function always returns 0.
     */    
    static int getRobotIndex() {return 0;}
};

/**
 * The class is a helper that allows to instantiate a class as an Aperios process.
 * ProcessBase contains the parts that need not to be implemented as a template.
 * It will only be used by the macro MAKE_PROCESS and should never be used directly.
 */
class ProcessBase : public OObject
{
  private:
    static int blockMask,            /**< A mask with bits set for all blocking senders and receivers. */
               eventMask;            /**< A mask with bits set for all senders and receivers that received an event */
 
    /**
     * The function calls the member ProcessNextFrame() of the only instance of this class.
     */
    static void nextFrame();

  protected:
    static ProcessBase* theInstance; /**< A pointer to the only instance of this class. */
    static int terminate;            /**< The reason why the process was terminated, 0 if still running */

    /**
     * The function is called once for each frame.
     */
    virtual void processNextFrame() = 0;

    /**
     * The function returns the address of the associated process.
     * @return The address of the object.
     */
    virtual PlatformProcess* getProcess() = 0;

  public:
    /**
     * The functions sets or resets a bit in the blocking mask.
     * After a bit is set in the blocking mask for a certain 
     * sender or receiver, a new frame will not be started before
     * this sender or receiver received an event.
     * @param id The id of the sender or receiver.
     * @param block Should it block or not?
     */
    static void setBlockingId(int id,bool block = true);

    /**
     * The function is called when an event was received. 
     * If this was the last event the process was waiting for, the next
     * frame is started, i.e. NextFrame() is called.
     * @param id The id of the sender or receiver that received an event.
     */
    static void setEventId(int id);

    /**
     * Returns the event mask.
     * @return The event mask.
     */
    static int getEventMask() {return eventMask;}

    /**
     * Resets the event mask.
     */
    static void resetEventMask() {eventMask = 0;}

    /**
     * The function is called when the process receives a terminate-signal.
     * @param reason The reason of termination.
     */
    static void onTerminate(int reason)
    {
      ASSERT(reason); // must be != 0
      terminate = reason;
    }
};

/**
 * The class is a helper that allows to instantiate a class as an Aperios process.
 * ProcessCreator contains the parts that need to be implemented as a template.
 * It will only be used by the macro MAKE_PROCESS and should never be used directly.
 */
template<class T> class ProcessCreator : public ProcessBase
{
  private:
    T* process; /**< A pointer to the only instance of the process. */
    const char* name; /**< The name of the process. */
    int lastTime; /**< The last time when Process::Main() was finished. */
    unsigned sleepUntil; /**< The process should sleep until this point in time is reached. */
    SenderList* senders[ENTRY_TABLE_MAX]; /**< Pointers to senders. */
    ReceiverList* receivers[ENTRY_TABLE_MAX]; /**< Pointers to receivers. */
    enum
    {
      INVALID,
      RECEIVER_CONNECT, 
      RECEIVER_NOTIFY,
      SENDER_CONTROL,
      SENDER_READY
    } type[ENTRY_TABLE_MAX]; /**< Determines which function has to be called for a certain id. */

    /**
     * The function registers the global functions for object communication.
     * It is called by OObject::Init().
     * @param event An Open-R event.
     * @return The status. Currently, the function always reports success.
     */
    virtual OStatus DoInit(const OSystemEvent& event)
    {
      setBlockingId(31);
      int id = ENTRY_TABLE_MIN;
      for(ReceiverList* p = process->getFirstReceiver(); p; p = p->getNext())
      {
        ASSERT(strlen(name) + strlen(p->getName()) + 2 <= NAME_LENGTH_MAX);
        char buf[NAME_LENGTH_MAX];
        strcpy(buf,name);
        strcat(buf,".");
        strcat(buf,p->getName());
        receivers[id] = p;
        type[id] = RECEIVER_CONNECT;
        OServiceEntry entry;
        entry.Set(myOID_,id++);
        VERIFY(RegisterServiceEntry(entry,buf) == oSUCCESS);
        receivers[id] = p;
        type[id] = RECEIVER_NOTIFY;
        entry.Set(myOID_,id++);
        VERIFY(p->SetNotifyEntry(entry) == oSUCCESS);
      }
      for(SenderList* q = process->getFirstSender(); q; q = q->getNext())
      {
        ASSERT(strlen(name) + strlen(q->getName()) + 2 <= NAME_LENGTH_MAX);
        char buf[NAME_LENGTH_MAX];
        strcpy(buf,name);
        strcat(buf,".");
        strcat(buf,q->getName());
        senders[id] = q;
        type[id] = SENDER_CONTROL;
        OServiceEntry entry;
        entry.Set(myOID_,id++);
        VERIFY(RegisterServiceEntry(entry,buf) == oSUCCESS);
        senders[id] = q;
        type[id] = SENDER_READY;
        entry.Set(myOID_,id++);
        VERIFY(q->SetReadyEntry(entry) == oSUCCESS);
      }
      return oSUCCESS;
    }

    /**
     * The function starts the process.
     * It is called by OObject::Start().
     * @param event An Open-R event.
     * @return The status. Currently, the function always reports success.
     */
    virtual OStatus DoStart(const OSystemEvent& event) 
    {
      for(ReceiverList* p = process->getFirstReceiver(); p; p = p->getNext())
        p->AssertReady();
      // Call processNextFrame if no blocking receivers are waiting
      setEventId(31);
      return oSUCCESS;
    };

    /**
     * The function stops the process.
     * It is called by OObject::Stop(). However, OObject::Stop() is 
     * never called. Therefore, this function is called neither.
     * @param event An Open-R event.
     * @return The status. Currently, the function always reports success.
     */
    virtual OStatus DoStop(const OSystemEvent& event)
    {
      for(ReceiverList* p = process->getFirstReceiver(); p; p = p->getNext())
        p->DeassertReady();
      if(process)
        delete process;
      process = 0;
      return oSUCCESS;
    }

    /**
     * The function destroys the process.
     * It is called by OObject::Destroy(). However, OObject::Destroy() is 
     * never called. Therefore, this function is called neither.
     * @param event An Open-R event.
     * @return The status. Currently, the function always reports success.
     */
    virtual OStatus DoDestroy(const OSystemEvent& event) {return oSUCCESS;}

    /**
     * The function is called once for each frame.
     * It calls Process::main() and will start a timer if Process::main()
     * returned a nonzero value.
     */
    void processNextFrame()
    {
      int frameTime = process->processMain();
      if(getProcess()->getFirstSender())
        getProcess()->getFirstSender()->finishFrame();
      if(getProcess()->getFirstReceiver())
        process->getFirstReceiver()->finishFrame();
      resetEventMask();
      int currentTime = SystemCall::getCurrentSystemTime();
      if(frameTime < 0)
        frameTime = lastTime - frameTime - currentTime;
      if(frameTime)
        sleepUntil = SystemCall::getCurrentSystemTime() + frameTime;
      else
        sleepUntil = 0;
      setBlockingId(31,frameTime != 0);
      lastTime = currentTime + frameTime;
    }
  
    /**
     * The function returns the address of the associated process.
     * @return The address of the object.
     */
    virtual PlatformProcess* getProcess() {return process;}

  public:
    /**
     * Constuctor. 
     * It is only called from MAKE_PROCESS.
     * @param name The name of the process.
     */
    ProcessCreator(const char* name)
    {
      theInstance = this;
      this->name = name;
      terminate = 0;
      sleepUntil = 0;
      for(int i = 0; i < ENTRY_TABLE_MAX; ++i)
        type[i] = INVALID;
      VERIFY(process = new T);
      lastTime = 0;
    }

    /**
     * The main loop of the process.
     * @param msqid The message id required for communication with other processes.
     * @return The reason for the termination of the process.
     */
    int mainLoop(int msqid)
    {
      signal(SIGINT,onTerminate);
      signal(SIGSEGV,onTerminate);
      signal(SIGTERM,onTerminate);
      while(!terminate)
      { // currentMsg is defined externally
        if(msgrcv(msqid,(msgbuf*)&currentMsg,currentMsg.Size(),msgType,IPC_NOWAIT) > 0)
        {
          void* pMsg = &currentMsg.data;
          switch(currentMsg.sel)
          {
            case 0:
            {
              OReasonMessage& reasonMsg = *(OReasonMessage*)pMsg;
              Init(OSystemEvent(reasonMsg.reason,reasonMsg.param,reasonMsg.paramSize));
              break;
            }
            case 1:
            {
              OReasonMessage& reasonMsg = *(OReasonMessage*)pMsg;
              Start(OSystemEvent(reasonMsg.reason,reasonMsg.param,reasonMsg.paramSize));
              break;
            }
            case 2:
            {
              OReasonMessage& reasonMsg = *(OReasonMessage*)pMsg;
              Stop(OSystemEvent(reasonMsg.reason,reasonMsg.param,reasonMsg.paramSize));
              break;
            }
            case 3:
            {
              OReasonMessage& reasonMsg = *(OReasonMessage*)pMsg;
              Destroy(OSystemEvent(reasonMsg.reason,reasonMsg.param,reasonMsg.paramSize));
              break;
            }
            default:
            {
              ASSERT(currentMsg.sel < ENTRY_TABLE_MAX && type[currentMsg.sel] != INVALID);
              switch(type[currentMsg.sel])
              {
                case RECEIVER_CONNECT:
                  receivers[currentMsg.sel]->ConnectHandler(*(OConnectMessage*)pMsg);
                  break;
                case RECEIVER_NOTIFY:
                  receivers[currentMsg.sel]->handleMessage(*(ONotifyMessage*)pMsg);
                  break;
                case SENDER_CONTROL:
                  senders[currentMsg.sel]->ControlHandler(*(OControlMessage*)pMsg);
                  break;
                case SENDER_READY:
                  senders[currentMsg.sel]->handleMessage(*(OReadyMessage*)pMsg);
              }
            }
          }
        }
        if(sleepUntil && sleepUntil <= SystemCall::getCurrentSystemTime())
        {
          sleepUntil = 0;
          setEventId(31);
        }
        usleep(1000);
      }
      return terminate;
    }
};

/**
 * The macro MAKE_PROCESS instantiates a class as an Aperios process.
 * As a convention, it should be used in the last line of the 
 * source file. In each Aperios process, MAKE_PROCESS must exactly be used
 * once.
 * @param theClass The type of the class to be instantiated.
 */
#define MAKE_PROCESS(theClass) \
  int ProcessBase::blockMask = 0, \
      ProcessBase::eventMask = 0, \
      ProcessBase::terminate = 0; \
  int main(int argc, char *argv[]) \
  { \
    strcpy(myname,#theClass); \
    SetRegistryManagerKey(argv[1]); \
    int msqid = msgget(getpid(),IPC_CREAT | IPC_EXCL | 0666); \
    ASSERT(msqid >= 0); \
    ProcessCreator<theClass> processCreator(#theClass); \
    Terminate(processCreator.mainLoop(msqid)); \
    return 0; \
  } \
  ProcessBase* ProcessBase::theInstance = 0

#endif //__ProcessFramework_h__

/*
 * Change log :
 * 
 * $Log: ProcessFramework.h,v $
 * Revision 1.1.1.1  2004/05/22 17:23:45  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.4  2003/01/22 14:57:06  dueffert
 * doxygen docu corrected
 *
 * Revision 1.3  2002/10/11 11:37:24  roefer
 * Router is working now
 *
 * Revision 1.2  2002/10/07 11:20:26  dueffert
 * gcc3.2 conform casting
 *
 * Revision 1.1  2002/09/10 15:40:04  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/07/23 16:40:11  roefer
 * Router and SimGT2002 adapted to new message queue and streams
 *
 * Revision 1.1.1.1  2002/05/10 12:40:18  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.1  2002/05/02 17:04:08  roefer
 * New router
 *
 * Revision 1.8  2002/02/05 18:08:40  roefer
 * PlatformProcess::getRobotIndex() inserted
 *
 * Revision 1.7  2002/01/15 16:24:32  roefer
 * Debugging now uses PlatformProcess::getIndex() to distiguish between different processes
 *
 * Revision 1.6  2001/12/28 09:00:42  roefer
 * Code for fixed cycle time corrected
 *
 * Revision 1.5  2001/12/15 20:32:08  roefer
 * Senders and receivers are now part of the processes
 *
 * Revision 1.4  2001/12/10 17:47:08  risler
 * change log added
 *
 */
