/**
 * @file Platform/Linux/File.h
 * 
 * Declaration of class File for Linux and Cygwin.
 */

#ifndef __FILE_H__
#define __FILE_H__

#include "GTAssert.h"
#include <stdio.h>
#include <stdarg.h>
#include <string.h>

/**
 * This class provides basic file input/output capabilies. 
 */
class File
{
  private:
    FILE* stream; /**< File handle. */
  public:
    /**
     * Constructor.
     * @param name File name or path. If it is a relative path, it is assumed
     *             to be relative to the path for configuration files. Otherwise,
     *             the path is used directly.
     * @param mode File open mode as used by fopen defined in stdio.h.
     */
    File(const char* name,const char* mode)
    {
      ASSERT(name);
      char buf[FILENAME_MAX];
      if(name[0] != '/' && name[0] != '.')
        sprintf(buf,"%s/CONF/",getGTDir());
      else
        buf[0] = 0;
      ASSERT(strlen(buf) + strlen(name) < FILENAME_MAX);
      strcat(buf,name);
      stream = fopen(buf,mode);
    }

    /**
     * Destructor.
     */
    ~File() {if(stream != 0) fclose(stream);}

    /**
     * The function read a number of bytes from the file to a certain
     * memory location.
     * @param p The start of the memory space the data is written to.
     * @param size The number of bytes read from the file.
     */
    void read(void* p,unsigned size) 
      {VERIFY(!eof());fread(p,1,size,stream);}

    /**
     * The function writes a number of bytes from a certain memory 
     * location into the file.
     * @param p The start of the memory space the data is read from.
     * @param size The number of bytes written into the file.
     */
    void write(const void *p,unsigned size) 
      {VERIFY(fwrite(p,1,size,stream) == size);}
    
    /**
     * The function implements printf for the stream represented by
     * instances of this class.
     * @param format Format string as used by printf defined in stdio.h.
     * @param ... See printf in stdio.h.
     */
    void printf(const char* format, ...)
    {
      va_list args;
      va_start(args,format);
      vfprintf(stream,format,args);
      va_end(args); 
    }

    /**
     * The function returns whether the file represented by an
     * object of this class actually exists.
     * @return The existence of the file.
     */
    bool exists() const {return stream != 0;}

    /**
     * The function returns whether the end of the file represented 
     * by an object of this class was reached.
     * @return End of file reached?
     */
    bool eof() const {return feof(stream) != 0;}

  /**
  * The function returns the current GT directory,
  * e.g. /MS/OPENR/APP or <...>/GT2003 or /usr/local/GT2003
  * @return The current GTDir
  */
  static char* getGTDir() { return "MS/OPEN-R/MW";}
};

#endif // __FILE_H__

/*
 * Change log :
 * 
 * $Log: File.h,v $
 * Revision 1.1.1.1  2004/05/22 17:23:41  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2004/01/22 12:42:26  dueffert
 * flush removed because it doesnt work properly on robots
 *
 * Revision 1.2  2004/01/16 15:46:27  dueffert
 * flush added to File and OutFile
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 11:35:13  juengel
 * comment changed
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.3  2002/11/18 17:32:35  dueffert
 * RobotControl should be startable in any path now
 *
 * Revision 1.2  2002/10/24 12:29:50  roefer
 * Default search path changed
 *
 * Revision 1.1  2002/09/10 15:40:04  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.3  2002/07/23 16:40:11  roefer
 * Router and SimGT2002 adapted to new message queue and streams
 *
 * Revision 1.2  2002/05/14 12:39:04  dueffert
 * corrected some documentation mistakes
 *
 * Revision 1.1.1.1  2002/05/10 12:40:18  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.1  2002/05/02 17:04:08  roefer
 * New router
 *
 * Revision 1.8  2001/12/22 08:25:30  roefer
 * Enhanced constructor of class File, removed second one
 *
 * Revision 1.7  2001/12/14 14:26:32  risler
 * fixed bug with verify in File::read
 *
 * Revision 1.6  2001/12/12 15:19:52  juengel
 * if(stream != 0) im Destruktor vor fclose(stream); ergnzt.
 *
 * Revision 1.5  2001/12/10 17:47:08  risler
 * change log added
 *
 */
