/** 
* @file  Platform/Aperios1.3.2/TCPEndpoint.cpp
*
* Implementation for IP Communication Classes
* @attention this is the Aperios implementation
*
* @author <A href=mailto:robocup@m-wachter.de>Michael Wachter</A>
* 
*/

#include "TCPEndpoint.h"
#include <TCPEndpointMsg.h>
#include <stdio.h>
#include <iostream.h>

#define MAXSENDSIZE 8192

TCPEndpoint::TCPEndpoint(int sendBufferSize, int receiveBufferSize) 
:IPEndpoint(sendBufferSize, receiveBufferSize)
{  
   status=closed;
   address="";
   port=0;
   wasListening=false;
}
       
TCPEndpoint::TCPEndpoint(char* address,int port)
: IPEndpoint()
{
   connect(address, port);
}
  
TCPEndpoint::~TCPEndpoint()
{
   close();
}



/*
 * Start connecting 
 */

int TCPEndpoint::connect(char* address, int port)
{

   antEnvCreateEndpointMsg tcpCreateMsg( EndpointType_TCP, 8192);
   tcpCreateMsg.Call(ipStackRef, sizeof(tcpCreateMsg));

   if (tcpCreateMsg.error != ANT_SUCCESS) 
   {
     cout << "TCPEndpoint::connect(): Can't create Endpoint, error " << antErrorString(tcpCreateMsg.error);
     return(1);
   }

   endpoint = tcpCreateMsg.moduleRef;
   
   // cout << "TCPEndpoint: Connecting to " << address << ":" << port << "\n";

   TCPEndpointConnectMsg connectMsg(endpoint, IP_ADDR_ANY, IP_PORT_ANY,
                                        address, port);

   connectMsg.continuation = this;
   connectMsg.Send(ipStackRef, *myOID_ , *connectContSelector, sizeof(connectMsg));

   status=connecting;
   return(0);
}

void TCPEndpoint::connectCont(antEnvMsg* msg)
{
    // cout << "TCPEndpoint: ConnectCont \n" << flush;
    TCPEndpointConnectMsg* connectMsg = (TCPEndpointConnectMsg*) msg;
    if (connectMsg->error != TCP_SUCCESS)
    {
       cout << "TCPEndpoint::connectCont() Connection failed : " << 
                   getErrorString(connectMsg->error) << " \n";
       status = closed;
    }
    else
    {
       status = connected;
    }


    if(status==connected)
    {
       onConnect();
       startReceiving();
    }
}


void TCPEndpoint::onConnect()
{
    

}

// Start listening
int TCPEndpoint::listen(int listenPort)
{
   antEnvCreateEndpointMsg tcpCreateMsg( EndpointType_TCP, 8192);
   tcpCreateMsg.Call(ipStackRef, sizeof(tcpCreateMsg));

   if (tcpCreateMsg.error != ANT_SUCCESS) 
   {
     cout << "TCPEndpoint::listen() : Can't create Endpoint, error " << antErrorString(tcpCreateMsg.error);
     return(1);
   }

   endpoint = tcpCreateMsg.moduleRef;

  cout << "TCPEndpoint::listen(" << listenPort << ") \n";
  port = listenPort;

  TCPEndpointListenMsg listenMsg(endpoint, IP_ADDR_ANY, port);
  listenMsg.continuation=this;
  VERIFY(*listenContSelector!=0); // Selectors not in construktor
  listenMsg.Send(ipStackRef, *myOID_, *listenContSelector, sizeof(listenMsg));
  
  status=listening;
  
  return(0);
}


// Called by Aperios 
void TCPEndpoint::listenCont(antEnvMsg* msg)
{
  // cout << "TCPEndpoint::listenCont() \n";
  TCPEndpointListenMsg* listenMsg = (TCPEndpointListenMsg*) msg;
  if (listenMsg->error != TCP_SUCCESS)
  {
     cout << "TCPEndpoint::listenCont() : Listen failed : " << 
          getErrorString(listenMsg->error) << "\n";
     
     return;
     
  }
  else
  {
     status=connected;
     wasListening=true;
     onConnect();
     startReceiving();

  }

}

// Sending 

int TCPEndpoint::send(void* data,int size)
{
   // cout << "TCPEndpoint::send(" << data << "," << size<< ") " << SystemCall::getCurrentSystemTime() << " \n";
   sendDataBuffer = data;
   sizeOfDataToSend = size;

   if (status == connected)
   {
       if (status != sending)
       {
           if (size>MAXSENDSIZE) size=MAXSENDSIZE;
           memcpy(sharedSendBuffer,sendDataBuffer,size);  // Copy data into shared memory buffer
     
      //     cout << "TCPEndpoint::send - Transfering " << size << " from " << sendDataBuffer << "\n";
           sendSendMessage(size);

           (char*)sendDataBuffer += size; 
           sizeOfDataToSend-=size;

           return(0);
       }
       else
       {
         return(-1);
       }
   }
   else
   {
     cout << "TCPEndpoint::send() : Not connected \n";
     return(-2);
   }
   
}

void TCPEndpoint::sendCont(antEnvMsg* msg)
{
    // cout << "TCPEndpoint::SendCont " << SystemCall::getCurrentSystemTime() << "\n";

    TCPEndpointSendMsg* sendMsg = (TCPEndpointSendMsg*) msg;
    if (sendMsg->error != TCP_SUCCESS)
    {
        cout << "TCPEndpoint: Send error :" 
             << getErrorString(sendMsg->error) << "\n";
        close();
        return;
    }

    // send was successfull.

    int size = sizeOfDataToSend;
    if (size>MAXSENDSIZE) size=MAXSENDSIZE;
      
    // cout << "TCPEndpoint::SendCont - sending " << size << " bytes of " << sizeOfDataToSend << " from " << sendDataBuffer << "\n";

    if (size > 0)  // Send rest of data
    {
       
       memcpy(sharedSendBuffer,sendDataBuffer,size);  // Copy data into shared memory buffer
       sendSendMessage(size);
       status = sending;

       sizeOfDataToSend-=size;
       (char*)sendDataBuffer += size; 
    }
    else
    {
      status= connected;
      onSendingDone();
     // startReceiving();
    }
}

void TCPEndpoint::sendSendMessage(int size)
{
   // cout << "  SENDING " << size << " BYTES NOW ! ";
   TCPEndpointSendMsg sendMsg(endpoint, sharedSendBuffer, size);
   sendMsg.continuation = this;
   sendMsg.Send(ipStackRef, *myOID_, *sendContSelector, sizeof(sendMsg));
   status = sending;
}

void TCPEndpoint::onSendingDone()
{
  return;
}


void TCPEndpoint::startReceiving()
{
   // cout << "TCPEndpoint::startReceiving " << SystemCall::getCurrentSystemTime() << "\n";
   status = connected;
   TCPEndpointReceiveMsg receiveMsg(endpoint,sharedReceiveBuffer,1,8192);
     // 1 = sizeMin = minumum size of data to be received 
   receiveMsg.continuation = this;
   receiveMsg.Send(ipStackRef, *myOID_, *receiveContSelector, sizeof(receiveMsg));
}

void TCPEndpoint::receiveCont(antEnvMsg* msg)
{
   // cout << "TCPEndpoint::ReceiveCont " << SystemCall::getCurrentSystemTime() << "\n";
 
   TCPEndpointReceiveMsg* receiveMsg = (TCPEndpointReceiveMsg*) msg;
   if (receiveMsg->error != TCP_SUCCESS)
   {
     if (receiveMsg->error == TCP_CONNECTION_BUSY)
     {
       // cout << "TCPEndpoint::ReveiceCont : Connection Busy \n" << flush ;
       return;
     }
     else
     {
         cout << "TCPEndpoint::receiveCont() : Error receiving " << 
             getErrorString(receiveMsg->error) << "\n";
         onClose(receiveMsg->error);
         close();
     return;
     }

   }
 
   // ASSERT(sharedReceiveBuffer == receiveMsg->buffer);

   onReceive(sharedReceiveBuffer,receiveMsg->sizeMin);
   // if (status!=sending)
   {
     startReceiving();
     status=connected;
   }

}

void TCPEndpoint::onReceive(void* Data, int size)
{
 
}

/*
 * Closes the connection
 */

int TCPEndpoint::close()
{
   // cout << "TCPEndpoint::close() \n";
   if ((status != closing) && (status != closed)) 
   {
     TCPEndpointCloseMsg closeMsg(endpoint);
     closeMsg.continuation = this;
     closeMsg.Send(ipStackRef, *myOID_, *closeContSelector, sizeof(closeMsg));
     status=closing;
   }
   return(0);
}
             

void TCPEndpoint::onClose(int reason)
{
  
}

void TCPEndpoint::closeCont(antEnvMsg* msg)
{
    // cout << "TCPEndpoint::closeCont WasListening " << wasListening << "\n";
    TCPEndpointCloseMsg* closeMsg = (TCPEndpointCloseMsg*) msg;
    if (closeMsg->error != TCP_SUCCESS)
    {
      cout << "TCPEndpoint::onClose() : error closing : " 
           << getErrorString(closeMsg->error) << "\n";
    status = closed;
    }
    if (wasListening) 
    {
      listen(port);
    }

}


const char* TCPEndpoint::getErrorString(TCPEndpointError error)
{
 switch (error)
    {
    case TCP_BUFFER_INVALID: return "TCP_BUFFER_INVALID (Address not in shared memory)";
    case TCP_CONNECTION_BUSY: return "TCP_CONNECTION_BUSY (Another Operaation is in Progress)";
    case TCP_CONNECTION_CLOSED: return "TCP_CONNECTION_CLOSED";
    case TCP_CONNECTION_RESET: return "TCP_CONNECTION_RESET";
    case TCP_CONNECTION_TIMEOUT: return "TCP_CONNECTION_TIMEOUT";
    case TCP_FAIL: return "TCP_FAIL (no more information is avialble)";
    case TCP_HOST_UNREACHABLE: return "TCP_HOST_UNREACHABLE";
    case TCP_MESSAGE_TOO_LONG: return "TCP_MESSAGE_TOO_LONG";
    case TCP_NETWORK_UNREACHABLE: return "TCP_NETWORK_UNREACHABLE";
    case TCP_OPERATION_INVALID: return "TCP_OPERATION_INVALID";
    case TCP_OPERATION_UNKNOWN: return "TCP_OPERATION_UNKNOWN";
    case TCP_PORT_UNREACHABLE: return "TCP_PORT_UNREACHABLE";
    case TCP_PROTOCOL_UNREACHABLE: return "TCP_PROTOCOL_UNREACHABLE";
    case TCP_SUCCESS: return "TCP_SUCCESS";
    case TCP_TIME_EXCEEDED: return "TCP_TIME_EXCEEDED";
    case TCP_TTL_EXCEEDED: return "TCP_TTL_EXCEEDED";
          
    default: return "TCP_UNKNOWN_ERROR please edit TCPEndpoint::getErrorSring";
    }
}




/*
 * Change log :
 * 
 * $Log: TCPEndpoint.cpp,v $
 * Revision 1.1.1.1  2004/05/22 17:23:40  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.7  2004/01/26 13:44:07  wachter
 * shared-memory-buffers now have variable sizes
 *
 * Revision 1.6  2004/01/16 16:23:58  wachter
 * Bugfixes
 *
 * Revision 1.5  2004/01/03 18:57:50  wachter
 * Debug-communication working now
 *
 * Revision 1.4  2004/01/03 16:18:25  wachter
 * debug-communication mostly working now
 *
 * Revision 1.3  2003/12/21 19:27:03  wachter
 * Added classes for Sender/Receiver over TCP and UDP.
 * ( PLEASE DO NOT USE THIS NOW ! )
 *
 * Revision 1.2  2003/12/11 15:02:37  wachter
 * Low-level TCP and UDP functions for aperios now working.
 *
 * Revision 1.1  2003/12/03 14:21:52  wachter
 * Splitted IPEndpoint.*
 *
 * 
 */

