/** 
* @file  Platform/Aperios1.3.2/SystemCall.cpp
*
* Static class for system calls, aperios implementation.
* @attention this is the Aperios implementation
*
* @author <A href=mailto:brunn@sim.informatik.tu-darmstadt.de>Ronnie Brunn</A>
* @author <A href=mailto:risler@sim.informatik.tu-darmstadt.de>Max Risler</A>
*/

#include <MCOOP.h>
#include <AnalyzerAPI.h>
#ifdef __GNUC__
#include <ERA201D1.h>
#endif
#include <OPENR/OPower.h>
#include <OPENR/OPENRAPI.h>
#include "SystemCall.h"
#include "GTAssert.h"

unsigned long SystemCall::getCurrentSystemTime() {
  SystemTime sysTime;
  GetSystemTime(&sysTime);
  return (sysTime.seconds * 1000 + sysTime.useconds / 1000);
}

unsigned long SystemCall::getFreeMem()
{
	size_t fmem;
	azrError err=AnalyzerGetSizeOfFreeMemory(&fmem);
	return (err==azrSUCCESS)?fmem:0;
}

unsigned char SystemCall::getRemainingPower()
{
  OPowerStatus stat;
	VERIFY(OPENR::GetPowerStatus(&stat)==0);
  unsigned char percent=stat.remainingCapacity;
  if (stat.robotStatus & orsbEX_POWER_CONNECTED) percent=100;
  return percent;
}

void SystemCall::reboot()
{
	OBootCondition cond(obcbBOOT_TIMER, 0, obcbttRELATIVE);
  VERIFY(OPENR::Shutdown(cond)==0);
}

void SystemCall::shutdown()
{
	OBootCondition cond(0xFFFF);
  VERIFY(OPENR::Shutdown(cond)==0);
}

void SystemCall::getMacAddress(unsigned char address[6])
{
#ifdef __GNUC__
  EtherDriverGetMACAddressMsg msg;
  VERIFY(!ERA201D1_GetMACAddress(&msg));
  for(int i = 0; i < 6; ++i)
    address[i] = msg.address.octet[i];
#else
  for(int i = 0; i < 6; ++i)
    address[i] = 0;
#endif
}

RobotDesign::Design SystemCall::getRobotDesign()
{
  static bool initialized = false;
  static RobotDesign::Design design;

  if (!initialized)
  {
    char robotDesign[20];
    VERIFY(OPENR::GetRobotDesign(robotDesign) == 0);
    if(!strcmp(robotDesign, "ERS-210"))
      design = RobotDesign::ERS210;
    else if(!strcmp(robotDesign, "ERS-7"))
      design = RobotDesign::ERS7;
    else
      design = RobotDesign::UNKNOWN;
    initialized = true;
  }
  return design;
}

/*
 * Change log :
 * 
 * $Log: SystemCall.cpp,v $
 * Revision 1.1.1.1  2004/05/22 17:23:40  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2004/01/05 17:08:25  risler
 * getRobotDesign optimized
 *
 * Revision 1.2  2003/12/31 12:21:25  roefer
 * getRobotDesign added
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/05/05 14:56:42  dueffert
 * Greenhills doesn't seem to know about MAC adresses
 *
 * Revision 1.5  2003/05/05 11:57:30  dueffert
 * using <> for System includes
 *
 * Revision 1.4  2003/05/03 13:05:49  roefer
 * getMacAddress added
 *
 * Revision 1.3  2002/12/11 15:17:14  dueffert
 * getRemainingPower(), shutdown(), reboot() implemented
 *
 * Revision 1.2  2002/12/04 13:43:33  dueffert
 * getFreeMem() implemented
 *
 * Revision 1.1  2002/09/10 15:40:04  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.1.1.1  2002/05/10 12:40:18  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.4  2001/12/10 17:47:08  risler
 * change log added
 *
 */
