/**
 * @file Platform/Aperios1.3.2/Sender.h
 * This file contains classes related to senders.
 * @author Thomas Rfer
 */
#ifndef __SENDER_H__
#define __SENDER_H__

#ifndef __ProcessFramework_h__
#error Never include this file directly. Include ProcessFramework.h instead.
#endif

#include "Tools/MessageQueue/MessageQueue.h"
#include "Platform/GTAssert.h"

const int RECEIVERS_MAX = 20; /**< The maximum number of receivers connected to a single sender */

/**
 * The class is the base class for senders.
 * A sender is an object that sends packages to an Aperios queue.
 * The class manages a global list of all senders in an Aperios process.
 */
class SenderList : public OSubject
{
  private:
    PlatformProcess* process;   /**< The process this sender is associated with. */
    SenderList* next;           /**< The successor of the current sender. */
    char name[NAME_LENGTH_MAX]; /**< The name of a sender without the module's name. */
  
  protected:
    int eventId;                /**< The id of the current sender in the range [0..30]. */
    bool blocking;              /**< Stores whether this is a blocking sender. */

    /**
     * The function sends a package to all receivers that requested it.
     */
    virtual void sendPackage() = 0;

  public:
    /**
     * The constructor.
     * @param process The process this sender is associated with.
     * @param senderName The Aperios connection name of the sender without the process name.
     * @param blocking Decides whether this sender blocks the execution of the next frame
     *                 until all connected receivers have requested a new package.
     */
    SenderList(PlatformProcess* process,const char* senderName,bool blocking);

    /**
     * Returns the begin of the list of all senders.
     * @return The first sender in the list, or 0 if the list ist empty.
     */
    SenderList*& getFirst();

    /**
     * Returns the next sender in the list.
     * @return The next sender in the list, or 0 if this sender is the last one.
     */
    SenderList* getNext() const {return next;}

    /**
     * Returns the Aperios connection name of the sender.
     * @return The Aperios connection name without the process name ("Sender.type.O")
     */
    const char* getName() const {return name;}

    /**
     * Fills the Aperios entry table for global functions.
     * The function will write two entries into the table.
     * @param entryTable The Aperios entry table.
     * @param id The first empty id in the entry table.
     * @return The new first empty id in the entry table.
     */
    virtual int fillEntryTable(ObjectEntry* entryTable,int id) = 0;

    /**
     * The function must be called to finish the current frame.
     */
    void finishFrame();

    /**
     * The functions sets or resets a bit in the blocking mask.
     * After a bit is set in the blocking mask for a certain 
     * sender or receiver, a new frame will not be started before
     * this sender or receiver received an event.
     * @param id The id of the sender or receiver.
     * @param block Should it block or not?
     */
    static void setBlockingId(int id,bool block = true);

    /**
     * The function is called when an event was received. 
     * If this was the last event the process was waiting for, the next
     * frame is started, i.e. NextFrame() is called.
     * @param id The id of the sender or receiver that received an event.
     */
    static void setEventId(int id);
};

/**
 * The class implements a the base class for senders.
 * A sender is an object that sends packages to an Aperios queue.
 * Note that the template parameter is required to force a different
 * instantiation of the static members for each package type.
 */
template<class T> class SenderBase : public SenderList, public T
{
  private:
    static SenderBase<T>* theInstance;         /**< The only instance of SenderBase<T>. */
    ObserverID receiver[RECEIVERS_MAX],        /**< A list of all receivers that are ready to receive a package. */
               alreadyReceived[RECEIVERS_MAX]; /**< A list of all receivers that have already received the current package. */
    int numOfReceivers,                        /**< The number of entries in the receiver list. */
        numOfAlreadyReceived;                  /**< The number of entries in the alreadyReceived list. */

  private:
    /**
     * The function is called when a receiver announces that it is ready to receive a package.
     * @param msg An Open-R message that contains the object id of the receiver.
     */    
    void handleMessage(OReadyMessage& msg)
    {
      ReadyHandler(msg);
      /** 
       * @todo Currently, assert fails in conjunction with wlan, so avoid crash! 
       * However, it may work correctly, because TCPGateway will be the only 
       * receiver. 
       */
      ASSERT(numOfReceivers < RECEIVERS_MAX);
      int i = 0;
      while(i < numOfReceivers && !(receiver[i] == msg.observerID))
        ++i;
      if(i == numOfReceivers && numOfReceivers < NumberOfObservers())
        numOfReceivers++;
      receiver[i] = msg.observerID;
      if(numOfReceivers == NumberOfObservers())
        setEventId(eventId);
    }

    /**
     * The Aperios control handler.
     * @param msg A message provided by Aperios.
     */
    static void aperiosControl(OControlMessage* msg)
    {
      theInstance->ControlHandler(*msg);
      Return();
    }

    /**
     * The Aperios handler that is called when a receiver announces that it is ready to receive a package.
     * @param msg An Open-R message that contains the object id of the receiver.
     */
    static void aperiosReady(OReadyMessage* msg)
    {
      theInstance->handleMessage(*msg);
      Return();
    }

  protected:
    /**
     * The function prepares a package.
     * The function should be pure virtual, but GreenHills does not like that.
     */
    virtual void preparePackage() {}

    /**
     * The functions sets a package for a receiver.
     * The function should be pure virtual, but GreenHills does not like that.
     * @param receiver The receive the package will be sent to.
     */
    virtual void setPackage(const ObserverID& receiver) {}

    /**
     * The function frees the package.
     * The function should be pure virtual, but GreenHills does not like that.
     */
    virtual void freePackage() {}

  private:
    /**
     * The function sends a package to all receivers that requested it.
     */
    void sendPackage()
    {
      if(numOfAlreadyReceived != -1)
      { // send() has been called at least once
        int i;
        for(i = 0; i < numOfReceivers; ++i)
        {
          int j;
          for(j = 0; j < numOfAlreadyReceived; ++j)
            if(receiver[i] == alreadyReceived[j])
              break;
          if(j == numOfAlreadyReceived)
            break; // receiver[i] has not received its requested package yet
        }
        if(i < numOfReceivers)
        { // at least one receiver has not received its requested package, so prepare one
          preparePackage();
          // Send this package to all the receivers that requested it and that have not received it yet
          i = 0;
          while(i < numOfReceivers)
          {
            int j;
            for(j = 0; j < numOfAlreadyReceived; ++j)
              if(receiver[i] == alreadyReceived[j])
                break;
            if(j == numOfAlreadyReceived)
            { // receiver[i] has not received its requested package yet
              setPackage(receiver[i]);
              VERIFY(NotifyObserver(receiver[i]) == oSUCCESS);
              // note that receiver[i] has received the current package
              ASSERT(numOfAlreadyReceived < RECEIVERS_MAX);
              alreadyReceived[numOfAlreadyReceived++] = receiver[i];
              // remove receiver[i] from the list 
              receiver[i] = receiver[--numOfReceivers];
            }
            else 
              ++i;
          }
          // The sender does not need the package anymore
          freePackage();
        }
      }
    }

    
  public:
    /**
     * The constructor.
     * @param process The process this sender is associated with.
     * @param senderName The Aperios connection name of the sender without the process name.
     * @param blocking Decides whether this sender blocks the execution of the next frame
     *                 until all connected receivers have requested a new package.
     */
    SenderBase(PlatformProcess* process,const char* senderName,bool blocking)
    : SenderList(process,senderName,blocking)
    {
      theInstance = this;
      numOfReceivers = 0;
      numOfAlreadyReceived = -1;
    }

    /**
     * Fills the Aperios entry table for global functions.
     * The function will write two entries into the table.
     * @param entryTable The Aperios entry table.
     * @param id The first empty id in the entry table.
     * @return The new first empty id in the entry table.
     */
    int fillEntryTable(ObjectEntry* entryTable,int id)
    {
      eventId = (id - ENTRY_TABLE_MIN) / 2;
      ASSERT(eventId <= 31); // should test on < 31
      setBlockingId(eventId,blocking);
      entryTable[id].selector = id;
      entryTable[id++].entry = (Entry) aperiosControl;
      entryTable[id].selector = id;
      entryTable[id++].entry = (Entry) aperiosReady;
      return id;
    }

    /**
     * Returns whether a new package was requested from the sender.
     * This is always true if this is a blocking sender.
     * @return Has a new package been requested?
     */
    bool requestedNew() const {return numOfReceivers > 0;}

    /**
     * Marks the package for sending and transmits it to all receivers that already requested for it.
     * All other receiver may get it later if they request for it before the package is changed.
     */
    void send()
    {
      setBlockingId(eventId,blocking);
      numOfAlreadyReceived = 0;
      sendPackage();
    }
};

template<class T> SenderBase<T>* SenderBase<T>::theInstance = 0;

/**
 * The class implements the default sender for packages.
 */
template<class T> class Sender : public SenderBase<T>
{
  private:
    RCRegion* package; /**< A pointer to the package. */

  protected:
    /**
     * The function prepares a package.
     */
    virtual void preparePackage()
    {
      const T& data = *static_cast<const T*>(this);
      OutBinarySize size;
      size << data;
      package = new RCRegion(size.getSize());
      ASSERT(package);
      OutBinaryMemory memory(package->Base());
      memory << data;
    }

    /**
     * The functions sets a package for a receiver.
     * @param receiver The receive the package will be sent to.
     */
    virtual void setPackage(const ObserverID& receiver)
    {
      VERIFY(SetData(receiver,package) == oSUCCESS);
    }

    /**
     * The function frees the package.
     */
    virtual void freePackage()
    {
      package->RemoveReference();
    }

  public:
    /**
     * The constructor.
     * @param process The process this sender is associated with.
     * @param senderName The Aperios connection name of the sender without the process name.
     * @param blocking Decides whether this sender blocks the execution of the next frame
     *                 until all connected receivers have requested a new package.
     */
    Sender(PlatformProcess* process,const char* senderName,bool blocking)
    : SenderBase<T>(process,senderName,blocking)
    {
    }
};

#endif

/*
 * Change log :
 * 
 * $Log: Sender.h,v $
 * Revision 1.1.1.1  2004/05/22 17:23:35  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/11/21 20:27:43  kerdels
 * Kommentare hinzugefgt
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.3  2003/04/04 21:28:35  roefer
 * Communication protocol changed
 *
 * Revision 1.2  2002/12/02 11:00:13  dueffert
 * doxygen docu corrected
 *
 * Revision 1.1  2002/09/10 15:40:04  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.7  2002/08/14 17:11:00  dueffert
 * adapted by Thomas Roefer to OPENR_SDK-1.1.3-r1 and OPENR_SYS-007
 *
 * Revision 1.6  2002/07/23 13:39:39  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - new design of debugging architecture
 * - exchanged StaticQueue with MessageQueue
 *
 * Revision 1.5  2002/07/13 10:54:58  roefer
 * New command and sound sender
 *
 * Revision 1.4  2002/07/06 20:08:10  roefer
 * Advances towards gcc
 *
 * Revision 1.3  2002/07/06 15:28:28  roefer
 * Prologue function removed
 *
 * Revision 1.2  2002/05/25 22:52:18  roefer
 * WLan, first working approach
 *
 * Revision 1.1.1.1  2002/05/10 12:40:18  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.11  2002/05/06 16:01:44  hebbel
 * Added buffered Sender
 *
 * Revision 1.10  2002/04/28 19:18:18  giese
 * SoundPlay added...
 *
 * Revision 1.9  2002/04/21 16:09:02  roefer
 * Anchor removed
 *
 * Revision 1.8  2001/12/15 20:32:08  roefer
 * Senders and receivers are now part of the processes
 *
 * Revision 1.7  2001/12/12 13:25:02  roefer
 * Blocking sender fixed
 *
 * Revision 1.6  2001/12/10 17:47:08  risler
 * change log added
 *
 */
