/** 
* @file  Platform/Aperios1.3.2/NetSender.h
*
* @author <A href=mailto:robocup@m-wachter.de>Michael Wachter</A>
*/

#ifndef	_NetSender_H_
#define	_NetSender_H_

/**
 * Base class for NetSenders. Has the template-independent parts
 */

class NetSenderBase
{
   public:
      NetSenderBase() 
      {
         size = 0;
         senderNumber = 0;
         package = NULL;
         sending = false;
      } 

      ~NetSenderBase()
      {
        freePackage();
      }

      virtual void send(void){};
            
      void* getPackage(int newsize)
      {
         if (newsize>size)
         {
            cout << "NetSender::getPackage " << newsize << "\n";
            if (package) free(package);
            package = malloc(newsize);
            size=newsize;
            ASSERT(package);
         }
        return (package);
      }

      void freePackage()
      {
         free(package);
         package = NULL;
         size = 0;
         sending = false;
      }

      void setHandler(NetHandler* h)
      {
         handler = h;
      }

      void sendDone()
      {
         // cout << "NetSender::sendDone() \n" << flush;
         sending=false;
         // cout << "Net Latency = " << SystemCall::getTimeSince (sendStartTimeStamp) << "\n" << flush;
      }
      
   public:
      NetHandler* handler;
      void* package;
      int size;
      bool sending;
      unsigned long sendStartTimeStamp;
      int senderNumber;
};


/**
 * Template class for NetSender
 */

template<class T> class NetSender : public T, public NetSenderBase
{
  public:
    NetSender():T() 
    { 
    }

    ~NetSender() 
    { 
    }
    
    virtual void send(void)
    {
      
      if(!sending)
      {
         T& data = *static_cast<T*>(this);

         // Find out the size of the stream
         OutBinarySize sizeStream;
         sizeStream << data;

         if (sizeStream.getSize() != 0)
         {
              // Get memory to prepare the data to send.
              // At first we send the size of the data and after that the data itself
              getPackage(sizeStream.getSize()+sizeof(int));

              // Store size
              *((int*)package) = sizeStream.getSize();
                       
              // Store data
              OutBinaryMemory memory((char*)package+sizeof(int));
              memory << data;
          
              // Send all this
              sendStartTimeStamp = SystemCall::getCurrentSystemTime();
              sending=true;
              handler->sendPackage(senderNumber,package,sizeStream.getSize()+sizeof(int));
              
           }
        }
     };
      
};


/**
 * Class NetDebugSender
 * Special NetSender for MessageQueues. Clears the data in the 
 * MessageQueue after sending
 */

class NetDebugSender : public NetSender<MessageQueue>
{
  public:
   virtual void send(void)
   {
     if (!sending)
     {
       MessageQueue& data = *static_cast<MessageQueue*>(this);

       // Get size of the Stream
       OutBinarySize sizeStream;
       sizeStream << data;

       int size = data.getStreamedSize();
       // cout << "NetDebugSender::send() Martin-Size = " << size << " Stream-size = " << sizeStream.getSize();

       size=sizeStream.getSize();
       

       if (size != 0)
       {
          // Get memory
          getPackage(size+sizeof(int));

          // Store size
          *((int*)package) = size;
                 
          // Store data
          OutBinaryMemory memory((char*)package+sizeof(int));
          memory << data;
    
          // Send all this
          sendStartTimeStamp = SystemCall::getCurrentSystemTime();
          handler->sendPackage(senderNumber,package,size+sizeof(int));
          sending=true;

          clear(); // of MessageQueue.
       }
     }
     else
     {
       // cout << "NetDebugSender::send() Send blocked " << SystemCall::getCurrentSystemTime() << "\n" << flush;
     }
   }

};



#endif 

/*
 *
 * Changelog :
 *
 * $Log: NetSender.h,v $
 * Revision 1.1.1.1  2004/05/22 17:23:27  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.9  2004/01/21 17:33:08  wachter
 * UDP Team-communication now working with packets<1400 bytes.
 * Not activated at the moment.
 *
 * Revision 1.8  2004/01/20 14:21:41  wachter
 * - Added sender-number to NetSender
 * - worked on with Team-Communication
 *
 * Revision 1.7  2004/01/19 17:09:31  wachter
 * Bugfix
 *
 * Revision 1.6  2004/01/16 16:23:58  wachter
 * Bugfixes
 *
 * Revision 1.5  2004/01/09 13:59:23  wachter
 * Removed problems if TCPHandler is disconnected while sending.
 *
 * Revision 1.4  2004/01/06 10:01:14  wachter
 * Added comments
 *
 * Revision 1.3  2004/01/03 18:57:50  wachter
 * Debug-communication working now
 *
 * Revision 1.2  2004/01/03 16:18:25  wachter
 * debug-communication mostly working now
 *
 * Revision 1.1  2003/12/21 19:27:03  wachter
 * Added classes for Sender/Receiver over TCP and UDP.
 * ( PLEASE DO NOT USE THIS NOW ! )
 *
 *
 */
