/**
 * @file
 * This file contains a sender for motor commands.
 */
#ifndef __MOTORCOMMANDSSENDER_H__
#define __MOTORCOMMANDSSENDER_H__

#ifndef __ProcessFramework_h__
#error Never include this file directly. Include ProcessFramework.h instead.
#endif

#include "Platform/ProcessFramework.h"
#include "Representations/Motion/MotorCommands.h"
#include <OPENR/OPENRAPI.h>

/**
 * This class implements a sender for motor commands.
 */
class MotorCommandsSender : public SenderBase<MotorCommands>
{
private:
  OPrimitiveID jointId[JointData::numOfJoint]; /**< The Open-R IDs of all joints. */
  bool jointGainsSet; /**< A flag that states whether the joint gains were initialized. */
  MemoryRegionID memID; /**< The memory region of the only package instance. */
  OCommandVectorData* cmdVec; /**< A pointer to the data in the package. */
  RCRegion* package; /**< The only instance of a package. */
  PIDData lastPidData; /**< The last joint gains sent. */
  int numOfLED; /**< The number of LEDs of the robot. */

  /**
    * Returns OPENR Primitive Joint Name
    */
  static const char* getPrimitiveJointName(int i) 
  {
    if(SystemCall::getRobotDesign() == RobotDesign::ERS210)
      switch (i) 
      {
        case 0: return "PRM:/r1/c1-Joint2:j1";            // head tilt
        case 1: return "PRM:/r1/c1/c2-Joint2:j2";         // head pan
        case 2: return "PRM:/r1/c1/c2/c3-Joint2:j3";      // head roll
        case 3: return "PRM:/r1/c1/c2/c3/c4-Joint2:j4";   // mouth
        case 4: return "PRM:/r1/c1/c2/c3/e1-Joint3:j5";   // ear L
        case 5: return "PRM:/r1/c1/c2/c3/e2-Joint3:j6";   // ear R
        case 6: return "PRM:/r4/c1-Joint2:j1";            // leg FR joint
        case 7: return "PRM:/r4/c1/c2-Joint2:j2";         // leg FR shoulder
        case 8: return "PRM:/r4/c1/c2/c3-Joint2:j3";      // leg FR knee
        case 9: return "PRM:/r2/c1-Joint2:j1";            // leg FL joint
        case 10: return "PRM:/r2/c1/c2-Joint2:j2";        // leg FL shoulder
        case 11: return "PRM:/r2/c1/c2/c3-Joint2:j3";     // leg FL knee
        case 12: return "PRM:/r5/c1-Joint2:j1";           // leg RR joint
        case 13: return "PRM:/r5/c1/c2-Joint2:j2";        // leg RR shoulder
        case 14: return "PRM:/r5/c1/c2/c3-Joint2:j3";     // leg RR knee
        case 15: return "PRM:/r3/c1-Joint2:j1";           // leg RL joint
        case 16: return "PRM:/r3/c1/c2-Joint2:j2";        // leg RL shoulder
        case 17: return "PRM:/r3/c1/c2/c3-Joint2:j3";     // leg RL knee
        case 18: return "PRM:/r6/c1-Joint2:j1";           // tail pan
        case 19: return "PRM:/r6/c2-Joint2:j2";           // tail tilt
        default: return ""; // not defined
      }
    else
      switch (i) 
      {
        case 0: return "PRM:/r1/c1-Joint2:11";            // head tilt 1
        case 1: return "PRM:/r1/c1/c2-Joint2:12";         // head pan
        case 2: return "PRM:/r1/c1/c2/c3-Joint2:13";      // head tilt 2
        case 3: return "PRM:/r1/c1/c2/c3/c4-Joint2:14";   // mouth
        case 4: return "PRM:/r1/c1/c2/c3/e1-Joint3:15";   // ear L
        case 5: return "PRM:/r1/c1/c2/c3/e2-Joint3:16";   // ear R
        case 6: return "PRM:/r4/c1-Joint2:41";            // leg FR joint
        case 7: return "PRM:/r4/c1/c2-Joint2:42";         // leg FR shoulder
        case 8: return "PRM:/r4/c1/c2/c3-Joint2:43";      // leg FR knee
        case 9: return "PRM:/r2/c1-Joint2:21";            // leg FL joint
        case 10: return "PRM:/r2/c1/c2-Joint2:22";        // leg FL shoulder
        case 11: return "PRM:/r2/c1/c2/c3-Joint2:23";     // leg FL knee
        case 12: return "PRM:/r5/c1-Joint2:51";           // leg RR joint
        case 13: return "PRM:/r5/c1/c2-Joint2:52";        // leg RR shoulder
        case 14: return "PRM:/r5/c1/c2/c3-Joint2:53";     // leg RR knee
        case 15: return "PRM:/r3/c1-Joint2:31";           // leg RL joint
        case 16: return "PRM:/r3/c1/c2-Joint2:32";        // leg RL shoulder
        case 17: return "PRM:/r3/c1/c2/c3-Joint2:33";     // leg RL knee
        case 18: return "PRM:/r6/c2-Joint2:62";           // tail pan
        case 19: return "PRM:/r6/c1-Joint2:61";           // tail tilt
        default: return ""; // not defined
      }

  }

  /**
    * Return OPENR Primitive LED Name
    */
  static const char* getPrimitiveLEDName(int i) 
  {
    if(SystemCall::getRobotDesign() == RobotDesign::ERS210)
      switch (i) 
      {
        case 0: return "PRM:/r1/c1/c2/c3/l1-LED2:l1";
        case 1: return "PRM:/r1/c1/c2/c3/l2-LED2:l2";
        case 2: return "PRM:/r1/c1/c2/c3/l3-LED2:l3";
        case 3: return "PRM:/r1/c1/c2/c3/l4-LED2:l4";
        case 4: return "PRM:/r1/c1/c2/c3/l5-LED2:l5";
        case 5: return "PRM:/r1/c1/c2/c3/l6-LED2:l6";
        case 6: return "PRM:/r1/c1/c2/c3/l7-LED2:l7";
        case 7: return "RPM:/r6/l1-LED2:l1"; //Blue LED
        case 8: return "RPM:/r6/l2-LED2:l2"; //Orange LED
        default: return ""; //not defined
      }
    else
      switch (i) 
      {
        case 0: return "PRM:/r1/c1/c2/c3/l1-LED2:l1"; // Head light(color)
        case 1: return "PRM:/r1/c1/c2/c3/l2-LED2:l2"; // Head light(white)
        case 2: return "PRM:/r1/c1/c2/c3/l3-LED2:l3"; // Mode Indicator(red)
        case 3: return "PRM:/r1/c1/c2/c3/l4-LED2:l4"; // Mode Indicator(green)
        case 4: return "PRM:/r1/c1/c2/c3/l5-LED2:l5"; // Mode Indicator(blue)
        case 5: return "PRM:/r1/c1/c2/c3/l6-LED2:l6"; // Wireless light
        case 6: return "PRM:/r1/c1/c2/c3/la-LED3:la"; // Face light1
        case 7: return "PRM:/r1/c1/c2/c3/lb-LED3:lb"; // Face light2
        case 8: return "PRM:/r1/c1/c2/c3/lc-LED3:lc"; // Face light3
        case 9: return "PRM:/r1/c1/c2/c3/ld-LED3:ld"; // Face light4
        case 10: return "PRM:/r1/c1/c2/c3/le-LED3:le"; // Face light5
        case 11: return "PRM:/r1/c1/c2/c3/lf-LED3:lf"; // Face light6
        case 12: return "PRM:/r1/c1/c2/c3/lg-LED3:lg"; // Face light7
        case 13: return "PRM:/r1/c1/c2/c3/lh-LED3:lh"; // Face light8
        case 14: return "PRM:/r1/c1/c2/c3/li-LED3:li"; // Face light9
        case 15: return "PRM:/r1/c1/c2/c3/lj-LED3:lj"; // Face light10
        case 16: return "PRM:/r1/c1/c2/c3/lk-LED3:lk"; // Face light11
        case 17: return "PRM:/r1/c1/c2/c3/ll-LED3:ll"; // Face light12
        case 18: return "PRM:/r1/c1/c2/c3/lm-LED3:lm"; // Face light13
        case 19: return "PRM:/r1/c1/c2/c3/ln-LED3:ln"; // Face light14
        case 20: return "PRM:/lu-LED3:lu"; // Back light(front, color)
        case 21: return "PRM:/lv-LED3:lv"; // Back light(front, white)
        case 22: return "PRM:/lw-LED3:lw"; // Back light(middle, color)
        case 23: return "PRM:/lx-LED3:lx"; // Back light(middle, white)
        case 24: return "PRM:/ly-LED3:ly"; // Back light(rear, color)
        case 25: return "PRM:/lz-LED3:lz"; // Back light(rear, white)
        default: return ""; //not defined
      }
  }

protected:
  /**
    * The function prepares a package.
    */
  virtual void preparePackage();

  /**
    * The functions sets a package for a receiver.
    * @param receiver The receive the package will be sent to.
    */
  virtual void setPackage(const ObserverID& receiver);

  /**
    * The function frees the package.
    */
  virtual void freePackage();

public:
  /**
    * The constructor.
    * @param process The process this sender is associated with.
    * @param blocking Decides whether this sender blocks the execution of the next frame
    *                 until all connected receivers have requested a new package.
    */
  MotorCommandsSender(PlatformProcess* process,bool blocking);

  /**
    * Destructor.
    */
  ~MotorCommandsSender();
};

/**
 * The macro declares a sender for motor commands.
 * It must be used inside a declaration of a process, after the macro DEBUGGING.
 */
#define SENDER_MOTORCOMMANDS \
  MotorCommandsSender theMotorCommandsSender

/**
 * The macro instantiates a sender for MotorCommands.
 * @param blocking Decides whether this sender blocks the execution of the next frame
 *                 until all connected receivers have requested a new package.
 */
#define INIT_SENDER_MOTORCOMMANDS(blocking) \
  theMotorCommandsSender(this,blocking)

#endif

/*
 * Change log :
 * 
 * $Log: MotorCommandsSender.h,v $
 * Revision 1.1.1.1  2004/05/22 17:23:26  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/12/31 14:29:19  roefer
 * Joints and LEDs for ERS-7
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 15:30:28  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.5  2003/05/16 17:02:17  roefer
 * JAM removed
 *
 * Revision 1.4  2002/12/02 11:00:13  dueffert
 * doxygen docu corrected
 *
 * Revision 1.3  2002/11/19 17:08:48  risler
 * added datatype PIDData
 * support for sending new pid values at runtime
 *
 * Revision 1.2  2002/09/11 17:26:31  loetzsch
 * continued change of module/solution mechanisms
 *
 * Revision 1.1  2002/09/10 15:40:04  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/07/14 14:03:07  roefer
 * First working gcc-version
 *
 * Revision 1.1  2002/07/13 10:54:58  roefer
 * New command and sound sender
 *
 *
 *
*/
