/**
* @file Platform/Aperios1.3.2/MessageQueueBase.h
* 
* Declaration of class MessageQueueBase for Aperios.
*
* @author Martin Ltzsch
*/

#ifndef __MessageQueueBase_h_
#define __MessageQueueBase_h_

#include "Tools/MessageQueue/MessageIDs.h"
#include "Tools/Player.h"

/**
* @class MessageQueueBase 
*
* The platform dependend base of class MessageQueue.
* This version works on a static memory buffer to avoid often memory allocations on the
* Aperios platform. 
*
* Use setSize(int size) to set the total size of data the queue can store. If a message
* doesn't fit into the queue, it is ignored.
*
* @author Martin Ltzsch
*/
class MessageQueueBase
{
public:
  /** Constructor */
  MessageQueueBase();

  /** Destructor */
  ~MessageQueueBase();

  /**
  * Sets the size of the queue. Ignored on the Win32 platform.
  * @param size The maximum size of the queue in Bytes. 
  */
  void setSize(int size);

  /** Removes all messages from the queue */
  void clear();

  /** Returns the number of the messages in the queue */
  int getNumberOfMessages() const;

  /**
  * The function adds a number of bytes to the last message in the queue.
  * @param p The address the data is located at.
  * @param size The number of bytes to be written.
  */
  virtual void write(const void* p,int size);

  /** 
  * Finishes the last message in the queue. 
  * The number of messages becomes increased and a new message can be started.
  * @param id The type id of the message.
  * @param timeStamp The time stamp of the message
  * @param teamColor the team color of the robot that sent the message
  * @param playerNumber the player number of the robot that sent the message 
  * @param messageWasSentFromAPhysicalRobot If true, then the message was sent 
  *     from a physical robot. Is set to false on all non-Aperios/OpenR platforms. 
  */
  void finishMessage(MessageID id, unsigned long timeStamp,
    Player::teamColor teamColor,Player::playerNumber playerNumber,
    bool messageWasSentFromAPhysicalRobot=true);

  /** Returns if the the currently selected message for reading was read completely. */
  bool eof() const; 

  /** 
  * Reads a number of bytes from the currently selected message for reading 
  * @param p The address the data is written to. Note that p
  *          must point to a memory area that is at least
  *          "size" bytes large.
  * @param size The number of bytes to be read.
  */
  void read(void* p,int size); 

  /**
  * Gives the MessageQueue direct read access to the selected message for reading.
  * @return the address of the first byte of the message
  */
  const char* getData() const;

  /** Returns the message id of the currently selected message for reading */
  MessageID getMessageID() const;

  /** 
  * Returns the time stamp of a message.
  * @param message the number of the message
  */
  unsigned long getTimeStamp(int message);

  /** Returns the time stamp of the currently selected message for reading */
  unsigned long getTimeStamp() const;

  /** Returns the message size of the currently selected message for reading */
  int getMessageSize() const;

  /** returns the team color of the robot that sent the currently selected message for reading. */
  Player::teamColor getTeamColor() const;

  /** returns the player number of the robot that sent the currently selected message for reading.  */
  Player::playerNumber getPlayerNumber() const;

  /** returns whether the currently selected message for reading was sent from a physical robot */
  bool getMessageWasSentFromAPhysicalRobot() const;

  /** 
  * Returns the read position of the currently selected message for reading 
  * so that the message can be read again.
  */
  void resetReadPosition();

  /*
  * Sets which message is selected for reading.
  * @param message The number of the message
  */
  void setSelectedMessageForReading(int message);

  /** Deletes all older messages from the queue if a newer
    * message of same type is already in the queue.
    * This method should not be called during message handling. */
  void removeRepetitions();

private:
  /** the buffer on that the queue works */
  char* buf;

  /** the position of the message that is selected for reading */
  int selectedMessageForReadingPosition;

  /** the number of stored messages */
  int numOfMessages;

  /** the queue size */
  int queueSize;

  /** the position where the next message starts (in bytes) */
  int nextMessagePosition;

  /** the current size of the next message */
  int nextMessageSize;

  /** If true, then the writing of the last message failed because there was not enough space */
  bool writingOfLastMessageFailed;

  /** The position up to where a message is already read */
  int readPosition;
};

#endif // __MessageQueueBase_h_

/*
* Change log :
* 
* $Log: MessageQueueBase.h,v $
* Revision 1.1.1.1  2004/05/22 17:23:26  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2003/12/06 23:23:55  loetzsch
* messages in a MessageQueue now contain
* - the team color of the robot which sent the message
* - the player number of the robot which sent the message
* - if the message was sent from a physical robot or not
*
* Revision 1.1  2003/10/07 10:06:59  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.2  2002/12/05 16:13:07  dueffert
* started implementing realtime sending
*
* Revision 1.1  2002/09/10 15:40:04  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.1  2002/07/23 13:36:39  loetzsch
* - exchanged StaticQueue by MessageQueue with platform dependend
*   base classes
*
*/
