/**
* @file YetAnotherInvKinWE.h
* 
* Definition of class YetAnotherInvKinWE and a helper class "Trajectory"
*
* @author Jan Hoffmann
*/

#ifndef __YetAnotherInvKinWE_h_
#define __YetAnotherInvKinWE_h_

#include "WalkingEngine.h"
#include "Tools/Actorics/Kinematics.h"
#include "Tools/Evolution/Individual.h"

#include <string.h>




/**
* @class YetAnotherInvKinWE
*
* Walking engine unsing inverse kinematics.
* Based on the idea that the paws are fixed on the ground
* the trunk is moving... 
*
* @author Jan Hoffmann
*/
class YetAnotherInvKinWE : public WalkingEngine
{
public:
/**
* Constructor
* @param interfaces The paramters of the WalkingEngine module.
  */
  YetAnotherInvKinWE(WalkingEngineInterfaces& interfaces);
  
  /**
  * Destructor
  */
  ~YetAnotherInvKinWE();
  
  /** Executes the engine */
  virtual bool executeParameterized(JointData& jointData, const WalkRequest& walkRequest, double positionInWalkingCycle);
  
  /** 
  * Called from a MessageQueue to distribute messages 
  * @param message The message that can be read.
  * @return true if the message was read (handled).
  */
  virtual bool handleMessage(InMessage& message);
  


private:
	enum whichLeg
	{
		legFR,
		legFL,
		legHR,
		legHL
	} ; //used differentiate fore, hind, left and right legs when computing the inv. kin.

	Vector3<double> 
		pawRel[4], 
		pawAbs[4], 
		shoulderInTrunkCoord[4];

	Vector3<double> trunkPos;

	double trunkTilt, trunkPan, trunkRoll;
	double kickPawLateralOffset, kickSpeed, useBody, kickLength, bodyExtra, kickHeight, usePaw, motionSmoothing;
  
  double localTime;
  
  /** currently executed motion request
  * speeds in mm/s
  */
  Pose2D currentRequest;
  
  /** odometry resulting from current request
  * speed in mm/tick
  */
  Pose2D odometry;

  /** 
  * Calculates the three angles shoulderRotation, shoulderSideway, knee for a robot leg
	* for either a fore or a hind leg (they differ slightly in geometry).
  * @param pawPosition Vector from the shoulder offset to the paw
  * @param legID Decides how the calculation is performed for a fore/hind left/right leg. Use the enum "whichLeg" defined below!
  * @param shoulderRotation One of the three leg joint angles
  * @param shoulderSideway One of the three leg joint angles
  * @param knee One of the three leg joint angles
  * @return true if all went okay.
  */
	bool calculateAngles(
		Vector3<double> pawPosition, 
		int legID, 
		double& shoulderRotation, 
		double& shoulderSideway, 
		double& knee);

};



#endif// __YetAnotherInvKinWE_h_

/*
* Change log :
* $Log: YetAnotherInvKinWE.h,v $
* Revision 1.4  2004/06/07 18:47:50  spranger
* extended interface of executeParametrized by positionInWalkCycle
*
* Revision 1.3  2004/06/04 13:47:03  dueffert
* log added
*
*/
