/**
* @file YetAnotherInvKinWE.cpp
* 
* Implementation of class YetAnotherInvKinWE
*
* @author Jan Hoffmann
*/

#include "YetAnotherInvKinWE.h"
#include "Tools/Player.h"
#include "Tools/RobotConfiguration.h"
#include "Tools/Streams/InStreams.h"
#include "Tools/Actorics/RobotDimensions.h"
#include "Tools/Debugging/Debugging.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Debugging/GenericDebugData.h"
#include "Platform/SystemCall.h"
#include "Tools/Math/Geometry.h"
#include "Tools/FieldDimensions.h"
#include <string.h>


YetAnotherInvKinWE::YetAnotherInvKinWE(WalkingEngineInterfaces& interfaces)
: WalkingEngine(interfaces), 
	trunkPos(0,0,105),
	trunkTilt(0.55), 
  trunkPan(0), 
  trunkRoll(0), 
  kickPawLateralOffset(0), 
  kickSpeed(0),
  kickLength(0),
  bodyExtra(0),
  kickHeight(ballRadius), 
  usePaw(0),
  motionSmoothing(100)
{
  //foreHeight,foreWidth,foreCenterX
	//61.522046, 70.900000, 45.635928,
  //95.742519, 72.600000, -49.733089,
	pawAbs[legFR] = Vector3<double>(+lengthBetweenLegs/2+50.0- 0.0, -bodyWidth/2-18.0, 0.0);
	pawAbs[legFL] = Vector3<double>(+lengthBetweenLegs/2+50.0+ 0.0, +bodyWidth/2+18.0, 0.0);
	pawAbs[legHR] = Vector3<double>(-lengthBetweenLegs/2-10.0+ 0.0, -bodyWidth/2-18.0, 0.0);
	pawAbs[legHL] = Vector3<double>(-lengthBetweenLegs/2-10.0- 0.0, +bodyWidth/2+18.0, 0.0);

	shoulderInTrunkCoord[legFR] = Vector3<double>(+lengthBetweenLegs/2, -bodyWidth/2, 0.0);
	shoulderInTrunkCoord[legFL] = Vector3<double>(+lengthBetweenLegs/2, +bodyWidth/2, 0.0);
	shoulderInTrunkCoord[legHR] = Vector3<double>(-lengthBetweenLegs/2, -bodyWidth/2, 0.0);
	shoulderInTrunkCoord[legHL] = Vector3<double>(-lengthBetweenLegs/2, +bodyWidth/2, 0.0);

  localTime = SystemCall::getCurrentSystemTime();
}


YetAnotherInvKinWE::~YetAnotherInvKinWE()
{
}


bool YetAnotherInvKinWE::executeParameterized(JointData& jointData,
	const WalkRequest& walkRequest, double positionInWalkingCycle)
{
	double shoulderRotation, shoulderSideways, knee;
  Vector3<double> trunk = trunkPos, paw;
	RotationMatrix trunkRot;
	trunkRot.fromKardanRPY(trunkPan, trunkTilt, trunkRoll);

  double time;

  if (kickSpeed)
  {
    int discreteTime = 2000 + (100-(int)kickSpeed) * 40;
    time = ((double) (SystemCall::getCurrentSystemTime()%discreteTime)) / (discreteTime / 2);
    localTime = time;
  }
  else 
  {
    time = localTime;
  }
 
  //use front right leg for kicking! 
  paw = pawAbs[legFR];
  paw.y -= kickPawLateralOffset;
  paw.z -= kickHeight;


  if (time > .750)
  {
    //paw.x -= (kickLength+bodyExtra);  
    // trunkPos.x = 30 * 0; 
    // do nothing
  }
  else if (time > .25)
  {
    double f = cos((time + .25) * pi2);
    paw.x -= (kickLength+bodyExtra) * (f-1)*.5; 
    trunk.x = trunkPos.x + bodyExtra * (f-1)*.5;
  }
  else
  {
    double f = (1 - 8*time);
    paw.x -= (kickLength+bodyExtra) * f; 
    trunkPos.x = trunkPos.x + bodyExtra * f;
  }


  int i=0;
  if (usePaw) 
  {
    i=1;
    paw -= (trunkRot * shoulderInTrunkCoord[legFR]) - trunk;
    calculateAngles(paw, legFR, shoulderRotation, shoulderSideways, knee);
    jointData.data[JointData::legFR1] = (long )shoulderRotation;
    jointData.data[JointData::legFR2] = (long )shoulderSideways;
    jointData.data[JointData::legFR3] = (long )knee;
  }

	for (; i < 4; i++)
	{   			
    pawRel[i] = pawAbs[i]-(trunkRot * shoulderInTrunkCoord[i]) + trunk;
    calculateAngles(pawRel[i], i, shoulderRotation, shoulderSideways, knee);
    if (motionSmoothing)
    {
      double jointDataBuffer;
      jointDataBuffer = ((double )jointData.data[JointData::legFR1 + i*3]) * (motionSmoothing)
        + shoulderRotation * (100 - motionSmoothing);
      jointDataBuffer /= 100;
      jointData.data[JointData::legFR1 + i*3] = (long )jointDataBuffer;

      jointDataBuffer = ((double )jointData.data[JointData::legFR2 + i*3]) * (motionSmoothing)
        + shoulderSideways * (100 - motionSmoothing);
      jointDataBuffer /= 100;
	    jointData.data[JointData::legFR2 + i*3] = (long )jointDataBuffer;

      jointDataBuffer = ((double )jointData.data[JointData::legFR3 + i*3]) * (motionSmoothing)
        + knee * (100 - motionSmoothing);
      jointDataBuffer /= 100;
	    jointData.data[JointData::legFR3 + i*3] = (long )jointDataBuffer;
    }
    else
    {
	    jointData.data[JointData::legFR1 + i*3] = (long )shoulderRotation;
	    jointData.data[JointData::legFR2 + i*3] = (long )shoulderSideways;
	    jointData.data[JointData::legFR3 + i*3] = (long )knee;
    }
	}

  jointData.data[JointData::neckTilt] = jointDataInvalidValue;
  jointData.data[JointData::headPan] = jointDataInvalidValue;
  // headRoll (ERS210) = headTilt2 (ERS7)
  jointData.data[JointData::headTilt] = jointDataInvalidValue;

  return true;
}



bool YetAnotherInvKinWE::calculateAngles(
	Vector3<double> pawPosition, 
	int legID,
	double& shoulderRotation, 
	double& shoulderSideway, 
	double& knee)
{
	double 
		upperLeg, 
		lowerLeg,
		lowerLegX,	// this is different for fore and hind legs!
		legConst,
		lengthShoulderToPaw, 
		r, s, t, u;		// these are geometric properties of the triangle involved, see Mathematica file for details 


	// some symmetry related changes to the 
	// positions
	if ((legID == legHL) || (legID == legHR))
	{
		pawPosition.y *= -1;
		lowerLegX = lowerHindLegLength;
	}
	else
	{
		pawPosition.x *= -1;
		lowerLegX = lowerForeLegLength;
	}

	if ((legID == legHL) || (legID == legFR))
	{
		pawPosition.y *= -1;
	}

	lengthShoulderToPaw = pawPosition.abs();

	upperLeg = upperLegLength;
	lowerLeg = sqrt(lowerLegX*lowerLegX + upperLegLengthX*upperLegLengthX);

	// geometric properties of the leg "triangle" used in the later calculations
	s = (upperLeg + lowerLeg + lengthShoulderToPaw)/2;
	u = (s-upperLeg)*(s-lowerLeg)*(s-lengthShoulderToPaw)/s;
	if (u > 0) r = sqrt(u);
	else 
	{
		r = 1; // there might be a smarter way of doing this?!
	}
	t = lengthShoulderToPaw*cos(zeroShoulderArc + 2*atan(r/(s-lowerLeg)));

	legConst = atan2(upperLegLengthX,lowerLegX)+zeroShoulderArc;
		
	knee							= 2*atan2(-r, -(lengthShoulderToPaw-s)) + legConst + pi;
	shoulderSideway		= asin(pawPosition.y/t);
	shoulderRotation	= -sgn(pawPosition.x)*acos(pawPosition.z/sqrt(pawPosition.x*pawPosition.x + pawPosition.z*pawPosition.z)) 
											- acos( t*cos(shoulderSideway)/sqrt(pawPosition.x*pawPosition.x + pawPosition.z*pawPosition.z));
  //to microrad
	shoulderRotation	*=  1000000;
	shoulderSideway		*=  1000000;
	knee							*=  1000000;
	return true;
}



bool YetAnotherInvKinWE::handleMessage(InMessage& message)
{
	bool handled = false;
  GenericDebugData d;
	
  switch(message.getMessageID())
  {
  case idYetAnotherInvKinParams:
    message.bin >> d;
		if(d.id == GenericDebugData::yetAnotherInvKinParams)
		{   
			OUTPUT(idText,text,"generic debug message (yetAnotherInvKinWE) handled by module yetAnotherInvKinWE");
    
			trunkPos.x = d.data[0];
			trunkPos.y = -d.data[1];
			trunkPos.z = d.data[2];

			trunkTilt = d.data[3] - 10;
			trunkPan	= d.data[4] - 10;
			trunkRoll = d.data[5] - 10;

      kickPawLateralOffset = d.data[6] - 20;
      kickSpeed = d.data[7];
      
      bodyExtra = d.data[8];
      kickLength = d.data[9];

//			centerOfMotion[legFR].x = d.data[6];
//			centerOfMotion[legFL].x = d.data[6];
//			centerOfMotion[legHR].x = -d.data[7];
//			centerOfMotion[legHL].x = -d.data[7];
			
		}
		handled = true;
		break;
  case idYetAnotherInvKinPaws:
    message.bin >> d;
		if(d.id == GenericDebugData::yetAnotherInvKinPaws)
		{   
			OUTPUT(idText,text,"generic debug message (yetAnotherInvKinWE) handled by module yetAnotherInvKinWE");

      double lateral, forward;

      forward = lengthBetweenLegs/2 + d.data[0];
      lateral = bodyWidth/2 + d.data[1];

      pawAbs[legFR].x = forward;
      pawAbs[legFR].y = -lateral;
      pawAbs[legFL].x = forward;
      pawAbs[legFL].y = lateral;

      forward = d.data[2];
      lateral = bodyWidth/2 + d.data[3];

      pawAbs[legHR].x = -forward;
      pawAbs[legHR].y = -lateral;
      pawAbs[legHL].x = -forward;
      pawAbs[legHL].y = lateral;

      kickHeight = d.data[4];
      usePaw = d.data[5];
      motionSmoothing = d.data[6];
		}
		handled = true;
		break;

	}
	return handled;
}





/*
 * Change log :
 * 
 * $Log: YetAnotherInvKinWE.cpp,v $
 * Revision 1.4  2004/06/07 18:47:50  spranger
 * extended interface of executeParametrized by positionInWalkCycle
 *
 * Revision 1.3  2004/06/02 17:18:23  spranger
 * MotionRequest cleanup
 *
 * Revision 1.2  2004/05/27 17:13:37  jhoffman
 * - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
 * - clipping included for setJoints
 * - removed some microrad/rad-bugs
 * - bodyPosture constructor and "=" operator fixed
 *
 * Revision 1.1.1.1  2004/05/22 17:23:17  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.11  2004/02/25 13:47:05  jhoffman
 * *** empty log message ***
 *
 * Revision 1.10  2004/02/17 10:05:07  kerdels
 * error removed
 *
 * Revision 1.9  2004/02/16 23:51:57  roefer
 * Warnings removed
 *
 * Revision 1.8  2004/02/16 19:22:47  jhoffman
 * added debug parameters for yet another inv kin
 *
 * Revision 1.7  2004/02/13 09:46:46  jhoffman
 * added experimental kicking stuff
 *
 * Revision 1.6  2004/02/12 10:59:31  jhoffman
 * stripped code of anything unnecessary. Walking engine now does nothing but "stand"
 *
 * Revision 1.5  2004/01/28 21:55:49  roefer
 * RobotDimensions revised
 *
 * Revision 1.4  2003/12/05 08:04:08  jhoffman
 * added normalize() to Vector3
 *
 * Revision 1.3  2003/11/24 09:31:39  jhoffman
 * worked on walking engine (paw placement,  trajectories, ...)
 *
 * Revision 1.2  2003/11/22 15:59:03  jhoffman
 * parameter sending using generic debug data to
 * walking engine now works if it is explicitely given
 * a messageID
 *
 * Revision 1.1  2003/11/22 09:33:01  jhoffman
 * added "Yet Another Inv Kin Walking Engine"
 * experimental stage, doesn't do much yet
 * no Fourier inside!
 *
 */
