/**
* @file UNSW2003WalkingEngine.h
* 
* Definition of class UNSW2003WalkingEngine
*/

#ifndef __UNSW2003WalkingEngine_h_
#define __UNSW2003WalkingEngine_h_

#include "WalkingEngine.h"

/* The number of walk locus end-points calculated per leg for each camera frame */
static const int numOfFrames = 1;

/* Robot body length constants */
static const double l1  = 64;
static const double l2  = 68.41286429;
static const double l3  = 12.8;
static const double lsh = 119;
static const double l4  = 75.92865072;

/**
* @class UNSW2003WalkingEngine
*
* A Walking Engine copied from the UNSW's 2003 code release
*/
class UNSW2003WalkingEngine : public WalkingEngine
{
public:
/**
* Constructor.
* @param interfaces The paramters of the WalkingEngine module.
  */
  UNSW2003WalkingEngine(const WalkingEngineInterfaces& interfaces);
  
  /**
  * Destructor
  */
  ~UNSW2003WalkingEngine();
  
  /** Executes the engine */
  virtual bool executeParameterized(JointData& jointData, const WalkRequest& walkRequest, double positionInWalkingCycle);
  
  /** 
  * Called from a MessageQueue to distribute messages 
  * @param message The message that can be read.
  * @return true if the message was read (handled).
  */
  virtual bool handleMessage(InMessage& message);
  
private:
  void pStep (
    int walktype, 
    
    /* specifies the distance in cm that the robot should move forward in each half-step
    * Negative values move the robot backwards
    **/
    double forward, // the length in cm to move forward per step
    
                    /* specifies the distance in cm that the robot should move to the left in each half-step
                    * Negative values move the robot backwards
                    **/
                    double left,    // the length in cm to move left per step
                    
                                    /* specifies the angle in degrees that the robot should turn counter clockwise per half-step.
                                    * Negative values move the robot clockwise
                                    **/
                                    double turnCCW, // the angle in degrees to turn counter clockwise per step
                                    
                                    int speed,
                                    
                                    double hf,
                                    double hb,
                                    double hdf,
                                    double hdb,
                                    double ffo,
                                    double fso,
                                    double bfo,
                                    double bso,
                                    JointData& jointData
                                    );
                                    /* Note: * = this is actually not entirely true since there's a small bug in the PWalk code
                                    * where one of the formulae is not correctly translated;
                                    * the values of hdf and hdb actually have to be set to be a bit higher than their intended values
  **/
  
  void readLocusOffets(const char *filename);
  
  double delta_forward, delta_left, delta_turn;
  
  bool pwalkNotUsingHead;
  
  
  //const pointers so that people on the outside can read 
  //walktype, step_, PG, but not write to it.
  const int* ptr_walktype;
  const int* ptr_step_;
  const int* ptr_PG;
  
  enum { NormalWalkWT, CanterWalkWT, ZoidalWalkWT, OffsetWalkWT };
  
  double fc, sc, tc; /* forward, left & turn left control in mm */
  int step_;         /* one step_ per frame */
  
                     /* height forward, the vertical height in mm from the shoulder joint to the ground
                     * The point on the ground vertically below the shoulder joint is called "shoulder ground"
  **/
  double hf;
  
  /* height back, the vertical height in mm from the hip joint to the ground
  * The point on the ground vertically below the hip joint is called "hip ground"
  **/
  double hb;
  
  /* height delta forward, the height in mm that the robot lifts its front paws when walking,
  * that is, the height of the walk locus for the front paws.
  **/
  double hdf;
  
  /* height delta back, the height in mm that the robot lifts its back paws when walking,
  * that is, the height of the walk locus for the back paws.
  **/
  double hdb;
  
  /* front forward offset, home position of the front paws in mm forward from shoulder ground
  **/
  double ffo;
  
  /* front sideways offset, home position of the front paws in mm sideways from shoulder ground.
  * Larger values spread the legs outwards, smaller values bring them in closer
  **/
  double fso;
  
  /* back forward offset, home position of the back paws in mm forward from hip ground 
  **/
  double bfo;
  
  /* back sideways offset, home position of the back paws in mm sideways from hip ground.
  * Larger values spread the legs outwards, smaller values bring them in closer
  **/
  double bso;
  
  double hfG;
  double hbG;
  
  /* period ground, the time taken to complete a half-step 
  * (for the robot to move along the ground, or the bottom edge of the walk locus
  * PG = numOfFrames * number of camera frames per half-step
  **/
  int PG;            /* steps per ground stroke */
  
  double theta5, sin5, cos5;
  
  double jfl1, jfl2, jfl3, jfr1, jfr2, jfr3, jbl1, jbl2, jbl3, jbr1, jbr2, jbr3;
  
  /* specifies the actual type of locomotion action to execute. 
  * Walktypes are definitions used to decide whether to execute a particular walking style or kicking action.
  * (exception: aimable forward kick is a kicking action that ignore all other parameter values)
  **/
  int walktype;
  
  /* locus point offsets
  */
  double bfl1, bsl1, bhl1, cfl1, csl1, chl1, dfl1, dsl1, dhl1, efl1, esl1, ehl1;
  double bfl2, bsl2, bhl2, cfl2, csl2, chl2, dfl2, dsl2, dhl2, efl2, esl2, ehl2;
  
  enum { numOfBuffer = 2};
  enum { numOfJoint = 15 };
  
  long microRad (double);
  double abs (double);
  double microRad2Deg (long);
  
  
  static inline double forwardCalib(const double forward, const double /*left*/, const double /*turnCCW*/) {
    double fc = 0;
    
    if (forward > 0) {
      fc = 5.02 * forward + 2.5614;
    } else if (forward < 0) {
      fc = 4.62 * forward - 6.0006;
    }
    
    return fc;
  }
  
  static inline double leftCalib(const double /*forward*/, const double left, const double /*turnCCW*/) {
    double sc = 0;
    
    if (left == 0) {
      sc = 0;
    } else if (left > 7.2) {
      sc = 52.33768448; 
      
      /* 11.1632576; 
      0.0172*pow(left, 3)[6.4198656] - 0.0028*pow(left, 2)[0.145152] +
      0.6482*left -0.0688[4.59824]; */
      
    } else if (left < -7.2) {
      sc = -52.33768448; 
      /* -11.1632576; */
    } else {
      sc = 0.0035 * pow(left,5) + 0.0016 * pow(left,4)
        - 0.1672 * pow(left,3) - 0.0771 * pow(left,2)
        + 6.4649 * left + 0.1724;
      
        /* sc = 0.0035*pow(left, 5)[67.72211712] +
        0.0016*pow(left, 4)[4.29981696] - 0.1672*pow(left, 3)[62.4070656] - 0.0771*pow(left, 2)[3.996864] +
      6.4649*left + 0.1724[46.71968]; */
      
    }
    
    return sc;
  }
  
  static inline double turnCCWCalib(const double forward, const double left, const double turnCCW) {
    double tc = 0;
    if (turnCCW > 0) {
      tc = 0.953 * turnCCW; 
      /* + 2.4048; */
    } else if (turnCCW < 0) {
      tc = 0.951 * turnCCW;
    }
    /* - 3.6498; */
    
    if (left < 0 && forward >= 0) {
      tc -= 2.4;
    }
    
    if (left == 0 && forward == 0 && turnCCW > 0) {
      tc += 2.4048;
    } else if (left == 0 && forward == 0 && turnCCW < 0) {
      tc -= 3.6498;
    }
    
    return tc;
  }
  
  inline long micro(double val)
  {
    return (long)(val * 1000000.0);
  }
  
};

#endif// __UNSW2003WalkingEngine_h_

/*
* Change log :
* 
* $Log: UNSW2003WalkingEngine.h,v $
* Revision 1.3  2004/06/07 18:47:50  spranger
* extended interface of executeParametrized by positionInWalkCycle
*
* Revision 1.2  2004/06/02 17:18:23  spranger
* MotionRequest cleanup
*
* Revision 1.1.1.1  2004/05/22 17:23:11  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2004/03/08 02:29:49  roefer
* Interfaces should be const
*
* Revision 1.1  2003/10/24 15:01:25  loetzsch
* added UNSW2003WalkingEngine
*
*/
