/**
* @file UNSW2003WalkingEngine.cpp
* 
* Implementation of class UNSW2003WalkingEngine
*/

#include "UNSW2003WalkingEngine.h"

UNSW2003WalkingEngine::UNSW2003WalkingEngine(const WalkingEngineInterfaces& interfaces)
: WalkingEngine(interfaces)
{
  fc = 0;
  sc = 0;
  tc = 0; /* no movement */
  
  /* see thesis for parameter definitions */
  PG    = 40; 
  step_ = 2 * PG;
  hf    = 70;
  hb    = 110;
  hdf   = 5;
  hdb   = 25;
  hfG   = hf;
  hbG   = hb;
  ffo   = 55;
  fso   = 15;
  bfo   = -55;
  bso   = 15;
  
  theta5 = atan (119.4 / lsh);
  sin5   = sin (theta5);
  cos5   = cos (theta5);
  
  jfl1 = 0;
  jfl2 = 0;
  jfl3 = 0;
  jfr1 = 0;
  jfr2 = 0;
  jfr3 = 0;
  jbl1 = 0;
  jbl2 = 0;
  jbl3 = 0;
  jbr1 = 0;
  jbr2 = 0;
  jbr3 = 0;
  
  walktype = CanterWalkWT;
  
  delta_forward = delta_left = delta_turn = 0;
  
  pwalkNotUsingHead = true;
  
  ptr_walktype = &walktype;
  ptr_step_ = &step_;
  ptr_PG = &PG;
  
  readLocusOffets("/MS/offsets.txt");
}

UNSW2003WalkingEngine::~UNSW2003WalkingEngine()
{
}

bool UNSW2003WalkingEngine::executeParameterized(JointData& jointData,
                                                 const WalkRequest& walkRequest,
                                                 double positionInWalkingCycle)
{
  double forward = walkRequest.walkParams.translation.x/26.5;
  double left = walkRequest.walkParams.translation.y/26.5;
  double turnCCW = walkRequest.walkParams.rotation*(17.8);
  
  //2do: these maxima are for one direction at a time only: for combinations they might be too high
  if (forward > 8.3) forward=8.3;
  if (forward < -8.3) forward=-8.3;
  if (left > 7) left=7;
  if (left < -7) left=-7;
  if (turnCCW > 35) turnCCW = 35;
  if (turnCCW < -35) turnCCW = -35;

  double factor=sqrt((forward/8.3)*(forward/8.3)+(left/7)*(left/7)+(turnCCW/35)*(turnCCW/35));
  if (factor>1)
  {
    forward /= factor;
    left /= factor;
    turnCCW /= factor;
  }

  pStep(CanterWalkWT,forward,left,turnCCW,
    40,70,110,5,25,55,15,-55,15,jointData);

  Pose2D odometry;
  // convert odometry from mm/s to mm/tick
  odometry.translation.x = forward*26.5  * 8.0 / 1000.0;
  odometry.translation.y = left*26.5     * 8.0 / 1000.0;
  // convert odometry rad/s to rad/tick
  odometry.rotation = turnCCW/17.8       * 8.0 / 1000.0;
  odometryData.conc(odometry);
  
  motionInfo.neckHeight = 127;
  motionInfo.motionIsStable = true;
  
  return (step_>1); //false = other motion possible... should we always return (false) instead?
}


bool UNSW2003WalkingEngine::handleMessage(InMessage& message)
{
  return false;
}

void UNSW2003WalkingEngine::pStep(int wT, double forward, double left, double turnCCW,
                                  int speed, double hF, double hB, double hdF, double hdB,
                                  double ffO, double fsO, double bfO, double bsO,
                                  JointData& jointData) 
{
  walktype = wT;
  
  jointData.data[JointData::neckTilt] = jointDataInvalidValue;
  jointData.data[JointData::headPan] = jointDataInvalidValue;
  jointData.data[JointData::headTilt] = jointDataInvalidValue;
  jointData.data[JointData::mouth] = jointDataInvalidValue;
  pwalkNotUsingHead = true;
  
  /* Setting Data - Legs */
  if (step_ == PG || step_ == 2 * PG) 
  {
    /* reset step_ at end of full stepcycle */
    if (step_ >= 2 * PG) 
    {
      step_ = 0;
      PG = speed;
    }
    
    hfG = hF;
    hbG = hB;
    ffo = ffO;
    fso = fsO;
    bfo = bfO;
    bso = bsO;
    hdf = hdF;
    hdb = hdB;
    
    if (walktype == CanterWalkWT) 
    {
      /* Calibration of canter walk */
      
      if (left > 7.2) 
      {
        left = 7.2;
      } 
      else if (left < -7.2) 
      {
        left = -7.2;
      }
      
      fc = forwardCalib(forward, left, turnCCW);
      sc = leftCalib(forward, left, turnCCW);
      tc = turnCCWCalib(forward, left, turnCCW);
    } 
    else if (walktype == NormalWalkWT) 
    {
      /* Calibration of normal walk */
      fc = (forward + .04) * 6.0;
      sc = left * 6.0;
      tc = turnCCW * 1.5 + sc / 6.0;
      /* last term is to correct turning on sideways movements BH 3/8/00 */
    } 
    else if (walktype == ZoidalWalkWT) 
    {
      /* Calibration of trapezoidal walk */
      fc = (forward + .04) * 5.4;
      sc = left * 4.5;
      tc = turnCCW + left * 6 / 4;
    } 
    else if (walktype == OffsetWalkWT) 
    {
    /* Calibration of OffsetWalkWT
    *
    * NOTE: this calibration is NOT ACCURATE - it has specifically been set for easiest behaviour level
    * integration (for replacing ZoidalWalk)
    * In particular, walking sideways is awful (most likely caused by the offsets being calibrated for
    * walking forwards)
      */
      turnCCW *= 1.2;
      fc = (forward / 8.6666*6 + .04) * 6.0;
      sc = left * 6.0;
      tc = turnCCW / 18.75 * 10 * 1.5 + sc / 6.0 + fc/100;
    }
    
    delta_forward = (double) numOfFrames / PG * forward;
    delta_left = (double) numOfFrames / PG * left;
    delta_turn = (double) numOfFrames / PG * turnCCW;
  }
  
  /* Now calculate transform to theta 1,2 & 3
  * (see thesis for explanation)
  */
  double theta1, theta2, theta3, tempu, tempv, x, y, z, temp1, temp2;
  double theta4, f = 0, s = 0, dd = 0;
  double h = 0, fstart, fend, sstart, send, fstep, sstep, hstep;
  double PGH, PGR;	// Will: XXX: This was defined as int.  Is this change to double ok?
  
  /* step = step_ 180 deg out of phase */
  int step; 
  
  /* this is a magic part of PWalk -- canter,
  * it moves the robots' motor up and down in a sine fashion
  * such that it will moves faster in result.
  * (not used in NormalWalk and ZoidalWalk)
  */
  double stepCycle = step_ / PG * 2.0 * pi;
  if (walktype == CanterWalkWT) 
  {
    hf = hfG + 5.0 * cos (stepCycle);
    hb = hbG + 5.0 * cos (stepCycle);
  } 
  else 
  {
    hf = hfG;
    hb = hbG;
  }
  
  /* theta4 = angle of robot body, relative to ground */
  theta4 = asin ((hb - hf) / lsh);
  
  /*
  * Front Right Limb
  */
  
  /* work out start and end of paw in f-s plane */
  fstart = fc + tc * sin5 + ffo;
  fend   = -fc - tc * sin5 + ffo;
  sstart = -sc - tc * cos5 + fso;
  send   = sc + tc * cos5 + fso;
  
  /* work out locus length */
  temp1 = fstart - fend;
  temp2 = sstart - send;
  dd = sqrt (temp1 * temp1 + temp2 * temp2);
  
  /* step = opposite of step_ in step cycle, use
  * step for front right, back left, and
  * step_ for front left, back right (trot gait)
  */
  step = step_ + PG;
  if (step >= 2 * PG) {
    step -= 2 * PG;
  }
  
  if (walktype == ZoidalWalkWT) {
    /* work out distance per step */
    fstep = (temp1) / PG;
    sstep = (temp2) / PG;
    if (dd == 0) {
      PGH = 0;
    } else {
      PGH = hdf / dd * PG;
    }
    
    if (PGH == 0) {
      hstep = 0;
    } else {
      hstep = hdf / PGH;
    }
    
    /* Calculate points in trapezoidal locus */
    if (step < PG / 2 - PGH) {
      f = (fstart + fend) / 2 - step * fstep;
      s = (sstart + send) / 2 - step * sstep;
      h = hf;
    }
    if (step < PG / 2 && step >= PG / 2 - PGH) {
      f = (fstart + fend) / 2 - step * fstep;
      s = (sstart + send) / 2 - step * sstep;
      h = hf - (step - PG / 2 + PGH) * hstep;
    }
    if (step < 3 * PG / 2 && step >= PG / 2) {
      f = fend + (step - PG / 2) * fstep;
      s = send + (step - PG / 2) * sstep;
      h = hf - hdf;
    }
    if (step < 3 * PG / 2 + PGH && step >= 3 * PG / 2) {
      f = fstart - (step - 3 * PG / 2) * fstep;
      s = sstart - (step - 3 * PG / 2) * sstep;
      h = hf - hdf + (step - 3 * PG / 2) * hstep;
    }
    if (step < 2 * PG && step >= 3 * PG / 2 + PGH) {
      f = fstart - (step - 3 * PG / 2) * fstep;
      s = sstart - (step - 3 * PG / 2) * sstep;
      h = hf;
    }
  } else if (walktype == NormalWalkWT || walktype == CanterWalkWT) {
    PGR = dd == 0 ? PG : (int) ((double) PG / (1 + hdf / dd));
    PGH = (PG - PGR) / 2;
    
    /* Calculate points in rectangular locus */
    if (step < PG / 2) {
      fstep = (fstart - fend) / PG;
      sstep = (sstart - send) / PG;
      f = (fstart + fend) / 2 - step * fstep;
      s = (sstart + send) / 2 - step * sstep;
      h = hf;
    }
    if (step < PG / 2 + PGH && step >= PG / 2 && PGH != 0) {
      f = fend;
      s = send;
      h = hf - (step - PG / 2) * hdf / PGH;
    }
    if (step < PG / 2 + PGH + PGR && step >= PG / 2 + PGH) {
      fstep = (fstart - fend) / PGR;
      sstep = (sstart - send) / PGR;
      f = fend + (step - PG / 2 - PGH) * fstep;
      s = send + (step - PG / 2 - PGH) * sstep;
      h = hf - hdf;
    }
    if (step < PG / 2 + 2 * PGH + PGR
      && step >= PG / 2 + PGH + PGR && PGH != 0) {
      f = fstart;
      s = sstart;
      h = hf - hdf + hdf / PGH * (step - PG / 2 - PGH - PGR);
    }
    if (step < 2 * PG && step >= PG / 2 + 2 * PGH + PGR) {
      fstep = (fstart - fend) / PG;
      sstep = (sstart - send) / PG;
      f = fstart - fstep * (step - PG / 2 - 2 * PGH - PGR);
      s = sstart - sstep * (step - PG / 2 - 2 * PGH - PGR);
      h = hf;
    }
  } else if (walktype == OffsetWalkWT) {
    double af, as, ah, bf, bs, bh, cf, cs, ch, df, ds, dh, ef, es, eh;
    double d1, d2, d3, d4, d5, dtotal;
    double t1, t2, t3, t4, t5;
    
    if (dd != 0)
    {
      double locusRotate = atan2(fstart-fend,sstart-send) - (pi/2.0);
      bf = fend + bsl1*sin(locusRotate) + bfl1*cos(locusRotate);
      bs = send + bsl1*cos(locusRotate) - bfl1*sin(locusRotate);
      cf = fend + csl1*sin(locusRotate) + cfl1*cos(locusRotate);
      cs = send + csl1*cos(locusRotate) - cfl1*sin(locusRotate);
      df = fstart + dsl1*sin(locusRotate) + dfl1*cos(locusRotate);
      ds = sstart + dsl1*cos(locusRotate) - dfl1*sin(locusRotate);
      ef = fstart + esl1*sin(locusRotate) + efl1*cos(locusRotate);
      es = sstart + esl1*cos(locusRotate) - efl1*sin(locusRotate);
      af = (ef + bf)/2.0;
      as = (es + bs)/2.0;
    }
    else
    {
      bf = cf = df = ef = af = (fstart + fend)/2.0;
      bs = cs = ds = es = as = (sstart + send)/2.0;
    }
    
    if (hdf != 0)
    {
      bh = hf + bhl1;
      ch = hf + chl1 - hdf;
      dh = hf + dhl1 - hdf;
      eh = hf + ehl1;
      ah = (eh + bh)/2.0;
    }
    else
    {
      bh = ch = dh = eh = ah = hf;
    }
    
    d1 = sqrt((af-bf)*(af-bf) + (as-bs)*(as-bs) + (ah-bh)*(ah-bh));
    d2 = sqrt((bf-cf)*(bf-cf) + (bs-cs)*(bs-cs) + (bh-ch)*(bh-ch));
    d3 = sqrt((cf-df)*(cf-df) + (cs-ds)*(cs-ds) + (ch-dh)*(ch-dh));
    d4 = sqrt((df-ef)*(df-ef) + (ds-es)*(ds-es) + (dh-eh)*(dh-eh));
    d5 = sqrt((ef-af)*(ef-af) + (es-as)*(es-as) + (eh-ah)*(eh-ah));
    dtotal = d1 + d2 + d3 + d4 + d5;
    
    if (dtotal != 0)
    {
      t1 = d1/dtotal * 2 * PG;
      t2 = d2/dtotal * 2 * PG;
      t3 = d3/dtotal * 2 * PG;
      t4 = d4/dtotal * 2 * PG;
      t5 = d5/dtotal * 2 * PG;
    }
    else
    {
      t1 = t2 = t3 = t4 = 0;
      t5 = 2 * PG;
    }
    
    if (step < t1) {
      fstep = (af - bf)/t1;
      sstep = (as - bs)/t1;
      hstep = (ah - ah)/t1;
      f     = af - step*fstep;
      s     = as - step*sstep;
      h     = ah - step*hstep;
    } else if (step < t1 + t2) {
      fstep = (bf - cf)/t2;
      sstep = (bs - cs)/t2;
      hstep = (bh - ch)/t2;
      f     = bf - (step-t1)*fstep;
      s     = bs - (step-t1)*sstep;
      h     = bh - (step-t1)*hstep;
    } else if (step < t1 + t2 + t3) {
      fstep = (cf - df)/t3;
      sstep = (cs - ds)/t3;
      hstep = (ch - dh)/t3;
      f     = cf - (step-t1-t2)*fstep;
      s     = cs - (step-t1-t2)*sstep;
      h     = ch - (step-t1-t2)*hstep;
    } else if (step < t1 + t2 + t3 + t4) {
      fstep = (df - ef)/t4;
      sstep = (ds - es)/t4;
      hstep = (dh - eh)/t4;
      f     = df - (step-t1-t2-t3)*fstep;
      s     = ds - (step-t1-t2-t3)*sstep;
      h     = dh - (step-t1-t2-t3)*hstep;
    } else {
      fstep = (ef - af)/t5;
      sstep = (es - as)/t5;
      hstep = (eh - ah)/t5;
      f     = ef - (step-t1-t2-t3-t4)*fstep;
      s     = es - (step-t1-t2-t3-t4)*sstep;
      h     = eh - (step-t1-t2-t3-t4)*hstep;
    }
  }
  
  /* f-s to x-y-z transformation  (need f, s and h) */
  z = s;
  y = h * cos (theta4) - f * sin (theta4);
  x = h * sin (theta4) + f * cos (theta4);
  
  /* x-y-z to theta1,2,3 transformation */
  temp1  = l1 * l1 + 2 * l3 * l3 + l2 * l2 - x * x - y * y - z * z;
  temp2  = 2 * sqrt ((l1 * l1 + l3 * l3) * (l2 * l2 + l3 * l3));
  theta3 = 2 * pi - atan (l1 / l3) - atan (l2 / l3) - acos (temp1 / temp2);
  tempu  = 2 * l3 * sin (theta3 / 2) * sin (theta3 / 2) + l2 * sin (theta3);
  tempv  = l1 + 2 * l3 * sin (theta3 / 2) * cos (theta3 / 2) + l2 * cos (theta3);
  theta1 = asin (z / tempv);
  temp1  = y * tempv * cos (theta1) + tempu * x;
  temp2  = tempu * tempu + tempv * tempv * cos (theta1) * cos (theta1);
  theta2 = acos (temp1 / temp2);
  
  /* bit of fudginess, since formula gives +/- for theta2 */
  if (abs(x - tempv * cos (theta1) * sin (theta2) - tempu * cos (theta2)) > .001)
    theta2 *= -1;
  
  jfr1 = theta1;
  jfr2 = theta2;
  jfr3 = theta3;
  
  /*
  * Front left Limb
  */
  /* work out start and end of paw in f-s plane */
  fstart = fc - tc * sin5 + ffo;
  fend   = -fc + tc * sin5 + ffo;
  sstart = sc + tc * cos5 + fso;
  send   = -sc - tc * cos5 + fso;
  
  /* work out locus length */
  temp1 = fstart - fend;
  temp2 = sstart - send;
  dd = sqrt (temp1 * temp1 + temp2 * temp2);
  
  if (walktype == ZoidalWalkWT) {
    fstep = (temp1) / PG;
    sstep = (temp2) / PG;
    if (dd == 0) {
      PGH = 0;
    } else {
      PGH = hdf / dd * PG;
    }
    
    if (PGH == 0) {
      hstep = 0;
    } else {
      hstep = hdf / PGH;
    }
    
    /* work out points on trapezoidal locus */
    if (step_ < PG / 2 - PGH) {
      f = (fstart + fend) / 2 - step_ * fstep;
      s = (sstart + send) / 2 - step_ * sstep;
      h = hf;
    }
    if (step_ < PG / 2 && step_ >= PG / 2 - PGH) {
      f = (fstart + fend) / 2 - step_ * fstep;
      s = (sstart + send) / 2 - step_ * sstep;
      h = hf - (step_ - PG / 2 + PGH) * hstep;
    }
    if (step_ < 3 * PG / 2 && step_ >= PG / 2) {
      f = fend + (step_ - PG / 2) * fstep;
      s = send + (step_ - PG / 2) * sstep;
      h = hf - hdf;
    }
    if (step_ < 3 * PG / 2 + PGH && step_ >= 3 * PG / 2) {
      f = fstart - (step_ - 3 * PG / 2) * fstep;
      s = sstart - (step_ - 3 * PG / 2) * sstep;
      h = hf - hdf + (step_ - 3 * PG / 2) * hstep;
    }
    if (step_ < 2 * PG && step_ >= 3 * PG / 2 + PGH) {
      f = fstart - (step_ - 3 * PG / 2) * fstep;
      s = sstart - (step_ - 3 * PG / 2) * sstep;
      h = hf;
    }
  } else if (walktype == NormalWalkWT || walktype == CanterWalkWT) {
    PGR = dd == 0 ? PG : (int) ((double) PG / (1 + hdf / dd));
    PGH = (PG - PGR) / 2;
    
    /* work out points on rectangular locus */
    if (step_ < PG / 2) {
      fstep = (fstart - fend) / PG;
      sstep = (sstart - send) / PG;
      f = (fstart + fend) / 2 - step_ * fstep;
      s = (sstart + send) / 2 - step_ * sstep;
      h = hf;
    }
    if (step_ < PG / 2 + PGH && step_ >= PG / 2 && PGH != 0) {
      f = fend;
      s = send;
      h = hf - (step_ - PG / 2) * hdf / PGH;
    }
    if (step_ < PG / 2 + PGH + PGR && step_ >= PG / 2 + PGH) {
      fstep = (fstart - fend) / PGR;
      sstep = (sstart - send) / PGR;
      f = fend + (step_ - PG / 2 - PGH) * fstep;
      s = send + (step_ - PG / 2 - PGH) * sstep;
      h = hf - hdf;
    }
    if (step_ < PG / 2 + 2 * PGH + PGR
      && step_ >= PG / 2 + PGH + PGR && PGH != 0) {
      f = fstart;
      s = sstart;
      h = hf - hdf + hdf / PGH * (step_ - PG / 2 - PGH - PGR);
    }
    if (step_ < 2 * PG && step_ >= PG / 2 + 2 * PGH + PGR) {
      fstep = (fstart - fend) / PG;
      sstep = (sstart - send) / PG;
      f = fstart - fstep * (step_ - PG / 2 - 2 * PGH - PGR);
      s = sstart - sstep * (step_ - PG / 2 - 2 * PGH - PGR);
      h = hf;
    }
  } else if (walktype == OffsetWalkWT) {
    double af, as, ah, bf, bs, bh, cf, cs, ch, df, ds, dh, ef, es, eh;
    double d1, d2, d3, d4, d5, dtotal;
    double t1, t2, t3, t4, t5;
    
    if (dd != 0)
    {
      double locusRotate = atan2(fstart-fend,sstart-send) - (pi/2.0);
      bf = fend + bsl1*sin(locusRotate) + bfl1*cos(locusRotate);
      bs = send + bsl1*cos(locusRotate) - bfl1*sin(locusRotate);
      cf = fend + csl1*sin(locusRotate) + cfl1*cos(locusRotate);
      cs = send + csl1*cos(locusRotate) - cfl1*sin(locusRotate);
      df = fstart + dsl1*sin(locusRotate) + dfl1*cos(locusRotate);
      ds = sstart + dsl1*cos(locusRotate) - dfl1*sin(locusRotate);
      ef = fstart + esl1*sin(locusRotate) + efl1*cos(locusRotate);
      es = sstart + esl1*cos(locusRotate) - efl1*sin(locusRotate);
      af = (ef + bf)/2.0;
      as = (es + bs)/2.0;
    }
    else
    {
      bf = cf = df = ef = af = (fstart + fend)/2.0;
      bs = cs = ds = es = as = (sstart + send)/2.0;
    }
    
    if (hdf != 0)
    {
      bh = hf + bhl1;
      ch = hf + chl1 - hdf;
      dh = hf + dhl1 - hdf;
      eh = hf + ehl1;
      ah = (eh + bh)/2.0;
    }
    else
    {
      bh = ch = dh = eh = ah = hf;
    }
    
    d1 = sqrt((af-bf)*(af-bf) + (as-bs)*(as-bs) + (ah-bh)*(ah-bh));
    d2 = sqrt((bf-cf)*(bf-cf) + (bs-cs)*(bs-cs) + (bh-ch)*(bh-ch));
    d3 = sqrt((cf-df)*(cf-df) + (cs-ds)*(cs-ds) + (ch-dh)*(ch-dh));
    d4 = sqrt((df-ef)*(df-ef) + (ds-es)*(ds-es) + (dh-eh)*(dh-eh));
    d5 = sqrt((ef-af)*(ef-af) + (es-as)*(es-as) + (eh-ah)*(eh-ah));
    dtotal = d1 + d2 + d3 + d4 + d5;
    
    if (dtotal != 0)
    {
      t1 = d1/dtotal * 2 * PG;
      t2 = d2/dtotal * 2 * PG;
      t3 = d3/dtotal * 2 * PG;
      t4 = d4/dtotal * 2 * PG;
      t5 = d5/dtotal * 2 * PG;
    }
    else
    {
      t1 = t2 = t3 = t4 = 0;
      t5 = 2 * PG;
    }
    
    if (step_ < t1) {
      fstep = (af - bf)/t1;
      sstep = (as - bs)/t1;
      hstep = (ah - ah)/t1;
      f     = af - step_*fstep;
      s     = as - step_*sstep;
      h     = ah - step_*hstep;
    } else if (step_ < t1 + t2) {
      fstep = (bf - cf)/t2;
      sstep = (bs - cs)/t2;
      hstep = (bh - ch)/t2;
      f     = bf - (step_-t1)*fstep;
      s     = bs - (step_-t1)*sstep;
      h     = bh - (step_-t1)*hstep;
    } else if (step_ < t1 + t2 + t3) {
      fstep = (cf - df)/t3;
      sstep = (cs - ds)/t3;
      hstep = (ch - dh)/t3;
      f     = cf - (step_-t1-t2)*fstep;
      s     = cs - (step_-t1-t2)*sstep;
      h     = ch - (step-t1-t2)*hstep;
    } else if (step_ < t1 + t2 + t3 + t4) {
      fstep = (df - ef)/t4;
      sstep = (ds - es)/t4;
      hstep = (dh - eh)/t4;
      f     = df - (step_-t1-t2-t3)*fstep;
      s     = ds - (step_-t1-t2-t3)*sstep;
      h     = dh - (step_-t1-t2-t3)*hstep;
    } else {
      fstep = (ef - af)/t5;
      sstep = (es - as)/t5;
      hstep = (eh - ah)/t5;
      f     = ef - (step_-t1-t2-t3-t4)*fstep;
      s     = es - (step_-t1-t2-t3-t4)*sstep;
      h     = eh - (step_-t1-t2-t3-t4)*hstep;
    }
  }
  
  
  /* f-s to x-y-z transformation  (need f, s and h) */
  z = s;
  y = h * cos (theta4) - f * sin (theta4);
  x = h * sin (theta4) + f * cos (theta4);
  
  /* x-y-z to theta1,2,3 transformation */
  temp1  = l1 * l1 + 2 * l3 * l3 + l2 * l2 - x * x - y * y - z * z;
  temp2  = 2 * sqrt ((l1 * l1 + l3 * l3) * (l2 * l2 + l3 * l3));
  theta3 = 2 * pi - atan (l1 / l3) - atan (l2 / l3) - acos (temp1 / temp2);
  tempu  = 2 * l3 * sin (theta3 / 2) * sin (theta3 / 2) + l2 * sin (theta3);
  tempv  = l1 + 2 * l3 * sin (theta3 / 2) * cos (theta3 / 2) + l2 * cos (theta3);
  theta1 = asin (z / tempv);
  temp1  = y * tempv * cos (theta1) + tempu * x;
  temp2  = tempu * tempu + tempv * tempv * cos (theta1) * cos (theta1);
  theta2 = acos (temp1 / temp2);
  
  if (abs(x - tempv * cos (theta1) * sin (theta2) - tempu * cos (theta2)) > .001) {
    theta2 *= -1;
  }
  
  jfl1 = theta1;
  jfl2 = theta2;
  jfl3 = theta3;
  
  /*
  * Back left Limb
  */
  
  /* work out start and end of paw in f-s plane */
  fstart = fc - tc * sin5 + bfo;
  fend   = -fc + tc * sin5 + bfo;
  sstart = sc - tc * cos5 + bso;
  send   = -sc + tc * cos5 + bso;
  
  /* work out locus length */
  temp1 = fstart - fend;
  temp2 = sstart - send;
  dd = sqrt (temp1 * temp1 + temp2 * temp2);
  
  /* step = opposite of step_ in step cycle, use
  * step for front right, back left, and
  * step_ for front left, back right (trot gait)
  */
  step = step_ + PG;
  if (step >= 2 * PG) {
    step -= 2 * PG;
  }
  
  if (walktype == ZoidalWalkWT) {
    /* work out distance per step */
    fstep = (temp1) / PG;
    sstep = (temp2) / PG;
    if (dd == 0) {
      PGH = 0;
    } else {
      PGH = hdb / dd * PG;
    }
    
    if (PGH == 0) {
      hstep = 0;
    } else {
      hstep = hdb / PGH;
    }
    
    /* work out points in trapezoidal locus */
    if (step < PG / 2 - PGH) {
      f = (fstart + fend) / 2 - step * fstep;
      s = (sstart + send) / 2 - step * sstep;
      h = hb;
    }
    if (step < PG / 2 && step >= PG / 2 - PGH) {
      f = (fstart + fend) / 2 - step * fstep;
      s = (sstart + send) / 2 - step * sstep;
      h = hb - (step - PG / 2 + PGH) * hstep;
    }
    if (step < 3 * PG / 2 && step >= PG / 2) {
      f = fend + (step - PG / 2) * fstep;
      s = send + (step - PG / 2) * sstep;
      h = hb - hdb;
    }
    if (step < 3 * PG / 2 + PGH && step >= 3 * PG / 2) {
      f = fstart - (step - 3 * PG / 2) * fstep;
      s = sstart - (step - 3 * PG / 2) * sstep;
      h = hb - hdb + (step - 3 * PG / 2) * hstep;
    }
    if (step < 2 * PG && step >= 3 * PG / 2 + PGH) {
      f = fstart - (step - 3 * PG / 2) * fstep;
      s = sstart - (step - 3 * PG / 2) * sstep;
      h = hb;
    }
  } else if (walktype == NormalWalkWT || walktype == CanterWalkWT) {
    PGR = dd == 0 ? PG : (int) ((double) PG / (1 + hdb / dd));
    PGH = (PG - PGR) / 2;
    
    /* work out points in rectangular locus */
    if (step < PG / 2) {
      fstep = (fstart - fend) / PG;
      sstep = (sstart - send) / PG;
      f = (fstart + fend) / 2 - step * fstep;
      s = (sstart + send) / 2 - step * sstep;
      h = hb;
    }
    if (step < PG / 2 + PGH && step >= PG / 2 && PGH != 0) {
      
      f = fend;
      s = send;
      h = hb - (step - PG / 2) * hdb / PGH;
    }
    if (step < PG / 2 + PGH + PGR && step >= PG / 2 + PGH) {
      
      fstep = (fstart - fend) / PGR;
      sstep = (sstart - send) / PGR;
      f = fend + (step - PG / 2 - PGH) * fstep;
      s = send + (step - PG / 2 - PGH) * sstep;
      h = hb - hdb;
    }
    if (step < PG / 2 + 2 * PGH + PGR
      && step >= PG / 2 + PGH + PGR && PGH != 0) {
      
      f = fstart;
      s = sstart;
      h = hb - hdb + hdb / PGH * (step - PG / 2 - PGH - PGR);
    }
    if (step < 2 * PG && step >= PG / 2 + 2 * PGH + PGR) {
      
      fstep = (fstart - fend) / PG;
      sstep = (sstart - send) / PG;
      f = fstart - fstep * (step - PG / 2 - 2 * PGH - PGR);
      s = sstart - sstep * (step - PG / 2 - 2 * PGH - PGR);
      h = hb;
    }
  } else if (walktype == OffsetWalkWT) {
    double af, as, ah, bf, bs, bh, cf, cs, ch, df, ds, dh, ef, es, eh;
    double d1, d2, d3, d4, d5, dtotal;
    double t1, t2, t3, t4, t5;
    
    if (dd != 0)
    {
      double locusRotate = atan2(fstart-fend,sstart-send) - (pi/2.0);
      bf = fend + bsl2*sin(locusRotate) + bfl2*cos(locusRotate);
      bs = send + bsl2*cos(locusRotate) - bfl2*sin(locusRotate);
      cf = fend + csl2*sin(locusRotate) + cfl2*cos(locusRotate);
      cs = send + csl2*cos(locusRotate) - cfl2*sin(locusRotate);
      df = fstart + dsl2*sin(locusRotate) + dfl2*cos(locusRotate);
      ds = sstart + dsl2*cos(locusRotate) - dfl2*sin(locusRotate);
      ef = fstart + esl2*sin(locusRotate) + efl2*cos(locusRotate);
      es = sstart + esl2*cos(locusRotate) - efl2*sin(locusRotate);
      af = (ef + bf)/2.0;
      as = (es + bs)/2.0;
    }
    else
    {
      bf = cf = df = ef = af = (fstart + fend)/2.0;
      bs = cs = ds = es = as = (sstart + send)/2.0;
    }
    
    if (hdb != 0)
    {
      bh = hb + bhl2;
      ch = hb + chl2 - hdb;
      dh = hb + dhl2 - hdb;
      eh = hb + ehl2;
      ah = (eh + bh)/2.0;
    }
    else
    {
      bh = ch = dh = eh = ah = hb;
    }
    
    d1 = sqrt((af-bf)*(af-bf) + (as-bs)*(as-bs) + (ah-bh)*(ah-bh));
    d2 = sqrt((bf-cf)*(bf-cf) + (bs-cs)*(bs-cs) + (bh-ch)*(bh-ch));
    d3 = sqrt((cf-df)*(cf-df) + (cs-ds)*(cs-ds) + (ch-dh)*(ch-dh));
    d4 = sqrt((df-ef)*(df-ef) + (ds-es)*(ds-es) + (dh-eh)*(dh-eh));
    d5 = sqrt((ef-af)*(ef-af) + (es-as)*(es-as) + (eh-ah)*(eh-ah));
    dtotal = d1 + d2 + d3 + d4 + d5;
    
    if (dtotal != 0)
    {
      t1 = d1/dtotal * 2 * PG;
      t2 = d2/dtotal * 2 * PG;
      t3 = d3/dtotal * 2 * PG;
      t4 = d4/dtotal * 2 * PG;
      t5 = d5/dtotal * 2 * PG;
    }
    else
    {
      t1 = t2 = t3 = t4 = 0;
      t5 = 2 * PG;
    }
    
    if (step < t1) {
      fstep = (af - bf)/t1;
      sstep = (as - bs)/t1;
      hstep = (ah - ah)/t1;
      f     = af - step*fstep;
      s     = as - step*sstep;
      h     = ah - step*hstep;
    } else if (step < t1 + t2) {
      fstep = (bf - cf)/t2;
      sstep = (bs - cs)/t2;
      hstep = (bh - ch)/t2;
      f     = bf - (step-t1)*fstep;
      s     = bs - (step-t1)*sstep;
      h     = bh - (step-t1)*hstep;
    } else if (step < t1 + t2 + t3) {
      fstep = (cf - df)/t3;
      sstep = (cs - ds)/t3;
      hstep = (ch - dh)/t3;
      f     = cf - (step-t1-t2)*fstep;
      s     = cs - (step-t1-t2)*sstep;
      h     = ch - (step-t1-t2)*hstep;
    } else if (step < t1 + t2 + t3 + t4) {
      fstep = (df - ef)/t4;
      sstep = (ds - es)/t4;
      hstep = (dh - eh)/t4;
      f     = df - (step-t1-t2-t3)*fstep;
      s     = ds - (step-t1-t2-t3)*sstep;
      h     = dh - (step-t1-t2-t3)*hstep;
    } else {
      fstep = (ef - af)/t5;
      sstep = (es - as)/t5;
      hstep = (eh - ah)/t5;
      f     = ef - (step-t1-t2-t3-t4)*fstep;
      s     = es - (step-t1-t2-t3-t4)*sstep;
      h     = eh - (step-t1-t2-t3-t4)*hstep;
    }
  }
  
  
  /* f-s to x-y-z transformation  (need f, s and h) */
  z = s;
  y = h * cos (theta4) - f * sin (theta4);
  x = h * sin (theta4) + f * cos (theta4);
  
  /* x-y-z to theta1,2,3 transformation */
  temp1  = l1 * l1 + 2 * l3 * l3 + l4 * l4 - x * x - y * y - z * z;
  temp2  = 2 * sqrt ((l1 * l1 + l3 * l3) * (l4 * l4 + l3 * l3));
  theta3 = 2 * pi - atan (l1 / l3) - atan (l4 / l3) - acos (temp1 / temp2);
  tempu  = 2 * l3 * sin (theta3 / 2) * sin (theta3 / 2) + l4 * sin (theta3);
  tempv  = l1 + 2 * l3 * sin (theta3 / 2) * cos (theta3 / 2) + l4 * cos (theta3);
  theta1 = asin (z / tempv);
  temp1  = y * tempv * cos (theta1) - tempu * x;
  temp2  = tempu * tempu + tempv * tempv * cos (theta1) * cos (theta1);
  theta2 = acos (temp1 / temp2);
  
  if (abs(-x - tempv * cos (theta1) * sin (theta2) - tempu * cos (theta2)) > .001)
    theta2 *= -1;
  
  jbl1 = theta1;
  jbl2 = theta2;
  jbl3 = theta3;
  
  /*
  * Back Right Limb
  */
  
  /* work out start and end of paw in f-s plane */
  fstart = fc + tc * sin5 + bfo;
  fend   = -fc - tc * sin5 + bfo;
  sstart = -sc + tc * cos5 + bso;
  send   = sc - tc * cos5 + bso;
  
  /* work out locus length */
  temp1 = fstart - fend;
  temp2 = sstart - send;
  dd = sqrt (temp1 * temp1 + temp2 * temp2);
  
  if (walktype == ZoidalWalkWT)
  {
    /* work out distance per step */
    fstep = (temp1) / PG;
    sstep = (temp2) / PG;
    if (dd == 0)
      PGH = 0;
    else
      PGH = hdb / dd * PG;
    
    if (PGH == 0)
      hstep = 0;
    else
      hstep = hdb / PGH;
    
    /* work out points in trapezoidal locus */
    if (step_ < PG / 2 - PGH) {
      
      f = (fstart + fend) / 2 - step_ * fstep;
      s = (sstart + send) / 2 - step_ * sstep;
      h = hb;
    }
    if (step_ < PG / 2 && step_ >= PG / 2 - PGH) {
      
      f = (fstart + fend) / 2 - step_ * fstep;
      s = (sstart + send) / 2 - step_ * sstep;
      h = hb - (step_ - PG / 2 + PGH) * hstep;
    }
    if (step_ < 3 * PG / 2 && step_ >= PG / 2) {
      
      f = fend + (step_ - PG / 2) * fstep;
      s = send + (step_ - PG / 2) * sstep;
      h = hb - hdb;
    }
    if (step_ < 3 * PG / 2 + PGH && step_ >= 3 * PG / 2) {
      
      f = fstart - (step_ - 3 * PG / 2) * fstep;
      s = sstart - (step_ - 3 * PG / 2) * sstep;
      h = hb - hdb + (step_ - 3 * PG / 2) * hstep;
    }
    if (step_ < 2 * PG && step_ >= 3 * PG / 2 + PGH) {
      
      f = fstart - (step_ - 3 * PG / 2) * fstep;
      s = sstart - (step_ - 3 * PG / 2) * sstep;
      h = hb;
    }
  }
  else if (walktype == NormalWalkWT || walktype == CanterWalkWT)
  {
    PGR = dd == 0 ? PG : (int) ((double) PG / (1 + hdb / dd));
    PGH = (PG - PGR) / 2;
    
    /* work out points in rectangular locus */
    if (step_ < PG / 2) {
      
      fstep = (fstart - fend) / PG;
      sstep = (sstart - send) / PG;
      f = (fstart + fend) / 2 - step_ * fstep;
      s = (sstart + send) / 2 - step_ * sstep;
      h = hb;
    }
    if (step_ < PG / 2 + PGH && step_ >= PG / 2 && PGH != 0) {
      
      f = fend;
      s = send;
      h = hb - (step_ - PG / 2) * hdb / PGH;
    }
    if (step_ < PG / 2 + PGH + PGR && step_ >= PG / 2 + PGH) {
      
      fstep = (fstart - fend) / PGR;
      sstep = (sstart - send) / PGR;
      f = fend + (step_ - PG / 2 - PGH) * fstep;
      s = send + (step_ - PG / 2 - PGH) * sstep;
      h = hb - hdb;
    }
    if (step_ < PG / 2 + 2 * PGH + PGR
      && step_ >= PG / 2 + PGH + PGR && PGH != 0) {
      
      f = fstart;
      s = sstart;
      h = hb - hdb + hdb / PGH * (step_ - PG / 2 - PGH - PGR);
    }
    if (step_ < 2 * PG && step_ >= PG / 2 + 2 * PGH + PGR) {
      
      fstep = (fstart - fend) / PG;
      sstep = (sstart - send) / PG;
      f = fstart - fstep * (step_ - PG / 2 - 2 * PGH - PGR);
      s = sstart - sstep * (step_ - PG / 2 - 2 * PGH - PGR);
      h = hb;
    }
  }
  else if (walktype == OffsetWalkWT)
  {
    double af, as, ah, bf, bs, bh, cf, cs, ch, df, ds, dh, ef, es, eh;
    double d1, d2, d3, d4, d5, dtotal;
    double t1, t2, t3, t4, t5;
    
    if (dd != 0)
    {
      double locusRotate = atan2(fstart-fend,sstart-send) - (pi/2.0);
      bf = fend + bsl2*sin(locusRotate) + bfl2*cos(locusRotate);
      bs = send + bsl2*cos(locusRotate) - bfl2*sin(locusRotate);
      cf = fend + csl2*sin(locusRotate) + cfl2*cos(locusRotate);
      cs = send + csl2*cos(locusRotate) - cfl2*sin(locusRotate);
      df = fstart + dsl2*sin(locusRotate) + dfl2*cos(locusRotate);
      ds = sstart + dsl2*cos(locusRotate) - dfl2*sin(locusRotate);
      ef = fstart + esl2*sin(locusRotate) + efl2*cos(locusRotate);
      es = sstart + esl2*cos(locusRotate) - efl2*sin(locusRotate);
      af = (ef + bf)/2.0;
      as = (es + bs)/2.0;
    }
    else
    {
      bf = cf = df = ef = af = (fstart + fend)/2.0;
      bs = cs = ds = es = as = (sstart + send)/2.0;
    }
    
    if (hdb != 0)
    {
      bh = hb + bhl2;
      ch = hb + chl2 - hdb;
      dh = hb + dhl2 - hdb;
      eh = hb + ehl2;
      ah = (eh + bh)/2.0;
    }
    else
    {
      bh = ch = dh = eh = ah = hb;
    }
    
    d1 = sqrt((af-bf)*(af-bf) + (as-bs)*(as-bs) + (ah-bh)*(ah-bh));
    d2 = sqrt((bf-cf)*(bf-cf) + (bs-cs)*(bs-cs) + (bh-ch)*(bh-ch));
    d3 = sqrt((cf-df)*(cf-df) + (cs-ds)*(cs-ds) + (ch-dh)*(ch-dh));
    d4 = sqrt((df-ef)*(df-ef) + (ds-es)*(ds-es) + (dh-eh)*(dh-eh));
    d5 = sqrt((ef-af)*(ef-af) + (es-as)*(es-as) + (eh-ah)*(eh-ah));
    dtotal = d1 + d2 + d3 + d4 + d5;
    
    if (dtotal != 0)
    {
      t1 = d1/dtotal * 2 * PG;
      t2 = d2/dtotal * 2 * PG;
      t3 = d3/dtotal * 2 * PG;
      t4 = d4/dtotal * 2 * PG;
      t5 = d5/dtotal * 2 * PG;
    }
    else
    {
      t1 = t2 = t3 = t4 = 0;
      t5 = 2 * PG;
    }
    
    if (step_ < t1) {
      fstep = (af - bf)/t1;
      sstep = (as - bs)/t1;
      hstep = (ah - ah)/t1;
      f     = af - step_*fstep;
      s     = as - step_*sstep;
      h     = ah - step_*hstep;
    } else if (step_ < t1 + t2) {
      fstep = (bf - cf)/t2;
      sstep = (bs - cs)/t2;
      hstep = (bh - ch)/t2;
      f     = bf - (step_-t1)*fstep;
      s     = bs - (step_-t1)*sstep;
      h     = bh - (step_-t1)*hstep;
    } else if (step_ < t1 + t2 + t3) {
      fstep = (cf - df)/t3;
      sstep = (cs - ds)/t3;
      hstep = (ch - dh)/t3;
      f     = cf - (step_-t1-t2)*fstep;
      s     = cs - (step_-t1-t2)*sstep;
      h     = ch - (step_-t1-t2)*hstep;
    } else if (step_ < t1 + t2 + t3 + t4) {
      fstep = (df - ef)/t4;
      sstep = (ds - es)/t4;
      hstep = (dh - eh)/t4;
      f     = df - (step_-t1-t2-t3)*fstep;
      s     = ds - (step_-t1-t2-t3)*sstep;
      h     = dh - (step_-t1-t2-t3)*hstep;
    } else {
      fstep = (ef - af)/t5;
      sstep = (es - as)/t5;
      hstep = (eh - ah)/t5;
      f     = ef - (step_-t1-t2-t3-t4)*fstep;
      s     = es - (step_-t1-t2-t3-t4)*sstep;
      h     = eh - (step_-t1-t2-t3-t4)*hstep;
    }
  }
  
  /* f-s to x-y-z transformation  (need f, s and h) */
  z = s;
  y = h * cos (theta4) - f * sin (theta4);
  x = h * sin (theta4) + f * cos (theta4);
  
  /* x-y-z to theta1,2,3 transformation */
  temp1  = l1 * l1 + 2 * l3 * l3 + l4 * l4 - x * x - y * y - z * z;
  temp2  = 2 * sqrt ((l1 * l1 + l3 * l3) * (l4 * l4 + l3 * l3));
  theta3 = 2 * pi - atan (l1 / l3) - atan (l4 / l3) - acos (temp1 / temp2);
  tempu  = 2 * l3 * sin (theta3 / 2) * sin (theta3 / 2) + l4 * sin (theta3);
  tempv  = l1 + 2 * l3 * sin (theta3 / 2) * cos (theta3 / 2) + l4 * cos (theta3);
  theta1 = asin (z / tempv);
  temp1  = y * tempv * cos (theta1) - tempu * x;
  temp2  = tempu * tempu + tempv * tempv * cos (theta1) * cos (theta1);
  theta2 = acos (temp1 / temp2);
  
  if (abs(-x - tempv * cos (theta1) * sin (theta2) - tempu * cos (theta2)) > .001) {
    theta2 *= -1;
  }
  
  jbr1 = theta1;
  jbr2 = theta2;
  jbr3 = theta3;
  
  /*
  * increment step_, fill motor buffers
  */
  step_++;
  
  for (int i = 3; i < numOfJoint; ++i) 
  {
    switch (i) 
    {
    case 3:  jointData.data[JointData::legFR2] = micro (jfr1); break;
    case 4:  jointData.data[JointData::legFR1] = micro (jfr2); break;
    case 5:  jointData.data[JointData::legFR3]  = micro (jfr3); break;
    case 6:  jointData.data[JointData::legFL2]  = micro (jfl1); break;
    case 7:  jointData.data[JointData::legFL1]  = micro (jfl2); break;
    case 8:  jointData.data[JointData::legFL3]  = micro (jfl3); break;
    case 9:  jointData.data[JointData::legHR2]  = micro (jbr1); break;
    case 10: jointData.data[JointData::legHR1]  = micro (jbr2); break;
    case 11: jointData.data[JointData::legHR3]  = micro (jbr3); break;
    case 12: jointData.data[JointData::legHL2]  = micro (jbl1); break;
    case 13: jointData.data[JointData::legHL1]  = micro (jbl2); break;
    case 14: jointData.data[JointData::legHL3]  = micro (jbl3); break;
    }
  } 
} 
        
        
        
        void UNSW2003WalkingEngine::readLocusOffets(const char *filename)
        {
          bfl1 = -3.64829;
          bsl1 = 4.65264;
          bhl1 = 5.44244;
          cfl1 = 9.40879;
          csl1 = -2.16859;
          chl1 = 1.80812;
          dfl1 = 0.50342;
          dsl1 = 2.12128;
          dhl1 = -0.298782;
          efl1 = 7.92933;
          esl1 = 10.6357;
          ehl1 = 4.47817;
          bfl2 = -6.30445;
          bsl2 = -4.70413;
          bhl2 = -5.22602;
          cfl2 = -2.58385;
          csl2 = 14.6642;
          chl2 = 10.4322;
          dfl2 = 13.8908;
          dsl2 = 2.5354;
          dhl2 = -4.09243;
          efl2 = -8.3251;
          esl2 = -1.80062;
          ehl2 = -1.51026;
        }
        
        
        
        long UNSW2003WalkingEngine::microRad (double radius) 
        {
          return micro (radius / 180.0 * pi);
        }
        
        double UNSW2003WalkingEngine::abs (double num) 
        {
          if (num < 0) {
            return (-num);
          } 
          else 
          {
            return num;
          }
        }
        
        double UNSW2003WalkingEngine::microRad2Deg (long rad) 
        {
          return rad * 180.0 / (1000000.0 * pi);
        }
        
        /*
        * Change log :
        * 
        * $Log: UNSW2003WalkingEngine.cpp,v $
        * Revision 1.6  2004/06/07 18:47:50  spranger
        * extended interface of executeParametrized by positionInWalkCycle
        *
        * Revision 1.5  2004/06/02 17:18:22  spranger
        * MotionRequest cleanup
        *
        * Revision 1.4  2004/05/27 17:13:37  jhoffman
        * - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
        * - clipping included for setJoints
        * - removed some microrad/rad-bugs
        * - bodyPosture constructor and "=" operator fixed
        *
        * Revision 1.3  2004/05/26 17:21:47  dueffert
        * better data types used
        *
        * Revision 1.2  2004/05/26 16:10:24  dueffert
        * better data types used
        *
        * Revision 1.1.1.1  2004/05/22 17:23:11  cvsadm
        * created new repository GT2004_WM
        *
        * Revision 1.2  2004/03/08 02:29:47  roefer
        * Interfaces should be const
        *
        * Revision 1.1  2003/10/24 15:01:25  loetzsch
        * added UNSW2003WalkingEngine
        *
        */
