/**
* @file InvKinWalkingParameters.h
* 
* Definition of class InvKinWalkingParameters
*
* @author Max Risler
*/

#ifndef __InvKinWalkingParameters_h_
#define __InvKinWalkingParameters_h_

#include "Tools/Evolution/Individual.h"
#include "Representations/Motion/JointData.h"
#include "UDParameterSet.h"
#include <string.h>

class UDParameters;

/**
* @class InvKinWalkingParameters
*
* Parameters for InvKinWalkingEngine
* 
* @author Max Risler
*/
class InvKinWalkingParameters: public Individual
{
public:
  char name[100]; ///< name of the parameter set
  
  enum FootMode {
    rectangle = 0,
      halfCircle,
      circle,
      rounded,
      optimized,
      freeFormQuad,
      numOfFootModes
  };
  
  FootMode footMode; ///< how to move the feet
  
  double foreHeight;  ///< fore walking height
  double foreWidth;   ///< fore walking width distance robot center to foot
  double foreCenterX; ///< fore x-axis foot center position
  
  double hindHeight;  ///< hind walking height
  double hindWidth;   ///< hind walking width distance robot center to foot
  double hindCenterX; ///< hind x-axis foot center position
  
  double foreFootLift;    ///< height of lifted fore foots
  double hindFootLift;    ///< height of lifted hind foots
  
  double foreFootTilt;  ///< tangens of tilt angle for fore foot movement
  double hindFootTilt;  ///< tangens of tilt angle for hind foot movement
  
  double legSpeedFactorX;  ///< speed factor by which fore legs are faster than hind legs
  double legSpeedFactorY;  ///< speed factor by which fore legs are faster than hind legs
  double legSpeedFactorR;  ///< speed factor by which fore legs are faster than hind legs
  
  ///maximum step size
  double maxStepSizeX; //mm/step
  double maxStepSizeY; //mm/step
  
  ///maximum change in requested x speed (in mm/s change per step)
  double maxSpeedXChange;
  ///maximum change in requested y speed (in mm/s change per step)
  double maxSpeedYChange;
  ///maximum change in requested rotation speed (in rad/s change per step)
  double maxRotationChange;
  
  /**
  * while walking sideways a rotational speed is added to prevent unwanted rotation
  * this factor is multiplied with y speed and added to rotation speed
  */
  double counterRotation;
  
  int stepLen;         ///< number of ticks for one complete step
  
  double groundPhase[2];   ///<part of leg phase with foot on ground, front/hind legs
  double liftPhase[2];     ///<part of leg phase lifting foot, front/hind legs
  double loweringPhase[2]; ///<part of leg phase lowering foot, front/hind legs
  double legPhase[4]; ///< relative leg phases, time indexes for lifting each leg
  
  ///value for shifting body away from lifted legs in x direction (mm when leg fully lifted)
  double bodyShiftX;
  ///value for shifting body away from lifted legs in y direction (mm when leg fully lifted)
  double bodyShiftY;
  ///value by which the shift body motion is delayed (relative part of leg phase)
  double bodyShiftOffset;
  
  /**
  * joint angles for joints not used for walking
  * these must be jointDataInvalidValue to enable head motion
  * use these for special walks that use the head eg for grabbing the ball
  */
  long headTilt, headPan, headRoll, mouth;
  
  /**
  * foot position coordinates for free form quad mode
  */
  double freeFormQuadPos[2][4][3];

  /**
  * whether engine can be left anytime or only when all four feet are on ground#
  */
  bool leaveAnytime;

  //!@name fixed gait parameters
  //!@{
  int groundTime[2];      ///< number of ticks foot is on ground
  int liftTime[2];       ///< number of ticks for lifting foot
  int loweringTime[2];       ///< number of ticks for lowering foot
  int airTime[2];         ///< number of ticks foot is in air
  int stepLift[2];       ///< time index for lifting foot
  int stepAir[2];         ///< time index when foot is fully lifted
  int stepLower[2];       ///< time index for lowering foot
  int legPhaseIndex[4];     ///< leg phases, time indexes for lifting each leg
  int firstStep;       ///< time index for first time all feet are on ground
  //!@}
  
  /** neck height in mm while walking */
  double neckHeight;
  
  /** body tilt angle */
  double bodyTilt,
         bodyTiltOffset; /**< parameter dependent body tilt offset. */
  
  /** 
  * maximal step size for rotation steps
  * this is calculated from maximal step sizes
  */
  double maxStepSizeR;
  
  /** Factors to convert requested body speed to internal leg speed 
  *  these values are read from odometry.cfg */
  class CorrectionValues {
  public:
    /** for walking forward */
    double forward;
    /** for walking backwards */
    double backward;
    /** for walking sideways */
    double sideward;
    /** for turning */
    double turning;
    /** distance from neck joint to center of rotation when turning in x-direction */
    double rotationCenter;
  } correctionValues;
  
  /** Constructor */
  InvKinWalkingParameters(
    const char name[] = "",
    int footMode = (int)rectangle,
    double foreHeight = 0,
    double foreWidth = 0,
    double foreCenterX = 0,
    double hindHeight = 0,
    double hindWidth= 0,
    double hindCenterX = 0,
    double foreFootLift = 0,
    double hindFootLift = 0,
    double foreFootTilt = 0,
    double hindFootTilt = 0,
    double legSpeedFactorX = 1.0,
    double legSpeedFactorY = 1.0,
    double legSpeedFactorR = 1.0,
    double maxStepSizeX = 0,
    double maxStepSizeY = 0,
    double maxSpeedXChange = 1000.0,
    double maxSpeedYChange = 1000.0,
    double maxRotationChange = 1000.0,
    double counterRotation = 0,
    int stepLen = 100,
    double groundPhaseF = 0.5,
    double liftPhaseF = 0,
    double loweringPhaseF = 0,
    double groundPhaseH = 0.5,
    double liftPhaseH = 0,
    double loweringPhaseH = 0,
    double legPhase0 = 0,
    double legPhase1 = 0.5,
    double legPhase2 = 0.5,
    double legPhase3 = 0,
    double bodyShiftX = 0,
    double bodyShiftY = 0,
    double bodyShiftOffset = 0,
    double bodyTiltOffset = 0,
    int leaveAnytime = 0,
    long headTilt = jointDataInvalidValue, 
    long headPan = jointDataInvalidValue, 
    long headRoll = jointDataInvalidValue, 
    long mouth = jointDataInvalidValue) :
  footMode((FootMode)footMode), 
    foreHeight(foreHeight),
    foreWidth(foreWidth),
    foreCenterX(foreCenterX),
    hindHeight(hindHeight),
    hindWidth(hindWidth),
    hindCenterX(hindCenterX),
    foreFootLift(foreFootLift),
    hindFootLift(hindFootLift),
    foreFootTilt(foreFootTilt),
    hindFootTilt(hindFootTilt),
    legSpeedFactorX(legSpeedFactorX),
    legSpeedFactorY(legSpeedFactorY),
    legSpeedFactorR(legSpeedFactorR),
    maxStepSizeX(maxStepSizeX),
    maxStepSizeY(maxStepSizeY),
    maxSpeedXChange(maxSpeedXChange),
    maxSpeedYChange(maxSpeedYChange),
    maxRotationChange(maxRotationChange),
    counterRotation(counterRotation),
    stepLen(stepLen),
    bodyShiftX(bodyShiftX),
    bodyShiftY(bodyShiftY),
    bodyShiftOffset(bodyShiftOffset),
    bodyTiltOffset(bodyTiltOffset),
    headTilt(headTilt), 
    headPan(headPan), 
    headRoll(headRoll), 
    mouth(mouth),
    leaveAnytime(leaveAnytime!=0)
  {
    strncpy(this->name,name,100);
    legPhase[0]=legPhase0;
    legPhase[1]=legPhase1;
    legPhase[2]=legPhase2;
    legPhase[3]=legPhase3;
    groundPhase[0]=groundPhaseF;
    liftPhase[0]=liftPhaseF;
    loweringPhase[0]=loweringPhaseF;
    groundPhase[1]=groundPhaseH;
    liftPhase[1]=liftPhaseH;
    loweringPhase[1]=loweringPhaseH;

    init();
    readValues();
  }
  
  /** Constructor */
  InvKinWalkingParameters(const UDParameters& udParam);
  
  virtual void getDimension(int& dim1, int& dim2)
  {
    dim1=35;
    //dim1=59;
    dim2=1;
  }
  
  virtual void getValue(int index1, int index2, double& min, double& max, double& value, ValueType& type)
  {
    switch(index1)
    {
      case 0: value=footMode; min=0; max=4; type=valueInt; break;
      //case 0: value=footMode; min=5; max=5; type=valueInt; break;
      case 1: value=foreHeight; min=30; max=170; type=valueDouble; break;
      case 2: value=foreWidth; min=40; max=120; type=valueDouble; break;
      case 3: value=foreCenterX; min=-20; max=80; type=valueDouble; break;
      case 4: value=hindHeight; min=40; max=190; type=valueDouble; break;
      case 5: value=hindWidth; min=40; max=140; type=valueDouble; break;
      case 6: value=hindCenterX; min=-80; max=20; type=valueDouble; break;
      case 7: value=foreFootLift; min=0; max=40; type=valueDouble; break;
      case 8: value=hindFootLift; min=0; max=40; type=valueDouble; break;
      case 9: value=foreFootTilt; min=-0.4; max=0.4; type=valueDouble; break;
      case 10: value=hindFootTilt; min=-0.4; max=0.44; type=valueDouble; break;
      case 11: value=legSpeedFactorX; min=0.3; max=3; type=valueDouble; break;
      case 12: value=legSpeedFactorY; min=0.3; max=3; type=valueDouble; break;
      case 13: value=legSpeedFactorR; min=0.3; max=3; type=valueDouble; break;
      case 14: value=maxStepSizeX; min=1; max=60; type=valueDouble; break;
      case 15: value=maxStepSizeY; min=1; max=60; type=valueDouble; break;
      case 16: value=maxSpeedXChange; min=10; max=400; type=valueDouble; break;
      case 17: value=maxSpeedYChange; min=10; max=400; type=valueDouble; break;
      case 18: value=maxRotationChange; min=0.1; max=2; type=valueDouble; break;
      case 19: value=counterRotation; min=-0.5; max=0.5; type=valueDouble; break;
      case 20: value=stepLen; min=10; max=200; type=valueInt; break;
      case 21: value=groundPhase[0]; min=0.01; max=0.99; type=valueDouble; break;
      case 22: value=liftPhase[0]; min=0.01; max=0.99; type=valueDouble; break;
      case 23: value=loweringPhase[0]; min=0.01; max=0.99; type=valueDouble; break;
      case 24: value=groundPhase[1]; min=0.01; max=0.99; type=valueDouble; break;
      case 25: value=liftPhase[1]; min=0.01; max=0.99; type=valueDouble; break;
      case 26: value=loweringPhase[1]; min=0.01; max=0.99; type=valueDouble; break;

      case 28: value=legPhase[0]; min=0; max=0.99; type=valueDouble; break;
      case 29: value=legPhase[1]; min=0; max=0.99; type=valueDouble; break;
      case 30: value=legPhase[2]; min=0; max=0.99; type=valueDouble; break;
      case 31: value=legPhase[3]; min=0; max=0.99; type=valueDouble; break;
      case 32: value=bodyShiftX; min=-30; max=30; type=valueDouble; break;
      case 33: value=bodyShiftY; min=-30; max=30; type=valueDouble; break;
      case 34: value=bodyShiftOffset; min=0; max=0.99; type=valueDouble; break;
/*
      case 35: value=freeFormQuadPos[0][0][0]; min=-10; max=10; type=valueDouble; break;
      case 36: value=freeFormQuadPos[0][0][1]; min=-10; max=10; type=valueDouble; break;
      case 37: value=freeFormQuadPos[0][0][2]; min=-10; max=10; type=valueDouble; break;
      case 38: value=freeFormQuadPos[0][1][0]; min=-10; max=10; type=valueDouble; break;
      case 39: value=freeFormQuadPos[0][1][1]; min=-10; max=10; type=valueDouble; break;
      case 40: value=freeFormQuadPos[0][1][2]; min=-10; max=10; type=valueDouble; break;
      case 41: value=freeFormQuadPos[0][2][0]; min=-10; max=10; type=valueDouble; break;
      case 42: value=freeFormQuadPos[0][2][1]; min=-10; max=10; type=valueDouble; break;
      case 43: value=freeFormQuadPos[0][2][2]; min=-10; max=10; type=valueDouble; break;
      case 44: value=freeFormQuadPos[0][3][0]; min=-10; max=10; type=valueDouble; break;
      case 45: value=freeFormQuadPos[0][3][1]; min=-10; max=10; type=valueDouble; break;
      case 46: value=freeFormQuadPos[0][3][2]; min=-10; max=10; type=valueDouble; break;
      case 47: value=freeFormQuadPos[1][0][0]; min=-10; max=10; type=valueDouble; break;
      case 48: value=freeFormQuadPos[1][0][1]; min=-10; max=10; type=valueDouble; break;
      case 49: value=freeFormQuadPos[1][0][2]; min=-10; max=10; type=valueDouble; break;
      case 50: value=freeFormQuadPos[1][1][0]; min=-10; max=10; type=valueDouble; break;
      case 51: value=freeFormQuadPos[1][1][1]; min=-10; max=10; type=valueDouble; break;
      case 52: value=freeFormQuadPos[1][1][2]; min=-10; max=10; type=valueDouble; break;
      case 53: value=freeFormQuadPos[1][2][0]; min=-10; max=10; type=valueDouble; break;
      case 54: value=freeFormQuadPos[1][2][1]; min=-10; max=10; type=valueDouble; break;
      case 55: value=freeFormQuadPos[1][2][2]; min=-10; max=10; type=valueDouble; break;
      case 56: value=freeFormQuadPos[1][3][0]; min=-10; max=10; type=valueDouble; break;
      case 57: value=freeFormQuadPos[1][3][1]; min=-10; max=10; type=valueDouble; break;
      case 58: value=freeFormQuadPos[1][3][2]; min=-10; max=10; type=valueDouble; break;
*/
    }
  }
  
  virtual void setValue(int index1, int index2, double value)
  {
    switch(index1)
    {
      case 0: footMode=(FootMode)((int)value); break;
      case 1: foreHeight=value; break;
      case 2: foreWidth=value; break;
      case 3: foreCenterX=value; break;
      case 4: hindHeight=value; break;
      case 5: hindWidth=value; break;
      case 6: hindCenterX=value; break;
      case 7: foreFootLift=value; break;
      case 8: hindFootLift=value; break;
      case 9: foreFootTilt=value; break;
      case 10: hindFootTilt=value; break;
      case 11: legSpeedFactorX=value; break;
      case 12: legSpeedFactorY=value; break;
      case 13: legSpeedFactorR=value; break;
      case 14: maxStepSizeX=value; break;
      case 15: maxStepSizeY=value; break;
      case 16: maxSpeedXChange=value; break;
      case 17: maxSpeedYChange=value; break;
      case 18: maxRotationChange=value; break;
      case 19: counterRotation=value; break;
      case 20: stepLen=(int)value; break;
      case 21: groundPhase[0]=value; break;
      case 22: liftPhase[0]=value; break;
      case 23: loweringPhase[0]=value; break;
      case 24: groundPhase[1]=value; break;
      case 25: liftPhase[1]=value; break;
      case 26: loweringPhase[1]=value; break;

      case 28: legPhase[0]=value; break;
      case 29: legPhase[1]=value; break;
      case 30: legPhase[2]=value; break;
      case 31: legPhase[3]=value; break;
      case 32: bodyShiftX=value; break;
      case 33: bodyShiftY=value; break;
      case 34: bodyShiftOffset=value; break;
/*
      case 35: freeFormQuadPos[0][0][0]=value; break;
      case 36: freeFormQuadPos[0][0][1]=value; break;
      case 37: freeFormQuadPos[0][0][2]=value; break;
      case 38: freeFormQuadPos[0][1][0]=value; break;
      case 39: freeFormQuadPos[0][1][1]=value; break;
      case 40: freeFormQuadPos[0][1][2]=value; break;
      case 41: freeFormQuadPos[0][2][0]=value; break;
      case 42: freeFormQuadPos[0][2][1]=value; break;
      case 43: freeFormQuadPos[0][2][2]=value; break;
      case 44: freeFormQuadPos[0][3][0]=value; break;
      case 45: freeFormQuadPos[0][3][1]=value; break;
      case 46: freeFormQuadPos[0][3][2]=value; break;
      case 47: freeFormQuadPos[1][0][0]=value; break;
      case 48: freeFormQuadPos[1][0][1]=value; break;
      case 49: freeFormQuadPos[1][0][2]=value; break;
      case 50: freeFormQuadPos[1][1][0]=value; break;
      case 51: freeFormQuadPos[1][1][1]=value; break;
      case 52: freeFormQuadPos[1][1][2]=value; break;
      case 53: freeFormQuadPos[1][2][0]=value; break;
      case 54: freeFormQuadPos[1][2][1]=value; break;
      case 55: freeFormQuadPos[1][2][2]=value; break;
      case 56: freeFormQuadPos[1][3][0]=value; break;
      case 57: freeFormQuadPos[1][3][1]=value; break;
      case 58: freeFormQuadPos[1][3][2]=value; break;
*/
    }
  }


#define interpolateInvalid(v1,v2,factor1) \
  (((v1==jointDataInvalidValue)||(v2==jointDataInvalidValue))? \
  ((factor1<0.5)?v2:v1):(factor1*v1+(1-factor1)*v2));
  
  inline void interpolate(InvKinWalkingParameters& p1, InvKinWalkingParameters& p2, double factor1)
  {
    footMode=(factor1<0.5)?p2.footMode:p1.footMode;
    
    foreHeight=factor1*p1.foreHeight+(1-factor1)*p2.foreHeight;
    foreWidth=factor1*p1.foreWidth+(1-factor1)*p2.foreWidth;
    foreCenterX=factor1*p1.foreCenterX+(1-factor1)*p2.foreCenterX;
    
    hindHeight=factor1*p1.hindHeight+(1-factor1)*p2.hindHeight;
    hindWidth=factor1*p1.hindWidth+(1-factor1)*p2.hindWidth;
    hindCenterX=factor1*p1.hindCenterX+(1-factor1)*p2.hindCenterX;
    
    foreFootLift=factor1*p1.foreFootLift+(1-factor1)*p2.foreFootLift;
    hindFootLift=factor1*p1.hindFootLift+(1-factor1)*p2.hindFootLift;
    foreFootTilt=factor1*p1.foreFootTilt+(1-factor1)*p2.foreFootTilt;
    hindFootTilt=factor1*p1.hindFootTilt+(1-factor1)*p2.hindFootTilt;
    
    legSpeedFactorX=factor1*p1.legSpeedFactorX+(1-factor1)*p2.legSpeedFactorX;
    legSpeedFactorY=factor1*p1.legSpeedFactorY+(1-factor1)*p2.legSpeedFactorY;
    legSpeedFactorR=factor1*p1.legSpeedFactorR+(1-factor1)*p2.legSpeedFactorR;
    
    maxStepSizeX=factor1*p1.maxStepSizeX+(1-factor1)*p2.maxStepSizeX;
    maxStepSizeY=factor1*p1.maxStepSizeY+(1-factor1)*p2.maxStepSizeY;
    maxSpeedXChange=factor1*p1.maxSpeedXChange+(1-factor1)*p2.maxSpeedXChange;
    maxSpeedYChange=factor1*p1.maxSpeedYChange+(1-factor1)*p2.maxSpeedYChange;
    maxRotationChange=factor1*p1.maxRotationChange+(1-factor1)*p2.maxRotationChange;
    
    counterRotation=factor1*p1.counterRotation+(1-factor1)*p2.counterRotation;
    stepLen=(int)(factor1*p1.stepLen+(1-factor1)*p2.stepLen);
    groundPhase[0]=factor1*p1.groundPhase[0]+(1-factor1)*p2.groundPhase[0];
    liftPhase[0]=factor1*p1.liftPhase[0]+(1-factor1)*p2.liftPhase[0];
    loweringPhase[0]=factor1*p1.loweringPhase[0]+(1-factor1)*p2.loweringPhase[0];
    groundPhase[1]=factor1*p1.groundPhase[1]+(1-factor1)*p2.groundPhase[1];
    liftPhase[1]=factor1*p1.liftPhase[1]+(1-factor1)*p2.liftPhase[1];
    loweringPhase[1]=factor1*p1.loweringPhase[1]+(1-factor1)*p2.loweringPhase[1];
    
    bodyShiftX=factor1*p1.bodyShiftX+(1-factor1)*p2.bodyShiftX;
    bodyShiftY=factor1*p1.bodyShiftY+(1-factor1)*p2.bodyShiftY;
    bodyShiftOffset=factor1*p1.bodyShiftOffset+(1-factor1)*p2.bodyShiftOffset;
    bodyTiltOffset=factor1*p1.bodyTiltOffset+(1-factor1)*p2.bodyTiltOffset;
    
    headTilt=(long)interpolateInvalid(p1.headTilt,p2.headTilt,factor1);
    headPan=(long)interpolateInvalid(p1.headPan,p2.headPan,factor1);
    headRoll=(long)interpolateInvalid(p1.headRoll,p2.headRoll,factor1);
    mouth=(long)interpolateInvalid(p1.mouth,p2.mouth,factor1);
    
    legPhase[0]=factor1*p1.legPhase[0]+(1-factor1)*p2.legPhase[0];
    legPhase[1]=factor1*p1.legPhase[1]+(1-factor1)*p2.legPhase[1];
    legPhase[2]=factor1*p1.legPhase[2]+(1-factor1)*p2.legPhase[2];
    legPhase[3]=factor1*p1.legPhase[3]+(1-factor1)*p2.legPhase[3];
    
    correctionValues.forward=factor1*p1.correctionValues.forward+(1-factor1)*p2.correctionValues.forward;
    correctionValues.sideward=factor1*p1.correctionValues.sideward+(1-factor1)*p2.correctionValues.sideward;
    correctionValues.turning=factor1*p1.correctionValues.turning+(1-factor1)*p2.correctionValues.turning;
    correctionValues.rotationCenter=factor1*p1.correctionValues.rotationCenter+(1-factor1)*p2.correctionValues.rotationCenter;
    neckHeight=factor1*p1.neckHeight+(1-factor1)*p2.neckHeight;

    for (int i=0; i<2; i++)
      for (int j=0; j<4; j++)
        for (int k=0; k<3; k++)
          freeFormQuadPos[i][j][k]=(factor1<0.5)?p2.freeFormQuadPos[i][j][k]:p1.freeFormQuadPos[i][j][k];

    if (factor1 == 1.0)
      leaveAnytime = p1.leaveAnytime;
    else if (factor1 == 0.0)
      leaveAnytime = p2.leaveAnytime;
    else
      leaveAnytime = p2.leaveAnytime || p1.leaveAnytime;
  }
  
  /** copies the walking parameters without correctionValues */
  void copyValuesFrom(const InvKinWalkingParameters & paramsToCopy);
    
  /** initialise fixed parameters */
  void init();
  
  /** reads the correction values, neck height from odometry.cfg */
  void readValues();
  
  /** returns true when all feet are on ground */
  bool allFeetOnGround(int currentStep);
};

/**
* Streaming operator that reads InvKinWalkingParameters from a stream.
* @param stream The stream from which is read.
* @param walkingParameters The InvKinWalkingParameters object.
* @return The stream.
*/ 
In& operator>>(In& stream,InvKinWalkingParameters& walkingParameters);

/**
* Streaming operator that writes InvKinWalkingParameters to a stream.
* @param stream The stream to write on.
* @param walkingParameters The InvKinWalkingParameters object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const InvKinWalkingParameters& walkingParameters);

#endif// __InvKinWalkingParameters_h_

/*
* Change log :
* 
* $Log: InvKinWalkingParameters.h,v $
* Revision 1.2  2004/07/07 15:10:57  dueffert
* copy'n'paste invkin added for ud
*
* Revision 1.1.1.1  2004/05/22 17:22:47  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2004/03/09 22:32:49  cesarz
* added function for assigning certain parameters
*
* Revision 1.1  2004/02/16 17:52:48  dueffert
* InvKin engine and parameters separated
*
*
*/
