/**
 * @file SensorFusionTeamBallLocator.cpp
 * 
 * This file contains the sensor fusion class for ball-localization.
 *
 * @author <A href=mailto:kai_engel@gmx.de>Kai Engel</A>
 * @author <A href=mailto:m_wachter@gmx.de>Michael Wachter</A>
 */

#include "SensorFusionTeamBallLocator.h"
#include <math.h>
#include "Tools/Math/Vector2.h"
#include "Tools/Math/Geometry.h"
#include "Platform/SystemCall.h" 
#include <stdio.h>

// This table is needed to calculate the values for sigmaMaj and sigmaMin (variances of the 
// Gaussian Distribution) from the distance from the robot to the ball. The values are
// taken from experiments in Distances from 0 to 400 cm.
const double factorToMakeSigmas = 0.7;
const Vector2<double> sigmaTable[20] = {
                                          Vector2<double>( 75.0,  25.0) * factorToMakeSigmas,
                                          Vector2<double>(100.0,  25.0) * factorToMakeSigmas,
                                          Vector2<double>(100.0,  75.0) * factorToMakeSigmas,
                                          Vector2<double>(150.0, 100.0) * factorToMakeSigmas,
                                          Vector2<double>(220.0, 100.0) * factorToMakeSigmas,
                                          Vector2<double>(230.0, 100.0) * factorToMakeSigmas,
                                          Vector2<double>(300.0, 125.0) * factorToMakeSigmas,
                                          Vector2<double>(500.0, 150.0) * factorToMakeSigmas,
                                          Vector2<double>(500.0, 200.0) * factorToMakeSigmas,
                                          Vector2<double>(550.0, 200.0) * factorToMakeSigmas, //10th entry

                                          Vector2<double>(550.0, 200.0) * factorToMakeSigmas,
                                          Vector2<double>(600.0, 200.0) * factorToMakeSigmas,
                                          Vector2<double>(650.0, 200.0) * factorToMakeSigmas,                                          
                                          Vector2<double>(700.0, 200.0) * factorToMakeSigmas,
                                          Vector2<double>(750.0, 200.0) * factorToMakeSigmas,
                                          Vector2<double>(800.0, 200.0) * factorToMakeSigmas,
                                          Vector2<double>(900.0, 200.0) * factorToMakeSigmas,
                                          Vector2<double>(1000.0, 300.0) * factorToMakeSigmas,
                                          Vector2<double>(1200.0, 300.0) * factorToMakeSigmas,
                                          Vector2<double>(1400.0, 300.0) * factorToMakeSigmas //20th entry
}; 

SensorFusionTeamBallLocator::SensorFusionTeamBallLocator(const TeamBallLocatorInterfaces& interfaces)
: TeamBallLocator(interfaces)
{ 

}


void SensorFusionTeamBallLocator::execute()
{
  
  SeenBallPosition seenPosition[NUMBER_OF_PLAYERS];
  double seenAngle[NUMBER_OF_PLAYERS];
  GaussBell gaussBell[NUMBER_OF_PLAYERS];
  RobotPose robotPoses[NUMBER_OF_PLAYERS];  

  int robot;
  int numberOfRobotsSeenBall = 0;
  
  // Init for local Robot
  seenPosition[0]  = seenBallPosition;
  robotPoses[0] = robotPose;
  seenAngle[0] = Geometry::angleTo(robotPose,seenPosition[0]) + robotPose.getAngle();

  // Init for remote Robots
  for (robot=1; robot < 4; robot++) {
    seenPosition[robot] = teamMessageCollection[robot-1].seenBallPosition;
    seenPosition[robot].timeWhenLastSeen = 
        teamMessageCollection[robot-1].getTimeInOwnTime(seenPosition[robot].timeWhenLastSeen);
    robotPoses[robot] = teamMessageCollection[robot-1].robotPose;
    seenAngle[robot] =  
        Geometry::angleTo( teamMessageCollection[robot-1].robotPose,seenPosition[robot]) + 
        teamMessageCollection[robot-1].robotPose.getAngle();
  }

  // All input-data is now in seenPosition, seenDirection and robotPoses. 
  // Now we throw away the ones which have not seen the ball for 2 seconds.
  double maxAge = 1000; //used to be 2000
  double timeUncertainty = 2;
  for (robot=0; robot < 4; robot++)
  {
    if ( SystemCall::getTimeSince(seenPosition[robot].timeWhenLastSeen) < maxAge)//used to be 2000
    {
       // Position of the center of the gaussBell
       gaussBell[numberOfRobotsSeenBall].setPositionOfMaximum(seenPosition[robot]);
       gaussBell[numberOfRobotsSeenBall].setRotationAngle(seenAngle[robot]);

       // Variances are calculated from the validity of the self-localisation of the 
       // robot and the distance to the ball
       double timeFactor = 1 + timeUncertainty*SystemCall::getTimeSince(seenPosition[robot].timeWhenLastSeen)/maxAge;
       double sigmaFactor =  1 / ( robotPoses[robot].getValidity() + 0.0001 );
       int distanceIndex = (int)Geometry::distance(seenPosition[robot],robotPoses[robot].translation) / 200;
       if (distanceIndex > 19) distanceIndex = 19;

       gaussBell[numberOfRobotsSeenBall].setSigmas(
                    sigmaTable[distanceIndex].x * sigmaFactor * timeFactor,
                    sigmaTable[distanceIndex].y * sigmaFactor * timeFactor);

       // Only use position of seeing Robots for the Debug-Drawing
       robotPoses[numberOfRobotsSeenBall] = robotPoses[robot];
       seenPosition[numberOfRobotsSeenBall] = seenPosition[robot];
       numberOfRobotsSeenBall++;
    }
  }
  // Now the Gaussbells are ready to be mergerd
  
  GaussBell result;
  if (numberOfRobotsSeenBall)
  { 
     // At least one Robot is seeing the ball    
     result = gaussBell[0];
     for (robot = 1; robot < numberOfRobotsSeenBall; robot++)
     {
         result.mergeBells(result,gaussBell[robot]);
     }
     // Add result to history-buffer
     result.setTimeStamp(SystemCall::getCurrentSystemTime());
     history.add(result);
     communicatedBallPosition.timeWhenLastObserved = SystemCall::getCurrentSystemTime();
  }
  else 
  {
     // init result with the first entry of the history-rinbuffer
     result = history[0];
  }
  
  // Merge History-entrys to result. 
  int historyEntry ;
  for (historyEntry = 1; historyEntry < history.getNumberOfEntries(); historyEntry++)
  {
    double tempSigmaMin, tempSigmaMaj;
    history[historyEntry].getSigmas(tempSigmaMaj,tempSigmaMin);
    // Update the sigmas of each history-table. They double by at about 5 runs of
    // the SensorFusionTeamBallLocator
    // The Sigmas must not become too big because
    // the robot crashes when they are bigger than sqrt( DOUBLE_MAX )
    if (tempSigmaMaj < 4000)
    {
      tempSigmaMaj *= 1.3;// used to be 1.1
      tempSigmaMin *= 1.3;// used to be 1.1
    }

    history[historyEntry].setSigmas(tempSigmaMaj,tempSigmaMin);
    // Merge history-entry
    result.mergeBells(result,history[historyEntry]);
  }

  if (communicatedBallPosition.timeWhenLastObserved < history[0].getTimeStamp() ) 
       communicatedBallPosition.timeWhenLastObserved = history[0].getTimeStamp();

  // Transfer result to communicatedBallPosition
  communicatedBallPosition.x = result.getPositionOfMaximum().x;
  communicatedBallPosition.y = result.getPositionOfMaximum().y;

  // This is needed by the Ruhrpott-Hellhounds-collective behavior. 
  // May be removed in the Future.
  result.getSigmas(communicatedBallPosition.distanceInMajDirection,communicatedBallPosition.distanceInMinDirection);
  communicatedBallPosition.orientationAngleOfGaussBell = result.getAngle();

  // Debug-Drawings

  COMPLEX_DRAWING(teamBallLocatorField,
    for (robot = 0; robot < numberOfRobotsSeenBall; robot++ )
    {

       // Line from the Robot to the position it sees the ball
       LINE(teamBallLocatorField,
              robotPoses[robot].translation.x , robotPoses[robot].translation.y ,
              seenPosition[robot].x , seenPosition[robot].y ,
              1, Drawings::ps_solid, Drawings::blue
            );

       // Dawing of the GaussBell as a cross with the width of simaMaj and height 
       // of SigmaMay and SigmaMin rotated by the rotationAngle
       Vector2<double> point[4];
       Matrix2x2<double> rotation;
       double sigmaMaj;
       double sigmaMin;
       gaussBell[robot].getSigmas(sigmaMaj, sigmaMin);
       point[0] = Vector2<double>(sigmaMaj,0);
       point[1] = Vector2<double>(-sigmaMaj,0);
       point[2] = Vector2<double>(0,sigmaMin);
       point[3] = Vector2<double>(0,-sigmaMin);
       rotation.c[0].x = cos(gaussBell[robot].getAngle());
       rotation.c[0].y = sin(gaussBell[robot].getAngle());
       rotation.c[1].x = - rotation.c[0].y;
       rotation.c[1].y = rotation.c[0].x;
       for (int i=0; i<4; i++) 
       {
          point[i] = rotation * point[i];
          point[i] = point[i] + gaussBell[robot].getPositionOfMaximum();
       }
       LINE(teamBallLocatorField, 
            point[0].x , point[0].y , point[1].x, point[1].y,  
            1, Drawings::ps_solid, Drawings::skyblue
            );
       LINE(teamBallLocatorField, 
            point[2].x , point[2].y , point[3].x, point[3].y,
            1, Drawings::ps_solid, Drawings::skyblue
            );
    }

    // Drawing of the History-Entrys as small yellow circles
    for (historyEntry = 0; historyEntry < history.getNumberOfEntries(); historyEntry++)
    {
       CIRCLE(teamBallLocatorField,
              history[historyEntry].getPositionOfMaximum().x ,
              history[historyEntry].getPositionOfMaximum().y ,
              50,1, Drawings::ps_solid, Drawings::yellow
              );
    }
  );

  DEBUG_DRAWING_FINISHED(teamBallLocatorField); 
}


/*
 * Change log :
 * 
 * $Log: SensorFusionTeamBallLocator.cpp,v $
 * Revision 1.1.1.1  2004/05/22 17:22:26  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.4  2004/05/06 12:33:33  nistico
 * Improved: percepts are filtered out if older than 1 second, variance is dependent on age, variance scale extended, history entries' variance increases much faster
 *
 * Revision 1.5  2004/05/01 22:26:27  nistico
 * SensorFusionTeamBallLocator "finalized"
 *
 * Revision 1.4  2004/05/01 21:12:14  nistico
 * New colortable which works also on bad robot
 * SensorFusionTeamBallLocator now works sensible! :)
 *
 * Revision 1.3  2004/03/08 02:11:53  roefer
 * Interfaces should be const
 *
 * Revision 1.2  2004/01/24 18:35:03  wachter
 * Reactivated debug-drawings in SensorFusionTeamBallLocator
 *
 * Revision 1.1  2004/01/23 23:02:25  wachter
 * Reinserted the famous GaussBellTeamBallLocator from the
 * as GT2003-code as SensorFusionTeamBallLocator.
 *
 * Revision 1.8  2003/05/27 15:44:17  wachter
 * Bugfix if ball is not seen for some time
 *
 * Revision 1.7  2003/05/19 15:34:42  wachter
 * Added calculation of timeWhenLastObserved
 * Moved genration of debug-drawing
 *
 * Revision 1.6  2003/05/18 19:31:26  wachter
 * Rewrote GaussBellTeamBallLocator
 * Renamed SensorFusionGaussBell to GaussBell as requested.
 *
 * Revision 1.5  2003/05/13 15:56:29  engel
 * cosmetics
 *
 * Revision 1.4  2003/05/11 14:39:22  roefer
 * Warnings removed
 *
 * Revision 1.3  2003/05/09 16:23:12  engel
 * variables updated... (for cooperation with the other GermanTeam-members)
 *
 * Revision 1.2  2003/05/01 19:31:28  roefer
 * Warnings removed
 *
 * Revision 1.1  2003/05/01 17:09:08  loetzsch
 * Redesign of ball modeling:
 * - Modularized class BallPosition
 * - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
 *   and "TeamBallLocator" for modelling communicated positions
 * - Removed solution JumpingBallLocator
 * - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
 * - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
 *
 * Revision 1.55  2003/04/25 11:32:40  engel
 * Comments inserted
 *
 * Revision 1.54  2003/04/24 16:27:32  engel
 * Debug-Drawings changed: Now after 1000 Locations all these will be drawn.
 * => We will be compare the Default-Locator with the SFBL
 * Will be continued...
 *
 * Revision 1.53  2003/04/24 11:10:15  engel
 * Some different improvements.
 *
 * Revision 1.52  2003/04/23 20:53:21  roefer
 * Warnings removed
 *
 * Revision 1.51  2003/04/22 15:03:39  engel
 * inserted some for-loops to make the code shorter
 *
 * Revision 1.50  2003/04/19 03:33:27  pruente
 * Merged in changes by Kai: - History changed: Now the last NUMBER_OF_TAKEN_HISTIORY_ENTRYS
 *                             many entrys will be merged. ;-)
 *
 *                           - Set ballPosition.knownPosition = ballPosition.seenPosition, if the ball is nearer
 *                             the robot than 50cm.
 *
 *                           - Set ballPosition.validity to 0.001, if no robot has seen the ball.
 *
 *                           - if the ball was seen in one of the penalty areas it will not be merged.
 *                             Cause: If the robots shows at the yellow goal, he will percept a ball,
 *                             even if there is no ball (because of red goalie in front of yellow goal).
 *
 * Revision 1.53  2003/04/12 17:05:23  Engel
 * History changed: Now the last NUMBER_OF_TAKEN_HISTIORY_ENTRYS
 * many entrys will be merged. ;-)
 *
 * Revision 1.52  2003/04/12 09:13:51  Engel
 * Set ballPosition.knownPosition = ballPosition.seenPosition, if the ball is nearer
 * the robot than 50cm.
 *
 * Revision 1.51  2003/04/11 12:32:26  Engel
 * Set ballPosition.validity to 0.001, if no robot has seen the ball.
 *
 * Revision 1.50  2003/04/11 10:20:55  Engel
 * If the ball was seen in one of the penalty areas it will not be merged.
 * Cause: If the robots shows at the yellow goal, he will percept a ball,
 * even if there is no ball (because of red goalie in front of yellow goal).
 *
 * Revision 1.49  2003/04/09 17:42:33  engel
 * comment included (no relevant changes....)
 *
 * Revision 1.48  2003/04/09 17:38:47  engel
 * One Line outcommentet.
 *
 * Revision 1.47  2003/04/09 16:56:15  engel
 * DefaultBallLocator for seen-Position in SFBL included (but only for seen-position).
 *
 * Revision 1.46  2003/04/09 13:19:17  engel
 * ballPosition.timeWhenLastKnown actualized
 *
 * Revision 1.45  2003/04/08 13:18:03  schmidt
 * Commented out some buggy lines
 *
 * Revision 1.44  2003/04/08 10:28:36  engel
 * Some improvements: Now the ball will be matched to the field in every case.
 *
 * Revision 1.43  2003/04/08 08:52:24  schmidt
 * Fixing
 *
 * Revision 1.42  2003/04/07 14:15:16  wachter
 * workaround for wrong history if no ball was seen.
 *
 * Revision 1.41  2003/04/06 16:26:29  engel
 * Cosmetics
 *
 * Revision 1.40  2003/04/06 15:57:52  engel
 * Little changes (cosmetic and some improvements).
 *
 * Revision 1.39  2003/04/06 11:43:42  engel
 * Some if-cases deleted (if ... numberOfTeamMessages())
 *
 * Revision 1.38  2003/04/04 15:06:45  engel
 * If numberOfTeamMessages is to low some values wont be set (now).
 *
 * Revision 1.37  2003/04/04 10:21:25  engel
 * Catch invalid values of doubles (doubleIsValid())
 *
 * Revision 1.36  2003/04/03 17:00:20  dickhoefer
 * changed calcutation because of problems with floating point addition (Kai Engel)
 *
 * Revision 1.35  2003/04/02 14:51:53  schumann
 * fixed possible division by zero
 *
 * Revision 1.34  2003/04/01 15:17:44  engel
 * 1) The merged ball is now allways ON the field!
 * 2) Deleted some methods
 * 3) Cosmetics
 *
 * Revision 1.33  2003/04/01 10:21:31  dueffert
 * unused warning removed
 *
 * Revision 1.32  2003/04/01 08:05:28  engel
 * determineRobotsSeenBall(): If one robot has not seen the ball for "a long time" its percept wont be merged.
 *
 * Revision 1.31  2003/03/27 16:17:24  engel
 * Invalid BallPercepepts will be destroyed or repaired...
 *
 * Revision 1.30  2003/03/26 15:36:00  engel
 * Body for calculateValidityOfBall(...) inserted.
 * Changes for test-text-files.
 * Calculation of sigmas from Validity.
 *
 * Revision 1.29  2003/03/22 18:11:23  roefer
 * Warnings removed
 *
 * Revision 1.28  2003/03/21 17:03:25  wachter
 * improved Debug-Drawings
 *
 * Revision 1.27  2003/03/21 15:11:52  engel
 * fixed crash for dx=dy=0
 * inserted getValitityAtPositionOfMaximum()
 *
 * Revision 1.26  2003/03/20 18:28:17  engel
 * Bug in Hisrtory fixed. The Value (0,0) is NOT an angle.
 *
 * Revision 1.25  2003/03/20 16:01:39  engel
 * This BallLocator is runing on the robot now (inc. History)
 *
 * Revision 1.24  2003/03/18 13:40:53  wachter
 * Bugfixes
 *
 * Revision 1.23  2003/03/14 16:04:46  wachter
 * Bugfixes
 *
 * Revision 1.22  2003/03/12 11:04:37  engel
 * Begin: If one roboter does not see the ball, its percept wont be taken...
 *
 * Revision 1.21  2003/03/11 16:27:54  engel
 * Changes for Debug...
 * Output: Yellow circles for History...
 *
 * Revision 1.20  2003/03/11 11:34:30  engel
 * Debug-changes
 *
 * Revision 1.19  2003/03/09 15:39:42  wachter
 * Added debug-drawing for local robot
 *
 * Revision 1.18  2003/03/07 15:35:59  wachter
 * Added debug-drawings for SensorFusionBall and PlayerLocator.
 *
 * Revision 1.17  2003/03/06 16:22:12  wachter
 * Divsion by zero fixed.
 *
 * Revision 1.16  2003/03/06 11:51:21  dueffert
 * unused variable and subscript out of range warning removed
 *
 * Revision 1.15  2003/03/06 11:33:05  engel
 * Debug-Changes
 *
 * Revision 1.14  2003/03/05 16:20:58  engel
 * Initialization changed.
 * If one ball was seen with validity 0 it will not be merged!
 * Still working on...
 *
 * Revision 1.13  2003/03/04 16:44:07  engel
 * still working on (testing)
 *
 * Revision 1.12  2003/03/04 11:55:04  engel
 * still working on...
 *
 * Revision 1.11  2003/02/28 16:29:26  engel
 * still working on ;-))
 *
 * Revision 1.10  2003/02/28 14:29:09  engel
 * still working on ;-)
 *
 * Revision 1.9  2003/02/28 12:08:35  engel
 * Still working on ;-)
 *
 * Revision 1.8  2003/02/27 13:30:40  engel
 * execute-Method extended
 *
 * Revision 1.7  2003/02/26 16:11:55  engel
 * Added calculation of validity to method "execute".
 *
 * Revision 1.6  2003/02/25 16:49:49  engel
 * some changes to calculate the validity (mean-Value)
 *
 * Revision 1.5  2003/02/21 16:34:16  dueffert
 * common pi in own code, warnings removed, platform compatibility restored
 *
 * Revision 1.4  2003/02/21 13:19:25  engel
 * some changes
 *
 * Revision 1.3  2003/02/21 12:19:32  engel
 * add the method erf(double _x)
 *
 * Revision 1.2  2003/02/19 16:21:48  engel
 * little changes for debugging the GaussBellTeamBallLocator (beginn to implement erf())
 *
 * Revision 1.1  2003/02/14 14:34:02  wachter
 * Added GaussBellTeamBallLocator
 *
 *
 */
