/**
* @file SpecialActions.h
* 
* This file contains a generic class for SpecialActions.
*/

#ifndef __SpecialActions_h_
#define __SpecialActions_h_

#include "Tools/Module/Module.h"

#include "Representations/Motion/JointData.h"
#include "Representations/Motion/PIDData.h"
#include "Representations/Motion/OdometryData.h"
#include "Representations/Motion/MotionInfo.h"

/** 
* @class SpecialActionsInterfaces
*
* Interfaces of the SpecialAction module.
*/
class SpecialActionsInterfaces
{
public:
  /** Constructor */
  SpecialActionsInterfaces(
    const MotionRequest& motionRequest,
    MotionRequest::MotionID& lastMotionType,
    PIDData& pidData,
    OdometryData& odometryData,
    MotionInfo& motionInfo)
    : motionRequest(motionRequest),
    lastMotionType(lastMotionType),
    lastWalkType(lastWalkType),
    pidData(pidData),
    odometryData(odometryData),
    motionInfo(motionInfo)
  {}
  
protected:
  /** The current motion request */
  const MotionRequest& motionRequest;
  
  /** The motion type of the previous frame */
  MotionRequest::MotionID& lastMotionType;
  
  /** The walk type of the previous frame */
  WalkRequest::WalkType& lastWalkType;

  /** The current PID servo gains */
  PIDData& pidData;
    
  /** Has to be incremented by the walking engine */
  OdometryData& odometryData;
  
  /** The neck height and the body tilt to be estimated by the walking engine */
  MotionInfo& motionInfo;
};

/**
* A generic class for SpecialActions.
*
* It generates JointData and OdometryData according to current MotionRequest
* executing special actions.
*/
class SpecialActions : public Module, public SpecialActionsInterfaces
{
public:
/*
* Constructor.
* @param interfaces The paramters of the SpecialActions module.
  */
  SpecialActions(const SpecialActionsInterfaces& interfaces)
    : SpecialActionsInterfaces(interfaces)
  {}
  
  /**
  * Calculates the next joint data set
  * @param jointData stores calculated frame
  * @return true if next set should be calculated by SpecialActions
  *         false if change to other module can occur
  */
  virtual bool executeParameterized(SpecialActionRequest specialActionRequest, JointData& jointData) = 0;

  /*
   *  Determines wether or not a Specialaction can be executed in current phase of Walking cycle 
   *	@param specialActionID SpecialActionID for Specialaction that information for is needed
   *  @return wether the Specialaction can be executed in current Walkingcycle
   */
  virtual bool specialActionIsExecutableInWalkingCycle(int specialActionID, double walkCycle)=0;
  
  /** Destructor */
  virtual ~SpecialActions() {}
  
private:
  /** That function is not used but has to be implemented */
  virtual void execute() {};
};

#endif //__SpecialActions_h_

/*
* Change log :
* 
* $Log: SpecialActions.h,v $
* Revision 1.5  2004/06/14 14:55:03  spranger
* changed interface of executeParametrized to specialActionRequest
*
* Revision 1.4  2004/06/02 17:18:24  spranger
* MotionRequest cleanup
*
* Revision 1.3  2004/05/27 09:29:28  loetzsch
* removed executedMotionRequest from Interfaces
*
* Revision 1.2  2004/05/26 16:10:24  dueffert
* better data types used
*
* Revision 1.1.1.1  2004/05/22 17:21:31  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2004/03/08 02:11:52  roefer
* Interfaces should be const
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.5  2003/03/06 12:09:00  dueffert
* execute with parameters renamed to avoid inheritance warnings, re-order warning removed
*
* Revision 1.4  2002/11/20 18:02:28  risler
* added PID values to GT2001MotionNetSpecialActions
* added playDead motion
*
* Revision 1.3  2002/10/11 15:19:55  risler
* fixed bug: MotionRequest has to be reference
*
* Revision 1.2  2002/09/17 23:55:22  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.1  2002/09/10 15:36:16  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.3  2002/08/30 14:35:06  dueffert
* removed unused includes
*
* Revision 1.2  2002/08/22 14:41:03  risler
* added some doxygen comments
*
* Revision 1.1.1.1  2002/05/10 12:40:16  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.5  2001/12/21 14:09:39  roefer
* Added several destructors
*
* Revision 1.4  2001/12/10 17:47:07  risler
* change log added
*
*/
