/**
* @file GT2003MotionNetSpecialActions.h
* 
* This file contains a class for GT2003 motion net special actions.
* @author Uwe Dffert, Max Risler
*/

#ifndef __GT2003MotionNetSpecialActions_h_
#define __GT2003MotionNetSpecialActions_h_

#include "../SpecialActions.h"
#include "MotionNetData.h"

/**
* This module executes special actions by using the GT2003
* motion net compiler
*/
class GT2003MotionNetSpecialActions : public SpecialActions
{
public:
/*
* Constructor.
* @param interfaces The paramters of the SpecialActions module.
  */
  GT2003MotionNetSpecialActions(SpecialActionsInterfaces& interfaces);
  
  /** Executes the special action */
  virtual bool executeParameterized(SpecialActionRequest specialActionRequest, JointData& jointData);

  /*
   *  Determines wether or not a Specialaction can be executed in current phase of Walking cycle 
   *	@param specialActionID SpecialActionID for Specialaction that information for is needed
   *  @return wether the Specialaction can be executed in current Walkingcycle
   */
  virtual bool specialActionIsExecutableInWalkingCycle(int specialActionID, double walkCycle);

  /** 
  * Called from a MessageQueue to distribute messages.
  * @param message The message that can be read.
  * @return true if the message was read (handled).
  */
  virtual bool handleMessage(InMessage& message);

protected:
  /** Get motion net motion index from motion type and special action id */
  int getMotionIndex(MotionRequest::MotionID motionType, SpecialActionRequest::SpecialActionID specialActionType);

  /** Get motion net motion index from a MotionRequest */
  int getMotionIndex(const SpecialActionRequest& request);

  /** Get next motion node from motion net */
  bool getNextData(const SpecialActionRequest& specialActionRequest);

  /** Calculates the next joint data vector by interpolating if necessary */
  void calculateJointData(JointData& jointData);

  /** The motion data array */
  MotionNetData motionNetData;

  /** Current motion net node */
  short currentNode;

  //! Current joint data
  JointData currentData;
  //! Last data for interpolation
  JointData lastData;
  
  //! Last data was set
  bool lastDataValid;
  //! True if values should be interpolated
  bool interpolationMode;

  //! Length of current data line in cycles
  long dataRepetitionLength;
  //! Cycle counter for current data line
  long dataRepetitionCounter;

  /** Odometry table entry
   * Contains type of odometry and odometry pose
   */
  class OdometryEntry {
    public:
    /** Enum for odometry types
     * none: No odometry, means no movement
     * once: Odometry pose is used once the motion is executed
     * homegeneous: Odometry pose describes speed and is used each tick
     */
    enum OdometryType {
      none,
      once,
      homogeneous
    };

    OdometryType type;
    OdometryData o;
    OdometryEntry() : type(none) {};
  };

  /** Odometry data table 
  * Contains one entry for each motion
  * describing robot movement per tick.
  */
  OdometryEntry odometryTable[SpecialActionRequest::numOfSpecialAction + 1];

  /** Head states for each special action */
  MotionInfo motionInfoTable[SpecialActionRequest::numOfSpecialAction + 1];

  /** A table for the executed motion requests */
  MotionRequest executedMotionRequestTable[SpecialActionRequest::numOfSpecialAction + 1];

  /** Odometry of current motion */
  OdometryEntry currentOdometry;

  /** Read odometry table from file */
  void readOdometryTable();

  /** type of last executed special action */
  SpecialActionRequest::SpecialActionID lastSpecialActionType;
};

#endif// __GT2003MotionNetSpecialActions_h_

/*
 * Change log :
 * 
 * $Log: GT2003MotionNetSpecialActions.h,v $
 * Revision 1.5  2004/06/14 15:04:40  spranger
 * changed interface of executeParametrized
 * (this is not final, the specialActionRequest will be put into the interface)
 *
 * Revision 1.4  2004/06/02 17:18:23  spranger
 * MotionRequest cleanup
 *
 * Revision 1.3  2004/05/27 19:00:31  loetzsch
 * bug fix
 *
 * Revision 1.2  2004/05/26 17:21:47  dueffert
 * better data types used
 *
 * Revision 1.1.1.1  2004/05/22 17:21:31  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/12/16 19:01:18  loetzsch
 * The motion net is not compiled into a C++ file but parsed at run time.
 *
 * Revision 1.1  2003/10/06 14:10:12  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.2  2003/06/21 14:55:06  risler
 * odometry only once per special action
 *
 * Revision 1.1  2003/03/10 14:14:08  dueffert
 * optimized MotionNet
 *
 * Revision 1.3  2003/03/06 12:06:50  dueffert
 * execute with parameters renamed to avoid inheritance warnings
 *
 * Revision 1.2  2002/09/17 23:55:22  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.1  2002/09/10 15:36:16  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/08/22 14:41:04  risler
 * added some doxygen comments
 *
 * Revision 1.1.1.1  2002/05/10 12:40:16  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.6  2002/05/05 15:14:23  risler
 * changed stand implementation
 *
 * Revision 1.5  2002/03/27 15:54:28  risler
 * added neckHeight and stableMotion in motion modules
 *
 * Revision 1.4  2001/12/18 14:02:06  risler
 * different odometry types in GT2001MotionNetSpecialActions
 *
 * Revision 1.3  2001/12/13 18:49:25  risler
 * added odometry updates in GT2001MotionNetSpecialActions
 *
 * Revision 1.2  2001/12/10 17:47:07  risler
 * change log added
 *
 */
