/**
* @file SensorDataProcessor.h
*
* Declaration of abstract class SensorDataProcessor.
*
* @author Matthias Juengel
* @author Martin Ltzsch
*/

#ifndef __SensorDataProcessor_h_
#define __SensorDataProcessor_h_

#include "Tools/RobotConfiguration.h"
#include "Tools/Module/Module.h"

#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/SensorDataBuffer.h"
#include "Representations/Perception/BodyPercept.h"
#include "Representations/Perception/BodyPosture.h"
#include "Representations/Perception/PSDPercept.h"
#include "Representations/Motion/MotionInfo.h"

/** 
* @class SensorDataProcessorInterfaces
* 
* Interfaces of the SensorDataProcessor module.
*/
class SensorDataProcessorInterfaces
{
public:
  /** Constructor */
  SensorDataProcessorInterfaces(
    const SensorDataBuffer& sensorDataBuffer,
    const unsigned long& imageFrameNumber,
    const MotionInfo& motionInfo,
    BodyPercept& bodyPercept,
    BodyPosture& bodyPosture,
    CameraMatrix& cameraMatrix,
    PSDPercept& psdPercept) 
    : sensorDataBuffer(sensorDataBuffer),
    imageFrameNumber(imageFrameNumber),
    motionInfo(motionInfo),
    bodyPercept(bodyPercept),
    bodyPosture(bodyPosture),
    cameraMatrix(cameraMatrix),
    psdPercept(psdPercept)
  {}

protected:
  /** The SensorDataBuffer to be processed. */
  const SensorDataBuffer& sensorDataBuffer;

  /** The frame number of the current image */
  const unsigned long& imageFrameNumber;

  /** The motion info last received from Motion */
  const MotionInfo& motionInfo;

  /** The BodyPercept to be calculated */
  BodyPercept& bodyPercept;

  /** The BodyPosture to be calculated */
  BodyPosture& bodyPosture;

  /** The offset and the rotation of the camera to be calculated */
  CameraMatrix& cameraMatrix;

  /** A spot relative to the robot that was detected by the PSD sensor to be calculated */
  PSDPercept& psdPercept;
};

/**
* @class SensorDataProcessor
*
* An abstract class that defines a processor for raw sensor data.
*/
class SensorDataProcessor : public Module, public SensorDataProcessorInterfaces, public RobotDimensions
{
public:
/*
* Constructor.
* @param interfaces The paramters of the ImageProcessor module.
  */
  SensorDataProcessor(const SensorDataProcessorInterfaces& interfaces)
    : SensorDataProcessorInterfaces(interfaces),
      RobotDimensions(getRobotConfiguration().getRobotDimensions())
  {}
  
  /** Destructor */
  virtual ~SensorDataProcessor() {}
};

#endif // __SensorDataProcessor_h_

/*
* Change log :
* 
* $Log: SensorDataProcessor.h,v $
* Revision 1.6  2004/05/27 09:33:18  loetzsch
* clean up
*
* Revision 1.5  2004/05/26 17:31:34  dueffert
* better data types used
*
* Revision 1.4  2004/05/26 17:21:47  dueffert
* better data types used
*
* Revision 1.3  2004/05/26 17:13:18  juengel
* Added bodyPosture.
*
* Revision 1.2  2004/05/26 15:18:44  juengel
* Added BodyPosture.
*
* Revision 1.1.1.1  2004/05/22 17:21:30  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/03/08 02:11:52  roefer
* Interfaces should be const
*
* Revision 1.2  2004/01/12 12:23:51  juengel
* SensorDataProcessor derives from RobotConfiguration now.
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.7  2003/05/02 18:26:18  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.6  2002/10/10 13:09:49  loetzsch
* First experiments with the PSD Sensor
* - SensorDataProcessor now calculates PSDPercept
* - Added the PerceptBehaviorControl solution PSDTest
* - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
*
* Revision 1.5  2002/09/17 23:55:22  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.4  2002/09/12 12:24:09  juengel
* continued change of module/solution mechanisms
*
* Revision 1.3  2002/09/10 21:07:30  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.2  2002/09/10 17:53:40  loetzsch
* First draft of new Module/Solution Mechanisms
*
* Revision 1.1  2002/09/10 15:36:16  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/08/30 14:34:37  dueffert
* removed unused includes
*
* Revision 1.1.1.1  2002/05/10 12:40:15  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.9  2002/04/02 13:10:20  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.8  2002/03/24 20:17:49  roefer
* Warning on noFrameNumber removed
*
* Revision 1.7  2002/03/24 18:15:00  loetzsch
* continued change to blocking sensor data receivers
*
* Revision 1.6  2002/03/24 14:19:46  roefer
* Input now waits for SensorData
*
* Revision 1.5  2002/01/20 23:35:06  loetzsch
* Added parameter imageFrameNumber to execute(...)
*
* Revision 1.4  2001/12/21 14:09:39  roefer
* Added several destructors
*
* Revision 1.3  2001/12/10 17:47:07  risler
* change log added
*
*/
