/**
* @file MotionRecognition.h
*
* Definition of class MotionRecognition
*
* @author <a href="mailto:ordyniak@informatik.hu-berlin.de">Sebastian Ordyniak</a>
* @author <a href="mailto:richert@informatik.hu-berlin.de">Marten Richert</a>
*/

#ifndef __MotionRecognition_h_
#define __MotionRecognition_h_

#include "Modules/SensorBehaviorControl/SensorBehaviorControl.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Debugging/DebugImages.h"
#include "Tools/Actorics/RobotDimensions.h"

/** distance of infinty in mm */
#define DISTANCE_INFINITY 3000
/** mm*/
#define BRENNWEITE 2.18
#define IMAGEBUFFERSIZE 2

/**
* @class MotionRecognition
*
* @author <a href="mailto:ordyniak@informatik.hu-berlin.de">Sebastian Ordyniak</a>
* @author <a href="mailto:richert@informatik.hu-berlin.de">Marten Richert</a>
*/ 

class MotionRecognition : public SensorBehaviorControl
{
public:
  /** 
  * Constructor.
  * @param interfaces The paramters of the MotionRecognition module.
  */
  MotionRecognition(const SensorBehaviorControlInterfaces& interfaces);

  /**
  * transforms pixelcoordinates to grad
  * @param pCentered - centered xy-pixelcoordinates
  * @return centered angle coors in grad
  */
  Vector2<double> getAngleYZ(Vector2<int> pCentered);


  int isDiff(Image& image1, Image& image2, Vector2<int> p1, Vector2<int> p2);

  /**
  * shifts coordinatesource from top-left to center of image
  * @param p top-left coordinates
  * @return centered imagecoordinates
  */
  Vector2<int> getCenteredCoor(Vector2<int> p);

  /**
  * transforms pixelcoordinates to mm
  * @param p pixel coordinates
  * @return coordinates in mm
  */
  Vector2<double> getCoorInmm(Vector2<int> p);
  
  /**
  * transforms pixelcoordinates to mm
  * @param p pixel coordinates
  * @return coordinates in mm
  */
  Vector2<int> getCoorInPixel(Vector2<double> p);

  /**
  * calculates xy-shift of a pixel
  * @param pCentered centered pixelcoordinates
  * @param z distance from the object belonging to pixel to camera
  * @param cameraMovement movement of the camera (in mm)
  * @param cameraRotation rotation of the camera (in grad)
  * @return xy-shift of pixel
  */
  Vector2<int> getPixelFlow(Vector2<double> pCentered, double z, Vector3<double> cameraMovement, Vector3<double> cameraRotation);

  /**
  * calculates newPixelposition after movement and rotation of the camera
  * @param p pixelcoordinates (coordinatesource = top-left
  * @param cameraMovement movement of the camera (in mm)
  * @param cameraRotation rotation of the camera (in grad)
  * @return new pixelposition
  */
  Vector2<int> getNewPixelPos(Vector2<int> p, Vector3<double> cameraMovement, Vector3<double> cameraRotation);

  /**
  * calculates distance from camera to object (xy-plain)
  * @param angleYZ angle of cameraray
  * @return istance from camera to object (xy-plain)
  */
  double getCameraZ(Vector2<double> angleYZ);
  
  /**
  * calculates cameratranslation
  * @param robotTranslation
  * @return cameraTranslation
  */
  Vector3<double> getCameraTranslation(Vector3<double> robotTranslation);

  Vector3<double> getCameraTranslation(double bounce);
  Vector3<double> getCameraRotation(double bounce);

	Vector3<double> getRobotTranslationFromOdometry();

	Vector3<double> getRobotTranslationFromBounce(double bounce);
	
	Vector3<double> getRobotRotationFromOdometry();

	Vector3<double> getRobotRotationFromBounce(double bounce);


  Vector3<double> getRobotTranslationForRotation(double robotRotationZ);
  int pixelInImage(Vector2<int> p);


  double getPixelDiff(Vector2<int> raster, double bounce, Image& image1, Image& image2, int draw);
  double getPixelDiff(Vector2<int> raster, double bounce, Image& image1, Image& image2);
  void drawPixelFlow(Vector3<double> cameraTranslation, Vector3<double> cameraRotation);
  void drawDynamicDiff(Image& image1, Image& image2, double timeDiff);

  /** Executes the module */
  virtual void execute();
  virtual bool handleMessage(InMessage& message);

  DECLARE_DEBUG_IMAGE(imageMotionRecognition);
  DECLARE_DEBUG_IMAGE(imageProcessorGradients);

private:
	OdometryData previousOdometry;
	Vector2<double> pixelDimensions;
	Image imageBuffer[IMAGEBUFFERSIZE];
	unsigned long imageTimes[IMAGEBUFFERSIZE];
	int start, currentImage, imagesLeft;

	// images for debugging
	Image motionRecognitionImage, processorGradientsImage;
};
#endif// __MotionRecognition_h_

/*
* Change log :
* 
* $Log: MotionRecognition.h,v $
* Revision 1.1.1.1  2004/05/22 17:20:52  cvsadm
* created new repository GT2004_WM
*
* Revision 1.9  2004/03/08 02:11:49  roefer
* Interfaces should be const
*
* Revision 1.8  2004/01/29 17:03:54  dueffert
* doxygen bug fixed
*
* Revision 1.7  2004/01/07 17:16:00  ordyniak
* debugimages angepasst
*
* Revision 1.6  2004/01/07 16:53:42  ordyniak
* umgestellt auf odometrydata fr logfiles
*
* Revision 1.5  2004/01/04 10:10:27  juengel
* Errors and warnings removed.
*
* Revision 1.4  2004/01/03 23:12:51  ordyniak
* new concept
*
*/
