/**
* @file DemoSensorActorLoop.cpp
*
* Implementation of class DemoSensorActorLoop
* 
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#include "DemoSensorActorLoop.h"
#include "Platform/SystemCall.h"
#include "Tools/Math/Common.h"
#include "Tools/Actorics/RobotDimensions.h"
#include "Tools/RobotConfiguration.h"

DemoSensorActorLoop::DemoSensorActorLoop(SensorActorLoopInterfaces& interfaces)
: SensorActorLoop(interfaces),
RobotDimensions(getRobotConfiguration().getRobotDimensions())
{
  tailCyclePosition = 0;
}

void DemoSensorActorLoop::execute()
{
  unsigned long currentTime = SystemCall::getCurrentSystemTime();
  // The module was not active for a while
  if( (currentTime - timeOfLastExecute) > 2000 ) init();
  timeOfLastExecute = currentTime;

  // change state when the back button is pressed
  if(
    SystemCall::getTimeSince(timeOfLastStateChange) > 300 &&
    sensorDataBuffer.lastFrame().data[SensorData::back] == 1
    )
  {
    timeOfLastStateChange = SystemCall::getCurrentSystemTime();
    switch(state)
    {
    case sleep:       state = fly; break;
    case fly:         state = motionClone; break;
    case motionClone: state = waveHand; break;
    case waveHand:    state = fly; break;
    }
  }

  // use the tail to show the remaining battery power
  tailCyclePosition += 3;
  tailCyclePosition %= 360;

  JointData &currentFrame = jointDataBuffer.frame[0];

  currentFrame.data[JointData::tailPan] = (int)(
    sin(fromDegrees(tailCyclePosition)) * 
    toMicroRad(jointLimitTailPanP) * 
    SystemCall::getRemainingPower() /
    100.0
    );

  currentFrame.data[JointData::tailTilt] = 0;


  // execute the behavior for the current state
  switch(state)
  {
  case sleep: 
    execute_sleep(); 
    ledRequest.redTopLEDs = LEDRequest::bothSlowBlink;
    ledRequest.redBottomLEDs = LEDRequest::bothSlowBlink;
    break;
  case fly: 
    execute_fly(); 
    ledRequest.redTopLEDs = LEDRequest::leftOnly;
    ledRequest.redBottomLEDs = LEDRequest::bothOff;
    break;
  case motionClone: 
    execute_motionClone();
    ledRequest.redTopLEDs = LEDRequest::bothOn;
    ledRequest.redBottomLEDs = LEDRequest::bothOff;
    break;
  case waveHand:
    execute_waveHand();
    ledRequest.redTopLEDs = LEDRequest::bothOn;
    ledRequest.redBottomLEDs = LEDRequest::leftOnly;
    break;
  }

}

void DemoSensorActorLoop::init()
{
  state = sleep;
  timeOfLastStateChange = SystemCall::getCurrentSystemTime();
}

void DemoSensorActorLoop::execute_sleep()
{
  pidData.setValues(JointData::neckTilt, 0, 0, 0);
  pidData.setValues(JointData::headPan, 0, 0, 0);
  pidData.setValues(JointData::headTilt, 0, 0, 0);
  
  pidData.setValues(JointData::legFR1, 0, 0, 0);
  pidData.setValues(JointData::legFR2, 0, 0, 0);
  pidData.setValues(JointData::legFR3, 0, 0, 0);
  
  pidData.setValues(JointData::legHR1, 0, 0, 0);
  pidData.setValues(JointData::legHR2, 0, 0, 0);
  pidData.setValues(JointData::legHR3, 0, 0, 0);
  
  pidData.setValues(JointData::legFL1, 0, 0, 0);
  pidData.setValues(JointData::legFL2, 0, 0, 0);
  pidData.setValues(JointData::legFL3, 0, 0, 0);
  
  pidData.setValues(JointData::legHL1, 0, 0, 0);
  pidData.setValues(JointData::legHL2, 0, 0, 0);
  pidData.setValues(JointData::legHL3, 0, 0, 0);
  
  JointData &currentFrame = jointDataBuffer.frame[0];

  currentFrame.data[JointData::legFL1] = 0;
  currentFrame.data[JointData::legFL2] = toMicroRad(fromDegrees(15));
  currentFrame.data[JointData::legFL3] = toMicroRad(pi_2);
  
  currentFrame.data[JointData::legHL1] = 0;
  currentFrame.data[JointData::legHL2] = toMicroRad(fromDegrees(15));
  currentFrame.data[JointData::legHL3] = toMicroRad(pi_2);
  
  currentFrame.data[JointData::legFR1] = 0;
  currentFrame.data[JointData::legFR2] = toMicroRad(fromDegrees(15));
  currentFrame.data[JointData::legFR3] = toMicroRad(pi_2);
  
  currentFrame.data[JointData::legHR1] = 0;
  currentFrame.data[JointData::legHR2] = toMicroRad(fromDegrees(15));
  currentFrame.data[JointData::legHR3] = toMicroRad(pi_2);
}

void DemoSensorActorLoop::execute_fly()
{
  pidData.setToDefaults();

  JointData &currentFrame = jointDataBuffer.frame[0];

  //calculate the average over the last acceleration sensor readings
  long int accelerationAverage[3] = {0,0,0};
  for (int i = 0; i < sensorDataBuffer.numOfFrames; i++)
  {
    accelerationAverage[0] += sensorDataBuffer.frame[i].data[SensorData::accelerationX];
    accelerationAverage[1] += sensorDataBuffer.frame[i].data[SensorData::accelerationY];
    accelerationAverage[2] += sensorDataBuffer.frame[i].data[SensorData::accelerationZ];
  }

  accelerationAverage[0] /= sensorDataBuffer.numOfFrames;
  accelerationAverage[1] /= sensorDataBuffer.numOfFrames;
  accelerationAverage[2] /= sensorDataBuffer.numOfFrames;
  
  // set the tail joints 
  currentFrame.data[JointData::tailTilt] = -accelerationAverage[0] / 2;
  currentFrame.data[JointData::tailPan] = -accelerationAverage[1] / 2;

  // determine the darkest point of a horizontal scan line
  unsigned char yMin = 255;
  int darkPointX = image.cameraInfo.resolutionWidth / 2;
  int y = image.cameraInfo.resolutionHeight / 2;
  
  for(int x = 0; x < image.cameraInfo.resolutionWidth; x++)
  {
    if(image.image[y][0][x] < yMin) 
    {
      yMin = image.image[y][0][x];
      darkPointX = x;
    }
  }

  // move the head to the direction of the dark point
  if(darkPointX < image.cameraInfo.resolutionWidth / 2 - 10)
  {
    currentFrame.data[JointData::headPan] = 
      sensorDataBuffer.frame[0].data[SensorData::headPan] + toMicroRad(fromDegrees(5));
  }
  else if(darkPointX < image.cameraInfo.resolutionWidth / 2 + 10)
  {
    currentFrame.data[JointData::headPan] = 
      sensorDataBuffer.frame[0].data[SensorData::headPan];
  }
  else
  {
    currentFrame.data[JointData::headPan] = 
      sensorDataBuffer.frame[0].data[SensorData::headPan] - toMicroRad(fromDegrees(5));
  }

  currentFrame.data[JointData::neckTilt] = 0;
  currentFrame.data[JointData::headTilt] = 0;
}

void DemoSensorActorLoop::execute_motionClone()
{
  pidData.setToDefaults();

  pidData.setValues(JointData::legFR1, 0, 0, 0);
  pidData.setValues(JointData::legFR2, 0, 0, 0);
  pidData.setValues(JointData::legFR3, 0, 0, 0);

  pidData.setValues(JointData::legHR1, 0, 0, 0);
  pidData.setValues(JointData::legHR2, 0, 0, 0);
  pidData.setValues(JointData::legHR3, 0, 0, 0);

  JointData &currentFrame = jointDataBuffer.frame[0];
  
  currentFrame.data[JointData::neckTilt] = 0;
  currentFrame.data[JointData::headPan] = 0;
  currentFrame.data[JointData::headTilt] = 0;

  currentFrame.data[JointData::legFL1] = sensorDataBuffer.frame[0].data[SensorData::legFR1];
  currentFrame.data[JointData::legFL2] = sensorDataBuffer.frame[0].data[SensorData::legFR2];
  currentFrame.data[JointData::legFL3] = sensorDataBuffer.frame[0].data[SensorData::legFR3];
  
  currentFrame.data[JointData::legHL1] = sensorDataBuffer.frame[0].data[SensorData::legHR1];
  currentFrame.data[JointData::legHL2] = sensorDataBuffer.frame[0].data[SensorData::legHR2];
  currentFrame.data[JointData::legHL3] = sensorDataBuffer.frame[0].data[SensorData::legHR3];
}

void DemoSensorActorLoop::execute_waveHand()
{
  pidData.setToDefaults();

  double psdMax = 900000.0;
  double psdAverage = 0;
  for (int i = 0; i < sensorDataBuffer.numOfFrames; i++)
  {
    psdAverage += sensorDataBuffer.frame[i].data[SensorData::psd];
  }
  psdAverage /= sensorDataBuffer.numOfFrames;

  JointData &currentFrame = jointDataBuffer.frame[0];
  
  currentFrame.data[JointData::neckTilt] = 0;
  currentFrame.data[JointData::headPan] = 0;
  currentFrame.data[JointData::headTilt] = 0;

  currentFrame.data[JointData::legFL1] = 0;
  currentFrame.data[JointData::legFL2] = toMicroRad(fromDegrees(15));
  currentFrame.data[JointData::legFL3] = toMicroRad(pi_2);
  
  currentFrame.data[JointData::legHL1] = 0;
  currentFrame.data[JointData::legHL2] = toMicroRad(fromDegrees(15));
  currentFrame.data[JointData::legHL3] = toMicroRad(pi_2);

  currentFrame.data[JointData::legFR1] = 0;
  currentFrame.data[JointData::legFR2] = (int)(toMicroRad(jointLimitLeg2P) * psdAverage / psdMax);
  currentFrame.data[JointData::legFR3] = (int)(toMicroRad(jointLimitLeg3P) * (1.0 - psdAverage / psdMax) );
  
  currentFrame.data[JointData::legHR1] = 0;
  currentFrame.data[JointData::legHR2] = toMicroRad(fromDegrees(15));
  currentFrame.data[JointData::legHR3] = toMicroRad(pi_2);
}


bool DemoSensorActorLoop::handleMessage(InMessage& message)
{
  return false;
}

/*
* Change log :
* 
* $Log: DemoSensorActorLoop.cpp,v $
* Revision 1.2  2004/05/27 17:13:37  jhoffman
* - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
* - clipping included for setJoints
* - removed some microrad/rad-bugs
* - bodyPosture constructor and "=" operator fixed
*
* Revision 1.1.1.1  2004/05/22 17:20:49  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/01/13 15:59:38  loetzsch
* made compilable
*
* Revision 1.2  2003/10/29 19:13:22  juengel
* Head can follow dark objects now.
*
* Revision 1.1  2003/10/29 13:02:51  juengel
* Added module SensorActorLoop.
*
*
*/
