/**
 * @file Modules/SelfLocator/MonteCarloSelfLocator.cpp
 * 
 * This file contains a class for self-localization based on the Monte Carlo approach.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 * @author <A href=mailto:asbre01@tzi.de>Andreas Sztybryc</A>
 */

#include "MonteCarloSelfLocator.h"
#include "Tools/FieldDimensions.h"


MonteCarloSelfLocator::MonteCarloSelfLocator(const SelfLocatorInterfaces& interfaces)
: SelfLocator(interfaces)
{
  largeField.x.min = field.x.min * 1.3;
  largeField.x.max = field.x.max * 1.3;
  largeField.y.min = field.y.min * 1.3;
  largeField.y.max = field.y.max * 1.3;
  sensorUpdated = false;
  factor = 50;
  numOfFlags = 0;
  reset();
}

void MonteCarloSelfLocator::execute()
{
  robotPose.setFrameNumber(landmarksPercept.frameNumber);

  Pose2D odometry = odometryData - lastOdometry;
  lastOdometry = odometryData;

  // generate templates for calculated sampleSet
  generatePoseTemplates(landmarksPercept,odometry);

  // First step: Motion Module
  updateByOdometry(odometry,
                   Pose2D(landmarksPercept.cameraOffset.x,
                                  landmarksPercept.cameraOffset.y),
                   sensorUpdated); 

  // Second step: Vision Module
  sensorUpdated = false;
  int i;  
  for(i = 0; i < landmarksPercept.numberOfFlags; ++i)
  {
    const Flag& flag = landmarksPercept.flags[i];
    if(!flag.isOnBorder(flag.x.max))
      updateFlag(flag.position,
                 LEFT_SIDE_OF_FLAG,
                 flag.x.max);
    if(!flag.isOnBorder(flag.x.min))
      updateFlag(flag.position,
                 RIGHT_SIDE_OF_FLAG,
                 flag.x.min);
  }
  for(i = 0; i < landmarksPercept.numberOfGoals; ++i)
  {
    const Goal& goal = landmarksPercept.goals[i];
    if(!goal.isOnBorder(goal.x.max))
      updateGoalPost(goal.leftPost,
                     goal.x.max);
    if(!goal.isOnBorder(goal.x.min))
      updateGoalPost(goal.rightPost,
                     goal.x.min);
  }

  if(sensorUpdated)
    resample();
  
  Pose2D pose;
  double validity;
  calcPose(pose,validity);

  robotPose.setPose(pose);
  robotPose.setValidity(validity);
  sampleSet.link(selfLocatorSamples);

  COMPLEX_DRAWING(selfLocatorField,draw(););
  DEBUG_DRAWING_FINISHED(selfLocatorField); 
}

void MonteCarloSelfLocator::reset()
{
  for(int i = 0; i < SAMPLES_MAX; ++i)
    sampleSet[i] = Sample(field.randomPose(),0); 
}

void MonteCarloSelfLocator::updateByOdometry(const Pose2D& odometry,
                                             const Pose2D& camera,
                                             bool noise)
{     
  double transNoise = noise ? 200 : 0,
         rotNoise = noise ? 0.5 : 0;    
  double dist = odometry.translation.abs();    
  double angle = fabs(odometry.getAngle());
  
  for(int i = 0; i < SAMPLES_MAX; ++i)
  {
    Sample& s = sampleSet[i];
    s += odometry;
    double rotError = ((random() * 2 - 1) * max(dist * 0.002 + angle * 0.2,rotNoise * (1 - s.probability)));
    Vector2<double> transError((random() * 2 - 1) * max(dist * 0.1,transNoise * (1 - s.probability)),
                               (random() * 2 - 1) * max(dist * 0.02,transNoise * (1 - s.probability)));
    s += Pose2D(rotError,transError);
    if(!largeField.isInside(s.translation))
      s = getTemplate();
    s.camera = s + camera;
    s.quality = 1;
  } 
}

void MonteCarloSelfLocator::updateFlag(const Vector2<double>& flag,
                                       FlagSides sideOfFlag,
                                       double measuredBearing)
{
  sensorUpdated = true;
  for(int i = 0; i < SAMPLES_MAX; ++i)
  {
    Sample& s = sampleSet[i];
    Pose2D p(flag.x,flag.y);
    p -= s.camera;
    double assumedBearing = atan2(p.translation.y,p.translation.x) + sideOfFlag * asin(flagRadius / p.translation.abs());
    double similarity = fabs((assumedBearing - measuredBearing)) / pi;
    if(similarity > 1)
      similarity = 2.0 - similarity;
    s.quality *= sigmoid(similarity);
  }
}

void MonteCarloSelfLocator::updateGoalPost(const Vector2<double>& goalPost,
                                           double measuredBearing)
{ 
  sensorUpdated = true;
  for(int i = 0; i < SAMPLES_MAX; ++i)
  {
    Sample& s = sampleSet[i];
    Pose2D p(goalPost.x,goalPost.y);
    p -= s.camera;
    double assumedBearing = atan2(p.translation.y,p.translation.x);
    double similarity = fabs((assumedBearing - measuredBearing)) / pi;
    if(similarity > 1)
      similarity = 2 - similarity;
    s.quality *= sigmoid(similarity); 
  }
}

void MonteCarloSelfLocator::resample()
{
  int i;

  // move the probability of all samples towards the quality
  const double upStep = 0.1,
               downStep = 0.05;
  for(i = 0; i < SAMPLES_MAX; ++i)
  {
    Sample& s = sampleSet[i];
    if(s.probability == 2)
      s.probability = s.quality;
    else if(s.quality < s.probability)
      if(s.quality < s.probability - downStep)
        s.probability -= downStep;
      else
        s.probability = s.quality;
    else 
      if(s.quality > s.probability + upStep)
        s.probability = s.probability + upStep;
      else
        s.probability = s.quality;
  }
    
  // swap sample arrays
  Sample* oldSet = sampleSet.swap();
  
  double probSum = 0;
  for(i = 0; i < SAMPLES_MAX; ++i)
    probSum += oldSet[i].probability;
  double prob2Index = (double) SAMPLES_MAX / probSum;

  double indexSum = 0;
  int j = 0;
  for(i = 0; i < SAMPLES_MAX; ++i)
  {
    indexSum += oldSet[i].probability * prob2Index;
    while(j < SAMPLES_MAX && j < indexSum - 0.5)
      sampleSet[j++] = oldSet[i];
  }

  for(i = 0; i < SAMPLES_MAX; ++i)
  {
    Sample& s = sampleSet[i];
    // if probability is too low, select a new random pose for the sample
    if(random() > s.probability)
      s = getTemplate();
  }
}
  
void MonteCarloSelfLocator::calcPose(Pose2D& pose,double& validity)
{
  Cell cells[GRID_MAX][GRID_MAX][GRID_MAX];
  double width = field.x.getSize(),
         height = field.y.getSize();

  // attach the samples to the corresponding grid cells
  for(int i = 0; i < SAMPLES_MAX; ++i)
  {
    Sample& p = sampleSet[i];
    if(p.probability != 2 && p.probability != 0)
    {
      int r = static_cast<int>((p.getAngle() / pi2 + 0.5) * GRID_MAX),
          y = static_cast<int>((p.translation.y / height + 0.5) * GRID_MAX),
          x = static_cast<int>((p.translation.x / width + 0.5) * GRID_MAX);
      if(x < 0)
        x = 0;
      else if(x >= GRID_MAX)
        x = GRID_MAX-1;
      if(y < 0)
        y = 0;
      else if(y >= GRID_MAX)
        y = GRID_MAX-1;
      if(r < 0)
        r = GRID_MAX-1;
      else if(r >= GRID_MAX)
        r = 0;
      Cell& c = cells[r][y][x];
      p.next = c.first;
      c.first = &p;
      ++c.count;
    }
  }  
  
  // determine the 2x2x2 sub-cube that contains most samples
  int rMax = 0,
      yMax = 0,
      xMax = 0,
      countMax = 0,
      r;
  for(r = 0; r < GRID_MAX; ++r)
  {
    int rNext = (r + 1) % GRID_MAX;
    for(int y = 0; y < GRID_MAX - 1; ++y)
      for(int x = 0; x < GRID_MAX - 1; ++x)
      {
        int count = cells[r][y][x].count +
                    cells[r][y][x+1].count +
                    cells[r][y+1][x].count +
                    cells[r][y+1][x+1].count +
                    cells[rNext][y][x].count +
                    cells[rNext][y][x+1].count +
                    cells[rNext][y+1][x].count +
                    cells[rNext][y+1][x+1].count;
        if(count > countMax)
        {
          countMax = count;
          rMax = r;
          yMax = y;
          xMax = x;
        }
      }
  }

  if(countMax > 0)
  {
    // determine the average pose of all samples in the winner sub-cube
    double xSum = 0,
                  ySum = 0,
                  cosSum = 0,
                  sinSum = 0,
                  probabilitySum = 0;
    for(r = 0; r < 2; ++r)
    {
      int r2 = (rMax + r) % GRID_MAX;
      for(int y = 0; y < 2; ++y)
        for(int x = 0; x < 2; ++x)
        {
          for(Sample* p = cells[r2][yMax + y][xMax + x].first; p; p = p->next)
          {
            xSum += p->translation.x * p->probability;
            ySum += p->translation.y * p->probability;
            cosSum += p->getCos() * p->probability;
            sinSum += p->getSin() * p->probability;
            probabilitySum += p->probability;
          }
        }
    }  
    if(probabilitySum)
    {
      pose = Pose2D(atan2(sinSum,cosSum),
                    Vector2<double>(xSum / probabilitySum,ySum / probabilitySum));
      validity = probabilitySum / SAMPLES_MAX;
      double diff = field.clip(pose.translation);
      if(diff > 1)
        validity /= sqrt(diff);
      return;
    }
  }
  validity = 0;
}

///////////////////////////////////////////////////////////////////////
// New samples, i.e. random and templates

bool MonteCarloSelfLocator::poseFromBearings(double dir0,double dir1,double dir2,
                                             const Vector2<double>& mark0,
                                             const Vector2<double>& mark1,
                                             const Vector2<double>& mark2,
                                             const Vector2<double>& cameraOffset,
                                             Pose2D& resultingPose) const 
{
  double w1 = dir1 - dir0,
         w2 = dir2 - dir1;
  if(sin(w1) != 0 && sin(w2) != 0)
  {
    Vector2<double> p = mark2 - mark1;
    double a2 = p.abs();
    p = (Pose2D(mark0) - Pose2D(atan2(p.y,p.x),mark1)).translation;
    double a1 = p.abs();
    if(a1 > 0 && a2 > 0)
    {
      double w3 = pi2 - w1 - w2 - atan2(p.y,p.x);
      double t = a1 * sin(w2) / (a2 * sin(w1)) + cos(w3);
      if(t != 0)
      {
        double w = atan(sin(w3) / t);
        double b = a1 * sin(w) / sin(w1);
        w = pi - w1 - w;
        p = mark0 - mark1;
        resultingPose = Pose2D(atan2(p.y,p.x) - w,mark1)
                        + Pose2D(Vector2<double>(b,0));
        p = (Pose2D(mark1) - resultingPose).translation;
        resultingPose += Pose2D(atan2(p.y,p.x) - dir1);
        resultingPose += Pose2D(Vector2<double>(-cameraOffset.x,
                                                -cameraOffset.y));
        return true;
      }
    }
  }
  return false;
}

int MonteCarloSelfLocator::poseFromBearingsAndDistance(
  double dir0,double dir1,double dist,
  const Vector2<double>& mark0,
  const Vector2<double>& mark1,
  const Vector2<double>& cameraOffset,
  Pose2D& resultingPose1,
  Pose2D& resultingPose2) const
{
  double alpha = dir0 - dir1;
  Vector2<double> diff = mark0 - mark1;
  double dir10 = atan2(diff.y,diff.x);
  double f = dist * sin(alpha) / diff.abs();
  if(fabs(f) <= 1)
  {
    double gamma = asin(f),
           beta = pi - alpha - gamma;
    resultingPose1 = Pose2D(dir10 + beta,mark1) + 
                     Pose2D(pi - dir1,Vector2<double>(dist,0)) +
                     Pose2D(Vector2<double>(-cameraOffset.x,
                            -cameraOffset.y));
    if(fabs(alpha) < pi_2 || fabs(alpha) > 3 * pi_2)
    {
      if(gamma > 0)
        gamma = pi - gamma;
      else
        gamma = -pi - gamma;
      beta = pi - alpha - gamma;
      resultingPose2 = Pose2D(dir10 + beta,mark1) + 
                       Pose2D(pi - dir1,Vector2<double>(dist,0)) +
                       Pose2D(Vector2<double>(-cameraOffset.x,
                              -cameraOffset.y));
      return 2;
    }
    else
      return 1;
  }
  else
    return 0;
}

bool MonteCarloSelfLocator::getBearing(const LandmarksPercept& landmarksPercept,int i,
                                       Vector2<double>& mark,
                                       double& dir,double& dist) const
{
  if(i < numOfFlags)
  {
    const Flag& flag = flags[i];
    mark = flag.position;
    dir = flag.angle;
    dist = flag.distance;
    return true;
  }
  else
  {
    i -= numOfFlags;
    const Goal& goal = landmarksPercept.goals[i >> 1];
    if((i & 1) == 0 && !goal.isOnBorder(goal.x.min))
    {
      mark = goal.rightPost;
      dir = goal.x.min;
      dist = goal.distance;
      return true;
    }
    else if((i & 1) == 1 && !goal.isOnBorder(goal.x.max))
    {
      mark = goal.leftPost;
      dir = goal.x.max;
      dist = goal.distance;
      return true;
    }
    else
      return false;
  }
}

void MonteCarloSelfLocator::addFlag(const Flag& flag)
{
  if(!flag.isOnBorder(flag.x.min) && !flag.isOnBorder(flag.x.max))
  {
    int i;
    for(i = 0; i < numOfFlags && flags[i].type != flag.type; ++i)
      ;
    if(i < numOfFlags)
      --numOfFlags; // remove one
    if(i < FLAGS_MAX)
      ++i; // still room at the end
    while(--i > 0)
      flags[i] = flags[i - 1];
    flags[0] = flag;
    if(numOfFlags < FLAGS_MAX)
      ++numOfFlags; // one was added
  }
}

void MonteCarloSelfLocator::generatePoseTemplates(const LandmarksPercept& landmarksPercept,
                                                  const Pose2D& odometry)
{
  randomFactor = 0;
  int i;
  for(i = 0; i < numOfFlags; ++i)
  {
    Flag& flag = flags[i];
    Vector2<double> p = (Pose2D(flag.angle) 
      + Pose2D(Vector2<double>(flag.distance,0)) - odometry).translation;
    flag.angle = atan2(p.y,p.x);
    flag.distance = p.abs();
  }
  for(i = 0; i < landmarksPercept.numberOfFlags; ++i)
    addFlag(landmarksPercept.flags[i]);

  numOfTemplates = 0;
  nextTemplate = 0;
  double dir0,
         dir1,
         dir2;
  double dist0,
         dist1;
  Vector2<double> mark0,
                  mark1,
                  mark2,
                  cameraOffset(landmarksPercept.cameraOffset.x,
                               landmarksPercept.cameraOffset.y);
  int n = numOfFlags + 2 * landmarksPercept.numberOfGoals;
  for(i = 0; i < n - 2; ++i)
    if(getBearing(landmarksPercept,i,mark0,dir0,dist0))
      for(int j = i + 1; j < n - 1; ++j)
        if(getBearing(landmarksPercept,j,mark1,dir1,dist1))
        {
          if(numOfTemplates >= SAMPLES_MAX - 4)
            return;
          numOfTemplates += poseFromBearingsAndDistance(dir0,dir1,dist1,mark0,mark1,cameraOffset,
                                                        templates[numOfTemplates],templates[numOfTemplates + 1]);
          numOfTemplates += poseFromBearingsAndDistance(dir1,dir0,dist0,mark1,mark0,cameraOffset,
                                                        templates[numOfTemplates],templates[numOfTemplates + 1]);
          for(int k = j + 1; k < n; ++k)
            if(getBearing(landmarksPercept,k,mark2,dir2,dist0))
              if(poseFromBearings(dir0,dir1,dir2,mark0,mark1,mark2,cameraOffset,
                                  templates[numOfTemplates]))
                if(++numOfTemplates == SAMPLES_MAX)
                  return;
        }
}

MonteCarloSelfLocator::Sample MonteCarloSelfLocator::getTemplate()
{
  if(nextTemplate >= numOfTemplates)
  {
    nextTemplate = 0;
    ++randomFactor;
  }
  while(nextTemplate < numOfTemplates)
  {
    const Pose2D& t = templates[nextTemplate++];
    if(field.isInside(t.translation))
    {
      return Sample(Pose2D::random(Range<double>(t.translation.x - randomFactor * 50,
                                                 t.translation.x + randomFactor * 50),
                                   Range<double>(t.translation.y - randomFactor * 50,
                                                 t.translation.y + randomFactor * 50),
                                   Range<double>(t.getAngle() - randomFactor * 0.1,
                                                 t.getAngle() + randomFactor * 0.1)),
                    2);
    }
  }
  return Sample(field.randomPose(),0);
}

///////////////////////////////////////////////////////////////////////
// Debug drawing

void MonteCarloSelfLocator::draw() const
{ 
  for(int j = 0; j < SAMPLES_MAX; ++j)
    draw(sampleSet[j], Drawings::black);
  for(int k = 0; k < numOfTemplates; ++k)
    draw(templates[k], Drawings::red);
}

void MonteCarloSelfLocator::draw(const Pose2D& pose, Drawings::Color color) const
{ 
  Pose2D p = pose + Pose2D(-100,0);
  LINE(selfLocatorField,
          (int)pose.translation.x,
          (int)pose.translation.y,
          (int)p.translation.x,
          (int)p.translation.y,
          1,
          Drawings::ps_solid,
          color);
  p = pose + Pose2D(-40,-40);
  LINE(selfLocatorField,
          (int)pose.translation.x,
          (int)pose.translation.y,
          (int)p.translation.x,
          (int)p.translation.y,
          1,
          Drawings::ps_solid,
          color);
  p = pose + Pose2D(-40,40);
  LINE(selfLocatorField,
          (int)pose.translation.x,
          (int)pose.translation.y,
          (int)p.translation.x,
          (int)p.translation.y,
          1,
          Drawings::ps_solid,
          color);    
}

/*
 * Change log :
 * 
 * $Log: MonteCarloSelfLocator.cpp,v $
 * Revision 1.1.1.1  2004/05/22 17:20:46  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.4  2004/03/08 02:11:47  roefer
 * Interfaces should be const
 *
 * Revision 1.3  2003/11/14 19:02:26  goehring
 * frameNumber added
 *
 * Revision 1.2  2003/11/13 16:47:38  goehring
 * frameNumber added
 *
 * Revision 1.1  2003/10/06 14:10:14  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/01 10:20:11  juengel
 * DebugDrawings clean-up 2
 * DebugImages clean-up
 * MessageIDs clean-up
 * Stopwatch clean-up
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.15  2003/05/27 09:43:45  mkunz
 * + 50 -> * 50 in getTemplate()
 *
 * Revision 1.14  2003/05/12 14:08:40  brunn
 * renamed selfLocationSampleSetProxy to selfLocatorSamples
 * "Ha, glad am I that no one knew that Rumpelstiltskin I am styled"
 *
 * Revision 1.13  2003/05/12 12:28:11  brunn
 * renamed sampleSetProxy to selfLocationSampleSetProxy
 * added selfLocationSampleSetProxy to BehaviourControl-Interfaces
 *
 * Revision 1.12  2003/05/08 23:52:24  roefer
 * SampleSet and SampleSetProxy added
 *
 * Revision 1.11  2003/03/22 18:11:24  roefer
 * Warnings removed
 *
 * Revision 1.10  2002/12/17 20:58:38  roefer
 * COMPLEX_DRAWING changed
 *
 * Revision 1.9  2002/12/12 22:09:41  roefer
 * Progress in LinesSelfLocator
 *
 * Revision 1.8  2002/11/12 10:49:02  juengel
 * New debug drawing macros - second step
 * -moved /Tools/Debugging/PaintMethods.h and . cpp
 *  to /Visualization/DrawingMethods.h and .cpp
 * -moved DebugDrawing.h and .cpp from /Tools/Debugging/
 *  to /Visualization
 *
 * Revision 1.7  2002/11/07 17:04:42  dueffert
 * gcc mag keine anonymen Referenzen. Wenn man eine Referenz
 * uebergeben will, muss man vorher eine ordentliche Variable hernehmen.
 *
 * Revision 1.6  2002/11/07 13:49:06  roefer
 * poseFromBearingAndDistance fixed, templates distributed and painted
 *
 * Revision 1.5  2002/09/23 13:54:49  risler
 * abs replaced by fabs/labs
 *
 * Revision 1.4  2002/09/22 18:40:53  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.3  2002/09/17 23:55:21  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.2  2002/09/12 12:24:09  juengel
 * continued change of module/solution mechanisms
 *
 * Revision 1.1  2002/09/10 15:36:16  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.7  2002/08/31 13:40:06  roefer
 * MonteCarlo added
 *
 * Revision 1.6  2002/08/30 14:34:06  dueffert
 * removed unused includes
 *
 * Revision 1.5  2002/06/22 04:35:35  Thomas Rfer
 * Locator less precise and less jumping
 *
 * Revision 1.4  2002/06/11 10:22:21  kallnik
 * Pose2D neue Version (RotationVector) fertig
 * PLEASE DON'T  READ ROTATION DIRECTLY USE getAngle()
 *
 * Revision 1.3  2002/06/02 23:21:09  roefer
 * Single color table and progress in LinesSelfLocator
 *
 * Revision 1.2  2002/05/28 20:29:29  roefer
 * Sample number reduced to 50
 *
 * Revision 1.1.1.1  2002/05/10 12:40:15  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.21  2002/05/02 11:32:08  risler
 * replaced sin,.. with sin,..
 *
 * Revision 1.20  2002/04/29 08:36:32  kallnik
 * changed several doubles to double
 * changed some GTMathClassDouble top GTMath
 *
 * Revision 1.19  2002/04/25 14:50:36  kallnik
 * changed double/float to double
 * added several #include GTMath
 *
 * PLEASE use double
 *
 * Revision 1.18  2002/04/08 21:33:20  roefer
 * Paderborn release of self location
 *
 * Revision 1.17  2002/04/02 13:10:20  dueffert
 * big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
 *
 * Revision 1.16  2002/03/30 00:19:39  roefer
 * Adaptation to new goal shape
 *
 * Revision 1.15  2002/03/23 22:25:45  roefer
 * LandmarksPercepts now in player coordinates
 *
 * Revision 1.14  2002/02/28 23:17:42  roefer
 * Calculated poses are only accepted if inside the field
 *
 * Revision 1.13  2002/02/27 07:02:01  roefer
 * Collaboration between SelfLocator and HeadControl improved
 *
 * Revision 1.12  2002/02/25 23:30:04  roefer
 * Crash on robot hopefully fixed
 *
 * Revision 1.11  2002/02/25 19:14:58  loetzsch
 * einen rgerlichen Absturz auskommentiert
 *
 * Revision 1.10  2002/02/22 08:43:24  roefer
 * Intelligent sensor resetting added
 *
 * Revision 1.9  2002/02/16 16:24:28  roefer
 * SensorData interpolation corrected
 *
 * Revision 1.8  2002/02/13 22:43:02  roefer
 * First working versions of DefaultLandmarksPerceptor and MonteCarloSelfLocator
 *
 * Revision 1.7  2002/02/12 09:45:09  roefer
 * Progress in DefaultLandmarksPerceptor and MonteCarloSelfLocator
 *
 * Revision 1.6  2002/02/07 16:27:51  loetzsch
 * delete [] samples added at the end of execute. removed in destructor
 *
 * Revision 1.5  2002/02/06 10:30:11  AndySHB
 * MonteCarloLocalization First Draft
 *
 * Revision 1.3  2001/12/13 14:14:38  risler
 * OdometryData derived from Pose2D
 *
 * Revision 1.2  2001/12/10 17:47:07  risler
 * change log added
 *
 */
