/**
* @file Modules/SelfLocator/LinesSelfLocator.h
* 
* This file contains a class for self-localization based on the Monte Carlo approach using field lines.
*
* @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
*/

#ifndef __LinesSelfLocator_h_
#define __LinesSelfLocator_h_

#include "SelfLocator.h"
#include "LinesTables.h"

/**
* The class implements a lines-based Monte Carlo self-localization.
*/
class LinesSelfLocator : public SelfLocator, public LinesTables
{
  private:
    /**
    * The class represents a sample.
    */
    class Sample : public PoseSample
    {
      public:
        double quality[LinesPercept::numberOfLineTypes - 1]; /**< The quality of the sample, i.e. a lowpass filtered probability. */
        Sample* next; /**< The next sample in the cell cube. */
    
        /**
        * Constructor.
        */
        Sample();

        /**
        * Constructor.
        * @param pose The pose of the sample.
        * @param quality The quality of the sample.
        */
        Sample(const Pose2D& pose,const double* quality);

        double getQuality() const {return probability;}

        void updateQuality(const double* average);

        void setProbability(LinesPercept::LineType type,double value);

        bool isValid() const {return probability != 2;}
    };
  
    /**
    * The class represents a cell in a cube that is used to determine the largest sample cluster.
    */
    class Cell
    {
    public:
      int count; /**< The number of samples in this cube. */
      Sample* first; /**< The first sample in this cube. */
    
      /**

      * Constructor.
      */
      Cell() {count = 0; first = 0;}
    };


    enum
    {
      SAMPLES_MAX = 100, /**< The number of samples. */
      GRID_MAX = 10 /**< The number of cells in one dimension. */
    };
  
    SampleSet<Sample, SAMPLES_MAX> sampleSet; /**< The sample set. */
    Pose2D lastOdometry, /**< The state of the odometry at the previous call of this module. */
           lastOdometry2;
    LinesPercept::LineType types[LinesPercept::numberOfLineTypes];
    int numberOfTypes;
    bool sensorUpdated; /**< Did any update of the samples by a sensor reading happen? */
    double average[LinesPercept::numberOfLineTypes - 1];
    unsigned timeStamp;
    double speed;

    /**
    * The function distributes the parameter in a Gaussian way.
    * @param d A value that should be distributed.
    * @return A transformation of d according to a Gaussian curve.
    */
    double sigmoid(double d) const {return exp(- d * d);}
  
    /** 
    * The function updates the samples by the odometry offset.
    * @param odometry The motion since the last call to this function.
    * @param noise Dermines whether some additional noise is added to the sample poses.
    */
    void updateByOdometry(const Pose2D& odometry,bool noise);
  
    /** 
    * The function updates the samples by a single line point recognized.
    * @param point The relative offset of the point.
    * @param type The type of the point.
    */
    void updateByPoint(const Vector2<int>& point,LinesPercept::LineType type);
  
    /**
    * The function re-distributes the samples according to their probabilities.
    */
    void resample();
  
    /**
    * The function determines the most probable pose from the sample distribution.
    * @param pose The pose is returned to this variable.
    * @param validity The validity of the pose is returned to this variable.
    */
    void calcPose(Pose2D& pose,double& validity);
  
    /**
    * The function creates a random sample inside the field boundaries.
    * @return The random sample.
    */
    Sample getTemplate() const;
  
    /**
    * The function draws an arrow to a debug drawing.
    * @param pose The position and direction of the arrow.
    * @param color The color of the arrow.
    */
    void draw(const Pose2D& pose,Drawings::Color color) const;
  
    /**
    * The function draws a point of a line percept.
    * @param point The relative position in field coordinates.
    * @param type The line type of the point.
    */
    void draw(const Vector2<int>& point,LinesPercept::LineType type) const;

  public:
    /** 
    * Constructor.
    * @param interfaces The paramters of the SelfLocator module.
    */
    LinesSelfLocator(const SelfLocatorInterfaces& interfaces);
    
    /**
     * The function executes the module.
     */
    virtual void execute();
    virtual bool handleMessage(InMessage& message);

    static double paramUp, // step size for increasing qualities
                  paramDown, // step size for decreasing qualities
                  paramDelay, // factor
                  paramHeight, // hypothetic height of head
                  paramZ, // sharpness of Gauss for rotational errors 
                  paramY, // sharpness of Gauss for translational errors
                  paramTrans, // maximum translational shift if quality is bad
                  paramRot; // maximum rotational shift if quality is bad
};

#endif// __LinesSelfLocator_h_

/*
* Change log :
* 
* $Log: LinesSelfLocator.h,v $
* Revision 1.1.1.1  2004/05/22 17:20:45  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/03/08 02:11:47  roefer
* Interfaces should be const
*
* Revision 1.2  2004/03/08 01:09:33  roefer
* Use of LinesTables restructured
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.20  2003/05/24 21:09:10  roefer
* Sensor resetting improved
*
* Revision 1.19  2003/05/22 07:53:05  roefer
* GT2003SelfLocator added
*
* Revision 1.18  2003/05/08 23:52:24  roefer
* SampleSet and SampleSetProxy added
*
* Revision 1.17  2003/04/27 08:14:02  roefer
* static vars -> static const vars
*
* Revision 1.16  2003/04/16 07:00:16  roefer
* Bremen GO checkin
*
* Revision 1.17  2003/04/11 07:20:54  roefer
* Bug fixed!!
*
* Revision 1.16  2003/04/10 21:38:35  roefer
* Improvements in self localization
*
* Revision 1.15  2003/03/31 21:01:44  roefer
* Moved class Field to Tools
*
* Revision 1.14  2003/03/22 11:07:02  roefer
* Only goals are used to generate candidate positions
*
* Revision 1.13  2003/02/21 22:20:12  roefer
* LinesSelfLocator is working
*
* Revision 1.12  2003/02/19 15:01:23  roefer
* LinesPerceptor2 added
*
* Revision 1.11  2003/01/15 13:47:28  roefer
* Progress in LinesSelfLocator, new debug drawing
*
* Revision 1.10  2002/12/15 18:15:13  roefer
* Using less points per cycle
*
* Revision 1.9  2002/12/14 21:21:42  roefer
* Drawing simplified
*
* Revision 1.8  2002/12/13 11:25:19  dueffert
* doxygen bugs fixed
*
* Revision 1.7  2002/12/12 22:09:41  roefer
* Progress in LinesSelfLocator
*
* Revision 1.6  2002/12/04 12:20:37  juengel
* Changed the data type of the points in the LinesPercept from Vector2<double> to Vector2<int>.
*
* Revision 1.5  2002/11/19 17:38:31  dueffert
* doxygen bugs corrected
*
* Revision 1.4  2002/11/12 10:49:02  juengel
* New debug drawing macros - second step
* -moved /Tools/Debugging/PaintMethods.h and . cpp
*  to /Visualization/DrawingMethods.h and .cpp
* -moved DebugDrawing.h and .cpp from /Tools/Debugging/
*  to /Visualization
*
* Revision 1.3  2002/09/22 18:40:53  risler
* added new math functions, removed GTMath library
*
* Revision 1.2  2002/09/12 12:24:09  juengel
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:15  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.7  2002/08/30 14:34:06  dueffert
* removed unused includes
*
* Revision 1.6  2002/07/23 13:33:44  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.5  2002/06/20 00:40:20  Thomas Rfer
* WLan crash removed
*
* Revision 1.4  2002/06/12 11:34:53  roefer
* Improvements in LinesSelfLocator
*
* Revision 1.3  2002/06/06 17:02:03  roefer
* LinesSelfLocator, first approach
*
* Revision 1.2  2002/06/02 23:21:09  roefer
* Single color table and progress in LinesSelfLocator
*
* Revision 1.1.1.1  2002/05/10 12:40:15  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.3  2002/04/03 17:27:45  roefer
* Reduced memory usage
*
* Revision 1.2  2002/04/02 13:10:20  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.1  2002/04/01 08:28:32  roefer
* Dummy LinesSelfLocator added
*
*
*/
