/**
 * @file DefaultObstaclesLocator.h
 * 
 * This file contains a class for obstacle localization.
 * @author Jan Hoffmann
 * @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
 */

#ifndef __DefaultObstaclesLocator_h_
#define __DefaultObstaclesLocator_h_

#include "ObstaclesLocator.h"
#include "Tools/Field.h"

/**
 * This class is a locator for obstacles.
 */
class DefaultObstaclesLocator : public ObstaclesLocator
{
public:

  enum {
    timeAfterWhichObstacleIsForgotten = 6000
  };

  /** 
  * Constructor.
  * @param interfaces The paramters of the ObstaclesLocator module.
  */
  DefaultObstaclesLocator(const ObstaclesLocatorInterfaces& interfaces);

  int relativeOrientation;

  Field field;
  Vector2<double> obstacles[ObstaclesModel::numOfSectors];
  ObstaclesPercept::ObstacleType obstacleTypes[ObstaclesModel::numOfSectors];

  unsigned long timestamps[ObstaclesModel::numOfSectors];
  
  double sectorWidth;
  OdometryData lastOdometry;
  Pose2D odometry;

  /** head tilt angle below which psd percepts are ignored */
  double headTiltClipPSD;

  /** Executes the module */
  virtual void execute();

  bool usePSD;
  bool useLinesPercept;
  bool useObstaclesPercept;
  bool useAging;

  void setObstaclesModel(bool addWorldModel = true);
  void moveObstaclesByOdometry();

  void addLinesPercept();
  void addObstaclesPercept();
  void addPSDPercept();
  enum UpdateMode{overwrite, extend, limit};
  void addObstaclePoint(const Vector2<double>& obstaclePoint, UpdateMode mode);
  void addObstaclePoints(
    const Vector2<int>& nearPoint, 
    const Vector2<int>& farPoint, 
    bool farPointIsOnImageBorder,
    ObstaclesPercept::ObstacleType obstacleType
    );

  virtual bool handleMessage(InMessage& message);

private:
  void determineFreePartsOfGoals();
  
  void determineNextFreeTeammate();

  /** Last time a free part of a goal was detected*/
  unsigned long lastTimeFreePartOfGoalWasDetermined[2];

  /** angle to a free part of the goal, updated by vision and turn(!) odometry*/
  double angleToFreePartOfGoal[2];

  /** the width of the free part of goal*/
  double widthOfFreePartOfGoal[2];

  /** the distance to the free part of the goal*/
  double distanceToFreePartOfGoal[2];

};

#endif// __DefaultObstaclesLocator_h_

/*
 * Change log :
 * 
 * $Log: DefaultObstaclesLocator.h,v $
 * Revision 1.1.1.1  2004/05/22 17:20:38  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.4  2004/03/16 14:00:21  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.3  2004/03/08 01:39:03  roefer
 * Interfaces should be const
 *
 * - added ATH2004HeadControl
 * - added ATH2004LEDControl
 * - headmotiontester shows "tilt2"
 * - motion process updates odometry while no new robotPose is received, added to motion request
 * - some ui adjustments
 * - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
 *
 *
 * Revision 1.2  2004/03/01 11:42:48  juengel
 * Different types are handled.
 *
 * Revision 1.1  2003/10/06 14:10:14  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.9  2003/06/25 18:45:10  juengel
 * Added nearPoints, farPoints[maxNumberOfPoints] and farPointIsOnImageBorder to ObstaclesPercept.
 *
 * Revision 1.8  2003/06/17 14:47:11  juengel
 * Added update mode.
 *
 * Revision 1.7  2003/06/13 14:52:12  juengel
 * Added addObstaclesPercept.
 *
 * Revision 1.6  2003/05/26 08:23:32  juengel
 * Added determineFreePartsOfGoals() and determineNextFreeTeammate();
 *
 * Revision 1.5  2003/05/21 11:44:02  risler
 * added aging
 * some optimizations
 *
 * Revision 1.4  2003/05/14 13:08:38  risler
 * removed DefaultObstaclesLocator
 * renamed MicroSectorsObstaclesLocator to DefaultObstaclesLocator
 * ObstaclesModel contains increased number of sectors
 * DefaultObstaclesLocator clean up
 *
 * Revision 1.5  2003/04/06 21:37:39  juengel
 * Added some methods.
 *
 * Revision 1.4  2003/04/06 17:14:37  jhoffman
 * added headTilt to PSDpercept and added debugging capabilities to microsectors
 *
 * Revision 1.3  2003/04/06 12:14:58  jhoffman
 * bug removed and code consolidated
 *
 * Revision 1.2  2003/04/05 12:46:11  jhoffman
 * worked on obstacle model
 *
 * Revision 1.1  2003/04/04 09:03:18  jhoffman
 * no message
 *
 *
 *
 */
