/**
* @file DefaultObstaclesLocator.cpp
*
* This file contains a class for obstacle localization.
* @author Jan Hoffmann
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#include "DefaultObstaclesLocator.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Math/Geometry.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Math/Matrix2x2.h"
#include "Tools/Debugging/GenericDebugData.h"
#include "Platform/SystemCall.h"

DefaultObstaclesLocator::DefaultObstaclesLocator(const ObstaclesLocatorInterfaces& interfaces)
: ObstaclesLocator(interfaces), headTiltClipPSD(-.3)
{
  usePSD = true;
  useLinesPercept = false;
  useObstaclesPercept = true;
  useAging = true;
  
  sectorWidth = pi2/(double)ObstaclesModel::numOfSectors;

  for(int i = 0; i < ObstaclesModel::numOfSectors; i++)
  {
    obstacles[i].x = 0;
    obstacles[i].y = 0;
    obstacleTypes[i] = ObstaclesPercept::unknown;
  }
}


void DefaultObstaclesLocator::execute()
{
  odometry = odometryData - lastOdometry;
  lastOdometry = odometryData;

  moveObstaclesByOdometry();

//  if(usePSD) addPSDPercept();
  if(useObstaclesPercept) addObstaclesPercept();
  if(useLinesPercept) addLinesPercept();

  setObstaclesModel(true);
  obstaclesModel.setFrameNumber(obstaclesPercept.frameNumber);


  determineFreePartsOfGoals();
  determineNextFreeTeammate();
  DEBUG_DRAWING_FINISHED(sketch);
}

void DefaultObstaclesLocator::moveObstaclesByOdometry()
{
  // move representatives by odometry
/*  if (odometry.translation.x == 0 &&
      odometry.translation.y == 0 &&
      odometry.rotation == 0) return;
*/

  Vector2<double> c1(cos(odometry.rotation),-sin(odometry.rotation));
  Vector2<double> c2(sin(odometry.rotation),cos(odometry.rotation));
  Matrix2x2<double> R(c1, c2);

  Vector2<double> movedObstacles[ObstaclesModel::numOfSectors];
  unsigned long movedTimestamps[ObstaclesModel::numOfSectors];
  ObstaclesPercept::ObstacleType movedObstacleTypes[ObstaclesModel::numOfSectors];
  int numOfObstacles = 0;

  int i;  

  for(i = 0; i < ObstaclesModel::numOfSectors; i++)
  {
    if (
      (obstacles[i].x != 0 || obstacles[i].y != 0) &&
      (!useAging ||
       SystemCall::getTimeSince(timestamps[i]) < timeAfterWhichObstacleIsForgotten)
       )
    {
      movedObstacles[numOfObstacles] = R*obstacles[i];
      movedObstacles[numOfObstacles] -= odometry.translation;
      movedObstacleTypes[numOfObstacles] = obstacleTypes[i];
      movedTimestamps[numOfObstacles] = timestamps[i];
      numOfObstacles++;
    }
    obstacles[i].x = obstacles[i].y = 0;
    obstacleTypes[i] = ObstaclesPercept::unknown;
  }

  for(i = 0; i < numOfObstacles; i++)
  {
    int targetSector = ObstaclesModel::getSectorFromAngle(movedObstacles[i].angle());
    double sqrDistance = movedObstacles[i]*movedObstacles[i];

    if
      (
        sqrDistance >= 10.0 &&
//        sqrDistance < ObstaclesModel::maxDistance*ObstaclesModel::maxDistance &&
        (
         (obstacles[targetSector].x == 0 && obstacles[targetSector].y == 0) || 
         sqrDistance <= obstacles[targetSector]*obstacles[targetSector]
        )
      )
    {
      obstacles[targetSector] = movedObstacles[i];
      obstacleTypes[targetSector] = movedObstacleTypes[i];
      timestamps[targetSector] = movedTimestamps[i];
    }
  }
}

void DefaultObstaclesLocator::setObstaclesModel(bool addWorldModel)
{
  int i;
  double distance;

  ObstaclesModel buffer;

  // set sector distances
  for(i = 0; i < ObstaclesModel::numOfSectors; i++)
  {
    distance = obstacles[i].abs();

    // if the player is the goalie, don't avoid own penalty area!
    if (getPlayer().getPlayerNumber() != Player::one)
    //if (addWorldModel)
    {
      if (FieldDimensions::distanceToOwnPenaltyArea(robotPose) > 0)
      {
        double distToPenaltyArea = field.getDistanceToOwnPenaltyArea(robotPose + Pose2D(ObstaclesModel::getAngleOfSector(i)));
        if (distToPenaltyArea > 50 && distToPenaltyArea < ObstaclesModel::maxDistance)
          if ((distToPenaltyArea < distance) || (distance == 0))
            distance = distToPenaltyArea;
      }
    }

    if(distance > 0)
    {
      buffer.distance[i] = (int)distance;
      buffer.obstacleType[i] = obstacleTypes[i];
    }
    else
    {
      buffer.distance[i] = ObstaclesModel::maxDistance;
      buffer.obstacleType[i] = ObstaclesPercept::unknown;
    }
  }



  // remove near peaks to make it more robust against 
  // noise!
  for(i = 0; i < ObstaclesModel::numOfSectors; i++)
  {
    // if the two neighbouring sectors have a distance
    // that is 200 mm greater than the current one, 
    // perform smoothing:
    if ((buffer.distance[i] - buffer.distance[(i+1)%ObstaclesModel::numOfSectors] < -200) &&
        (buffer.distance[i] - buffer.distance[(i+ObstaclesModel::numOfSectors-1)%ObstaclesModel::numOfSectors]) < -200)
    {
      obstaclesModel.distance[i] = (buffer.distance[(i+1)%ObstaclesModel::numOfSectors] +
        buffer.distance[(i-1)%ObstaclesModel::numOfSectors])/2;
    }
    else
    {
      obstaclesModel.distance[i] = buffer.distance[i];
    }
    //obstaclesModel.distance[i] = buffer.distance[i];
    obstaclesModel.obstacleType[i] = buffer.obstacleType[i];
  }

  // calculate corridor

  /* the following is pretty useless !! - thinks JH 

  
 
  */
  double closest = ObstaclesModel::maxDistance; 
  for(i = ObstaclesModel::getSectorFromAngle(-pi_2); i < ObstaclesModel::getSectorFromAngle(pi_2); i++)
  {
    if ((obstacles[i].x < closest) && (obstacles[i].x > 10)
      && (obstacles[i].y < 90) && (obstacles[i].y > -90))
      closest = obstacles[i].x;
  }
  obstaclesModel.corridorInFront = (int )closest;

  obstaclesModel.bodyPSD = psdPercept[psdPercept.numOfPercepts - 1].body;
}

void DefaultObstaclesLocator::addPSDPercept()
{
  for (int i = 0; i < psdPercept.numOfPercepts; i++)
  {
    if (psdPercept[i].neckTilt > headTiltClipPSD && psdPercept[i].isValid)
    {
      // latest measurement
      Vector2<double> psdPoint(psdPercept[i].x, psdPercept[i].y);
      int sector = ObstaclesModel::getSectorFromAngle(psdPoint.angle());
      
       // point is on the ground  
      if(psdPercept[i].z < 90)
      {    
        double sqrDistance = obstacles[sector] * obstacles[sector];
        // only update if the point seen is farther away
        if (sqrDistance > 0 &&
            sqrDistance < psdPoint * psdPoint)
        {
          obstacles[sector].x = psdPoint.x;
          obstacles[sector].y = psdPoint.y;
          timestamps[sector] = SystemCall::getCurrentSystemTime();
        }
      }
      // point is an obstacle
      else if (!psdPercept[i].tooFarAway) 
      {
        double angleToPoint = normalize(psdPoint.angle());
        double angleToBall = Geometry::angleTo(robotPose.getPose(), ballModel.seen);

        if(toDegrees(normalize(angleToPoint - angleToBall)) < 10 &&
          toDegrees(normalize(angleToPoint - angleToBall)) > -10 && 
          SystemCall::getTimeSince(ballModel.seen.timeWhenLastSeen) < 1000)
        {
          DOT(sketch, 20, 10, Drawings::green, Drawings::green);
        }
        else
        {
          obstacles[sector].x = psdPoint.x;
          obstacles[sector].y = psdPoint.y;
          timestamps[sector] = SystemCall::getCurrentSystemTime();
        }
      }
      // if nothing in this direction, perform reset
      else 
      {
        obstacles[sector].x = 0;
        obstacles[sector].y = 0;
      }
    }
  }
}

void DefaultObstaclesLocator::addLinesPercept()
{
  int i;
  for(i = 0; i < linesPercept.numberOfPoints[LinesPercept::redRobot]; i++)
  {
    Vector2<double> obstaclePoint(linesPercept.points[LinesPercept::redRobot][i].x, linesPercept.points[LinesPercept::redRobot][i].y);
    addObstaclePoint(obstaclePoint, limit);
  }
  for(i = 0; i < linesPercept.numberOfPoints[LinesPercept::blueRobot]; i++)
  {
    Vector2<double> obstaclePoint(linesPercept.points[LinesPercept::blueRobot][i].x, linesPercept.points[LinesPercept::blueRobot][i].y);
    addObstaclePoint(obstaclePoint, limit);
  }
  for(i = 0; i < linesPercept.numberOfPoints[LinesPercept::border]; i++)
  {
    Vector2<double> obstaclePoint(linesPercept.points[LinesPercept::border][i].x, linesPercept.points[LinesPercept::border][i].y);
    addObstaclePoint(obstaclePoint, extend);
  }
}

void DefaultObstaclesLocator::addObstaclesPercept()
{
  int i;
  //for(i = 1; i < obstaclesPercept.numberOfPoints - 1; i++)
  for(i = 0; i < obstaclesPercept.numberOfPoints ; i++)
  {
//    Vector2<double> obstaclePoint(obstaclesPercept.farPoints[i].x, obstaclesPercept.farPoints[i].y);
//    addObstaclePoint(obstaclePoint, limit);
  
//    if(
//      Geometry::distance(
//      obstaclesPercept.farPoints[i], 
//      obstaclesPercept.farPoints[i-1]) < 300 
//      &&
//      Geometry::distance(
//      obstaclesPercept.farPoints[i], 
//      obstaclesPercept.farPoints[i+1]) < 300 
//      )
    addObstaclePoints(
      obstaclesPercept.nearPoints[i], 
      obstaclesPercept.farPoints[i], 
      obstaclesPercept.farPointIsOnImageBorder[i],
      obstaclesPercept.obstacleType[i]);
  }
}

void DefaultObstaclesLocator::addObstaclePoint
(
 const Vector2<double>& obstaclePoint,
 UpdateMode mode
 )
{
  double angleToPoint = normalize(obstaclePoint.angle());

  //Vision based obstacles percept does not contain balls.
/*
  double angleToBall = Geometry::angleTo(robotPose.getPose(), ballModel.seen);
  if(toDegrees(normalize(angleToPoint - angleToBall)) < 10 &&
    toDegrees(normalize(angleToPoint - angleToBall)) > -10 && 
    SystemCall::getTimeSince(ballModel.seen.timeWhenLastSeen) < 1000)
  {
    DOT(sketch, 10, 10, Drawings::red, Drawings::red);
    return;
  }
*/ 

  int sector = ObstaclesModel::getSectorFromAngle(angleToPoint);

  if ( sector >= ObstaclesModel::numOfSectors || ObstaclesModel::numOfSectors < 0 ) 
  {
    sector = 0;
    return;

    OUTPUT ( idText, text, "sector >= ObstaclesModel::numOfSectors" );
  }
  double sqrDistanceInModel = obstacles[sector] * obstacles[sector];
  double sqrDistanceToPoint = obstaclePoint * obstaclePoint;

  if(sqrt(sqrDistanceToPoint) < 150)
  {
    DOT(sketch, 30, 10, Drawings::blue, Drawings::blue);
    return;
  }

  switch(mode)
  {
  case overwrite:
      obstacles[sector].x = obstaclePoint.x;
      obstacles[sector].y = obstaclePoint.y;
      timestamps[sector] = SystemCall::getCurrentSystemTime();
    break;
  case extend:
    // only update if the new point seen is farther away than the old one
    if(sqrDistanceInModel < sqrDistanceToPoint)
    {
      obstacles[sector].x = obstaclePoint.x;
      obstacles[sector].y = obstaclePoint.y;
      timestamps[sector] = SystemCall::getCurrentSystemTime();
    }
    break;
 case limit:
    // only update if the new point is closer than the old one
    if(
      (obstacles[sector].x == 0 && obstacles[sector].y == 0) ||
       sqrDistanceToPoint < sqrDistanceInModel)
    {
      obstacles[sector].x = obstaclePoint.x;
      obstacles[sector].y = obstaclePoint.y;
      timestamps[sector] = SystemCall::getCurrentSystemTime();
    }
    break;
  }
}

void DefaultObstaclesLocator::addObstaclePoints
(
 const Vector2<int>& nearPoint,
 const Vector2<int>& farPoint,
 bool farPointIsOnImageBorder,
 ObstaclesPercept::ObstacleType obstacleType
 )
{
  double angleToPoints = normalize(farPoint.angle());
  int sector = ObstaclesModel::getSectorFromAngle(angleToPoints);

  if ( sector >= ObstaclesModel::numOfSectors || ObstaclesModel::numOfSectors < 0 ) 
  {
    sector = 0;
    return;
    OUTPUT ( idText, text, "sector >= ObstaclesModel::numOfSectors" );
  }
  double sqrDistanceInModel = obstacles[sector] * obstacles[sector];
  double sqrDistanceToNearPoint = nearPoint * nearPoint;
  double sqrDistanceToFarPoint = farPoint * farPoint;

  if(sqrDistanceToNearPoint > sqrDistanceInModel && sqrDistanceInModel != 0) return;
  else if(!farPointIsOnImageBorder)
  {
    obstacles[sector].x = farPoint.x;
    obstacles[sector].y = farPoint.y;
    if(obstacleType != ObstaclesPercept::unknown || (fabs(sqrDistanceToFarPoint - sqrDistanceInModel) > 500)) 
      obstacleTypes[sector] = obstacleType;
    timestamps[sector] = SystemCall::getCurrentSystemTime();
  }
  else if(sqrDistanceInModel < sqrDistanceToFarPoint && sqrDistanceInModel != 0)
  {
    obstacles[sector].x = farPoint.x;
    obstacles[sector].y = farPoint.y;
    timestamps[sector] = SystemCall::getCurrentSystemTime();
  }
}


void DefaultObstaclesLocator::determineFreePartsOfGoals()
{
  for(int i = 0; i < 2; i++)
  {
    if (obstaclesPercept.angleToFreePartOfGoalWasDetermined[i])
    {
      lastTimeFreePartOfGoalWasDetermined[i] = SystemCall::getCurrentSystemTime();
      if(
        obstaclesPercept.widthOfFreePartOfGoal[i] > widthOfFreePartOfGoal[i] 
        || 
        (
        angleToFreePartOfGoal[i] - widthOfFreePartOfGoal[i] / 2.0 <
        obstaclesPercept.angleToFreePartOfGoal[i]
        &&
        obstaclesPercept.angleToFreePartOfGoal[i] <
        angleToFreePartOfGoal[i] + widthOfFreePartOfGoal[i] / 2.0
        )
        )
      {
        angleToFreePartOfGoal[i] = obstaclesPercept.angleToFreePartOfGoal[i];
        widthOfFreePartOfGoal[i] = obstaclesPercept.widthOfFreePartOfGoal[i];
        distanceToFreePartOfGoal[i] = obstaclesPercept.distanceToFreePartOfGoal[i];

        obstaclesModel.angleToFreePartOfGoalWasDetermined[i] = true;
      }
    }
    else if(SystemCall::getTimeSince(lastTimeFreePartOfGoalWasDetermined[i]) < 2000)
    {
      angleToFreePartOfGoal[i] -= odometry.getAngle();
      obstaclesModel.angleToFreePartOfGoalWasDetermined[i] = true;
    }
    else
    {
      angleToFreePartOfGoal[i] = 0.0;
      widthOfFreePartOfGoal[i] = 0.0;
      distanceToFreePartOfGoal[i] = 1000.0;

      obstaclesModel.angleToFreePartOfGoalWasDetermined[i] = false;
    }
    obstaclesModel.angleToFreePartOfGoal[i] = angleToFreePartOfGoal[i];
    obstaclesModel.widthOfFreePartOfGoal[i] = widthOfFreePartOfGoal[i];
    obstaclesModel.distanceToFreePartOfGoal[i] = distanceToFreePartOfGoal[i];
  }
}

void DefaultObstaclesLocator::determineNextFreeTeammate()
{
  int distanceToNextTeammate = 10000;
  double angleToNextTeammate = 0;

  bool foundTeammate = false;

  for(int i = 0; i < playerPoseCollection.numberOfOwnPlayers; i++)
  {
    PlayerPose currentPose = playerPoseCollection.getOwnPlayerPose(i);
    int currentDistance = (int)((currentPose.getPose().translation - robotPose.translation).abs());
    double currentAngle = normalize(Geometry::angleTo(robotPose, currentPose.getPose().translation));

    int targetSector = ObstaclesModel::getSectorFromAngle(currentAngle);
    int distanceOfTargetSector = (int)obstacles[targetSector].abs();

    if(currentDistance < distanceToNextTeammate && 
      (
      distanceOfTargetSector == 0 ||
      distanceOfTargetSector + 200 > distanceToNextTeammate
      )
      )
    {
      foundTeammate = true;
      angleToNextTeammate = currentAngle;
      distanceToNextTeammate = currentDistance;
    }
  }
  if(foundTeammate)
  {
    obstaclesModel.angleToNextFreeTeammateWasDetermined = true;
    obstaclesModel.angleToNextFreeTeammate = angleToNextTeammate;
    obstaclesModel.distanceToNextFreeTeammate = distanceToNextTeammate;
  }
  else
  {
    obstaclesModel.angleToNextFreeTeammateWasDetermined = false;
  }
}


bool DefaultObstaclesLocator::handleMessage(InMessage& message)
{
  bool handled = false;

  switch(message.getMessageID())
  {
  case idGenericDebugData:
    GenericDebugData d;
    message.bin >> d;
    if(d.id == GenericDebugData::defaultObstaclesLocator)
    {
      OUTPUT(idText,text,"generic debug message handled by module DefaultObstaclesLocator");

      headTiltClipPSD = d.data[0];
      switch((int)d.data[1])
      {
      case 0: usePSD = true; useLinesPercept = false; useAging = false;
        break;
      case 1: usePSD = false; useLinesPercept = true; useAging = false;
        break;
      case 2: usePSD = false; useLinesPercept = true; useAging = true;
        break;
      case 3: usePSD = true; useLinesPercept = true; useAging = false;
        break;
      case 4: usePSD = true; useLinesPercept = true; useAging = true;
        break;
      };
    }
    handled = true;
    break;

  }
  return handled;
}
 
/*
* Change log :
* 
* $Log: DefaultObstaclesLocator.cpp,v $
* Revision 1.6  2004/06/17 11:12:30  risler
* penalty area only in obstacle model if distance smaller than maxDistance
*
* Revision 1.5  2004/06/16 18:17:48  risler
* change obstacle type to unknown if distance to obstacle is different
* dont change obstacle type if obstacle was not seen
*
* Revision 1.4  2004/06/16 18:12:53  risler
* change obstacle type to unknown if distance to obstacle is different
* dont change obstacle type if obstacle was not seen
*
* Revision 1.3  2004/05/27 17:13:37  jhoffman
* - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
* - clipping included for setJoints
* - removed some microrad/rad-bugs
* - bodyPosture constructor and "=" operator fixed
*
* Revision 1.2  2004/05/22 22:52:02  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.1.1.1  2004/05/22 17:20:38  cvsadm
* created new repository GT2004_WM
*
* Revision 1.13  2004/04/07 12:28:59  risler
* ddd checkin after go04 - first part
*
* Revision 1.3  2004/04/02 00:32:37  dthomas
* fix bug in obstacle model
*
* Revision 1.2  2004/04/02 00:18:46  dthomas
* fix bug with goalie and penalty-area
*
* Revision 1.1.1.1  2004/03/29 08:28:47  Administrator
* initial transfer from tamara
*
* Revision 1.12  2004/04/05 17:56:47  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.2  2004/04/01 21:06:30  hoffmann
* bug fixed thanks to thomas
*
* Revision 1.1.1.1  2004/03/31 11:16:47  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.11  2004/03/29 15:04:07  juengel
* Bug fixed.
*
* Revision 1.10  2004/03/25 09:13:34  jhoffman
* - obstacles percepts: all percepts are put into the model (again! no filtering)
* - filtering when model is created: single "peaks" are removed if neighboring sectors are further away
*
* Revision 1.9  2004/03/17 10:28:37  jhoffman
* uncommented matthias' code to make it more robust against bad obstacle percepts (false positives)
*
* Revision 1.8  2004/03/16 14:00:21  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.7  2004/03/10 14:16:33  risler
* body psd value added to PSDPercept and ObstaclesModel
* - added ATH2004HeadControl
* - added ATH2004LEDControl
* - headmotiontester shows "tilt2"
* - motion process updates odometry while no new robotPose is received, added to motion request
* - some ui adjustments
* - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
*
* Revision 1.6  2004/03/08 01:39:03  roefer
* Interfaces should be const
*
* Revision 1.5  2004/03/01 11:42:47  juengel
* Different types are handled.
*
* Revision 1.4  2004/02/28 14:08:23  juengel
* Added ObstacleType to ObstaclesModel.
*
* Revision 1.3  2003/11/14 19:02:26  goehring
* frameNumber added
*
* Revision 1.2  2003/11/10 11:35:24  juengel
* Added distance check for to neighbouring points.
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 11:38:52  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.28  2003/06/25 18:45:10  juengel
* Added nearPoints, farPoints[maxNumberOfPoints] and farPointIsOnImageBorder to ObstaclesPercept.
*
* Revision 1.27  2003/06/21 12:53:18  juengel
* Changed maximal ground height to 90 mm, psdPercept.isValid is used.
*
* Revision 1.26  2003/06/19 12:09:45  juengel
* No max distance for odometry update.
*
* Revision 1.25  2003/06/18 13:58:25  juengel
* Added ballP_osition to ObstaclesLocatorInterfaces.
* Ball is not an obstacle.
*
* Revision 1.24  2003/06/17 18:26:54  thomas
* modified: auskommentiert von PropagatedBallP_osition::getPropagatedSpeed und range abgefangen in DefaultObstaclesLocator::addObstaclePoint
*
* Revision 1.23  2003/06/17 14:47:11  juengel
* Added update mode.
*
* Revision 1.22  2003/06/13 14:51:56  juengel
* Added addObstaclesPercept.
*
* Revision 1.21  2003/06/12 16:52:37  juengel
* Activated usage of vision.
*
* Revision 1.20  2003/06/02 14:59:13  risler
* bugfix: crash if angle is not normalized
*
* Revision 1.19  2003/06/02 09:08:51  loetzsch
* 6 seconds free part of opponent goal modeling
* -> 2 seconds
*
* Revision 1.18  2003/05/26 08:23:32  juengel
* Added determineFreePartsOfGoals() and determineNextFreeTeammate();
*
* Revision 1.17  2003/05/22 10:06:55  risler
* odometry update optimized
*
* Revision 1.16  2003/05/21 11:44:02  risler
* added aging
* some optimizations
*
* Revision 1.15  2003/05/16 14:07:48  risler
* added radar view obstacles drawing
*
* Revision 1.14  2003/05/15 00:56:47  risler
* added functions for sector to angle mapping to ObstaclesModel
* sector to angle mapping unified
* added rule AvoidObstacles
*
* Revision 1.13  2003/05/14 19:54:42  risler
* PSDPercept contains all points from one frame
*
* Revision 1.12  2003/05/14 13:08:37  risler
* removed DefaultObstaclesLocator
* renamed MicroSectorsObstaclesLocator to DefaultObstaclesLocator
* ObstaclesModel contains increased number of sectors
* DefaultObstaclesLocator clean up
*
* Revision 1.9  2003/04/14 16:10:12  loetzsch
* ATH after GermanOpen CVS merge
*
* Revision 1.7  2003/04/10 22:28:57  Jan Hoffmann
* obstacle avoider tuning
*
* Revision 1.6  2003/04/10 15:08:14  Jan Hoffmann
* disabled vision
*
* Revision 1.5  2003/04/10 14:14:46  Jan Hoffmann
* added corridor to obstacle model
*
* Revision 1.4  2003/04/09 20:15:13  juengel
* Added Border to obstacles.
*
* Revision 1.3  2003/04/09 20:12:16  Jan Hoffmann
* no message
*
* Revision 1.2  2003/04/09 18:38:20  Jan Hoffmann
* added parameter for searchForBall to specify time before scanning starts when ball was not seen
*
* Revision 1.1.1.1  2003/04/09 14:22:29  loetzsch
* started Aibo Team Humboldt's GermanOpen CVS
*
* Revision 1.8  2003/04/06 21:37:39  juengel
* Added some methods.
*
* Revision 1.7  2003/04/06 17:14:37  jhoffman
* added headTilt to PSDpercept and added debugging capabilities to microsectors
*
* Revision 1.6  2003/04/06 16:31:55  jhoffman
* no message
*
* Revision 1.5  2003/04/06 12:14:58  jhoffman
* bug removed and code consolidated
*
* Revision 1.4  2003/04/05 17:33:08  jhoffman
* if long distances are perceived as free, the micro sector is reset
*
* Revision 1.3  2003/04/05 12:46:11  jhoffman
* worked on obstacle model
*
* Revision 1.2  2003/04/04 09:35:09  jhoffman
* added obstacle locator
*
*
*/
