/**
* @file MotionControlSelector.h
*
* Declaration of class MotionControlSelector
*
* @author Martin Ltzsch
*/

#ifndef __MotionControlSelector_h_
#define __MotionControlSelector_h_

#include "Tools/Module/ModuleSelector.h"
#include "DefaultMotionControl.h"
#include "DebugMotionControl.h"

/**
* @class MotionControlSelector
* 
* A module selector for motion control modules
*
* @author Martin Ltzsch
*/
class MotionControlSelector : public ModuleSelector, public MotionControlInterfaces
{
public:
/**
* Constructor.
* @param handler The module handler of the process.
* @param interfaces The paramters of the MotionControl module.
  */
  MotionControlSelector(ModuleHandler &handler, const MotionControlInterfaces& interfaces) 
    : ModuleSelector(SolutionRequest::motionControl),
    MotionControlInterfaces(interfaces),
    moduleHandler(handler)
  {
    handler.setModuleSelector(SolutionRequest::motionControl,this);
  }
  
  /** 
  * Is called on start and when the selected solution changes
  * to create a specific solution.
  * @param id The id of the solution to create
  * @return The created solution or 0
  */
  Module* createSolution(SolutionRequest::ModuleSolutionID id)
  {
    switch (id)
    {
    case SolutionRequest::defaultMotionControl:
      return new DefaultMotionControl(moduleHandler,*this);
    case SolutionRequest::debugMotionControl:
      return new DebugMotionControl(*this);
    default:
      return 0;
    }
  }
  
  /** 
  * A reference to the ModuleHandler of the process, needed to create 
  * the submodules of DefaultMotionControl
  */
  ModuleHandler& moduleHandler;
};  

#endif //__MotionControlSelector_h_

/*
* Change log:
*
* $Log: MotionControlSelector.h,v $
* Revision 1.1.1.1  2004/05/22 17:20:37  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/03/08 01:39:03  roefer
* Interfaces should be const
*
* Revision 1.2  2004/01/21 14:31:58  loetzsch
* Module Selectors create only the selected solution.
* When the solution changes, the old solution is erased and the new
* one ist created using createSolution(..)
*
* Revision 1.1  2003/10/06 14:10:13  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.2  2002/09/11 00:06:58  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:15  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.4  2002/08/30 13:38:00  dueffert
* removed unused includes
*
* Revision 1.3  2002/07/23 13:33:42  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.2  2002/05/14 12:39:04  dueffert
* corrected some documentation mistakes
*
* Revision 1.1.1.1  2002/05/10 12:40:15  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.5  2002/03/12 14:13:00  jhoffman
* no message
*
* Revision 1.4  2002/02/12 16:34:49  risler
* finished MofTester
*
* Revision 1.3  2002/02/08 20:00:01  risler
* added DebugMotionControl
*
* Revision 1.2  2002/02/08 17:48:57  risler
* SensorData to MotionControl
*
* Revision 1.1  2002/02/05 03:51:37  loetzsch
* added a few new module selectors
*
*/
