/**
* @file MotionControl.h
* 
* Definition of class MotionControl.
*/

#ifndef __MotionControl_h_
#define __MotionControl_h_

#include "Tools/Module/Module.h"

#include "Representations/Motion/HeadMotionRequest.h"
#include "Representations/Motion/JointDataBuffer.h"
#include "Representations/Motion/PIDData.h"
#include "Representations/Motion/OdometryData.h"
#include "Representations/Motion/MotionInfo.h"

#include "Representations/Perception/SensorDataBuffer.h"
#include "Representations/Perception/BodyPosture.h"

#include "Modules/WalkingEngine/UDParameterSet.h"

#ifdef _WIN32
#pragma warning(disable:4786) 
// the constructor with all it's long parameter type names results in a too
// long debug identifier
#endif

/**
* @class MotionControlInterfaces
*
* The interfaces of the MotionControl module.
*/
class MotionControlInterfaces
{
public:
  /** Constructor */ 
  MotionControlInterfaces(
    const unsigned long& frameNumber,
    const MotionRequest& motionRequest, 
    const HeadMotionRequest& headMotionRequest,
    const SensorDataBuffer& sensorDataBuffer,
    const BodyPosture& bodyPosture,
    const InvKinWalkingParameters& invKinWalkingParameters,
    const UDParameters& udParameters,
    const unsigned long& walkParameterTimeStamp,
    const bool& receivedNewSensorData,
    JointDataBuffer& jointDataBuffer, 
    PIDData& pidData,
    OdometryData& odometryData,
    MotionInfo& motionInfo,
    bool headIsBlockedBySpecialActionOrWalk)
    :
    frameNumber(frameNumber),
    motionRequest(motionRequest),
    headMotionRequest(headMotionRequest),
    sensorDataBuffer(sensorDataBuffer),
    bodyPosture(bodyPosture),
    invKinWalkingParameters(invKinWalkingParameters),
    udParameters(udParameters),
    walkParameterTimeStamp(walkParameterTimeStamp),
    receivedNewSensorData(receivedNewSensorData),
    jointDataBuffer(jointDataBuffer),
    pidData(pidData),
    odometryData(odometryData),
    motionInfo(motionInfo),
    headIsBlockedBySpecialActionOrWalk(headIsBlockedBySpecialActionOrWalk)
{}

protected:  
  /** A reference to the frame number */
  const unsigned long& frameNumber;

  /** A request from the behavior control */  
  const MotionRequest& motionRequest;
  
  /** Head joint values from the head control */
  const HeadMotionRequest& headMotionRequest;
  
  /** The current body sensor data */
  const SensorDataBuffer& sensorDataBuffer;
  
  /** The current body posture */
  const BodyPosture& bodyPosture;
  
  /** walking parameter sets to be used by the WalkingEngine, eg calculated by evolution behavior */
  const InvKinWalkingParameters& invKinWalkingParameters;
  const UDParameters& udParameters;
  const unsigned long& walkParameterTimeStamp;

  /** Indicates that the Motion Process received a new SensorDataBuffer */
  const bool& receivedNewSensorData;
  
  /** A buffer of joint data to be filled */
  JointDataBuffer& jointDataBuffer;

  /** The current PID servo gains */
  PIDData& pidData;
  
  /** The odometry to be incremented */
  OdometryData& odometryData;

  /** The height of the neck and the body tilt to be calculated */
  MotionInfo& motionInfo;

  /** Specifies if the head is blocked by a special action or walk.*/
  bool headIsBlockedBySpecialActionOrWalk;
};

/** 
* @class MotionControl
*
* A generic class for motion control modules.
* MotionControl is responsible for setting values for the joints of the robot 
*/
class MotionControl : public Module, public MotionControlInterfaces, public RobotDimensions
{
public:
/*
* Constructor.
* @param interfaces The paramters of the MotionControl module.
  */
  MotionControl(const MotionControlInterfaces& interfaces)
    : MotionControlInterfaces(interfaces),
      RobotDimensions(getRobotConfiguration().getRobotDimensions())
  {}
  
  /** Destructor */
  virtual ~MotionControl() {}
};

#endif //__MotionControl_h_

/*
* Change log :
* 
* $Log: MotionControl.h,v $
* Revision 1.6  2004/06/08 15:30:53  juengel
* Added frameNumber to the interfaces.
*
* Revision 1.5  2004/05/27 09:29:28  loetzsch
* removed executedMotionRequest from Interfaces
*
* Revision 1.4  2004/05/26 20:14:44  juengel
* Removed RobotVerticesBuffer.
* Added "receivedNewSensorData".
*
* Revision 1.3  2004/05/26 17:31:34  dueffert
* better data types used
*
* Revision 1.2  2004/05/26 16:10:24  dueffert
* better data types used
*
* Revision 1.1.1.1  2004/05/22 17:20:37  cvsadm
* created new repository GT2004_WM
*
* Revision 1.5  2004/03/20 09:55:25  roefer
* Preparation for improved odometry
*
* Revision 1.4  2004/03/08 01:39:03  roefer
* Interfaces should be const
*
* Revision 1.3  2004/02/16 18:02:01  dueffert
* packageCognitionMotion extended with invkin parameters
*
* Revision 1.2  2004/01/05 11:34:44  juengel
* RobotDimensions is Base of MotionControl now.
*
* Revision 1.1  2003/10/06 14:10:13  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.8  2003/05/27 16:15:52  juengel
* Added headIsBlockedBySpecialActionOrWalk.
*
* Revision 1.7  2003/05/02 18:26:18  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.6  2002/11/19 17:13:21  risler
* added datatype PIDData
* support for sending new pid values at runtime
*
* Revision 1.5  2002/09/17 23:55:21  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.4  2002/09/12 13:17:27  loetzsch
* made member variables of module interfaces protected
*
* Revision 1.3  2002/09/11 20:06:19  loetzsch
* continued experiments with modules/solutions
*
* Revision 1.2  2002/09/11 00:06:58  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:15  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/07/23 13:33:42  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.1.1.1  2002/05/10 12:40:15  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.5  2002/02/08 17:48:57  risler
* SensorData to MotionControl
*
* Revision 1.4  2001/12/21 14:09:39  roefer
* Added several destructors
*
* Revision 1.3  2001/12/10 17:47:06  risler
* change log added
*
*/
