/**
* @file SLAMBeaconDetector.h
* 
* Declaration of class SLAMBeaconDetector.
*
* @author <a href="mailto:timlaue@tzi.de">Tim Laue</a>
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/ 

#ifndef SLAMBeaconDetector_h_
#define SLAMBeaconDetector_h_

#include "Tools/Math/Vector2.h"
#include "Tools/Math/Matrix2x2.h"
#include "SLAMFlagSpecialist.h"
#include "Modules/ImageProcessor/ImageProcessorTools/ImageProcessorUtilityClasses.h"
#include "Modules/ImageProcessor/ImageProcessorTools/SUSANEdgeDetectionLite.h"
#include "Modules/ImageProcessor/ImageProcessorTools/BresenhamLineScan.h"
#include "Representations/Perception/SpecialPercept.h"

#define MAX_NUMBER_OF_PINK_RUNS 20


/**
* @class SLAMBeaconDetector
*
* A class for detecting beacons in images
*/
class SLAMBeaconDetector
{
public:
  
  enum {edgeThresholdU = 13, edgeThresholdV = 15};

  /** Constructor */ 
  SLAMBeaconDetector(const Image& image, const CameraMatrix& cameraMatrix,
                       const CameraMatrix& prevCameraMatrix,
                       const ImageInfo& imageInf,
                       const ColorTable& colorTable, 
                       const ColorCorrector& colorCorrector,
                       LandmarksPercept& landmarksPercept,
                       SpecialPercept& specialPercept);

  /** Executes the beacon detection*/
  void execute();

  /** Gains information about pink from the current color table*/
  void analyzeColorTable();

private:
  
  SpecialPercept& specialPercept;
  
  /** Edge detection operators (U and V components) used to find the boundaries of flags*/
  const SUSANEdgeDetectionLite edgeDetectionU;
  const SUSANEdgeDetectionLite edgeDetectionV;

  /** The image*/
  const Image& image;
  /** The camera matrix*/
  const CameraMatrix& cameraMatrix;
  /** The previous camera matrix*/
  const CameraMatrix& prevCameraMatrix;
  /** Additional Image information*/
  const ImageInfo& imageInf;
  /** The color table*/
  const ColorTable& colorTable;
  /** The landmarks percept*/
  LandmarksPercept& landmarksPercept;
  /** The number of possible beacons*/
  int numOfBeaconCandidates;
  /** The list of candidates*/
  Run beaconCandidates[MAX_NUMBER_OF_PINK_RUNS];
  /** A buffer for transformed candidates*/
  TransformedRun transformedCandidates[MAX_NUMBER_OF_PINK_RUNS];
  /** The base offset for horizontal scan lines*/
  const double horizontalBaseOffset;
  /** The number of horizontal scan lines above the horizon*/
  const int numOfHorizontalScanLineAbove;
  /** The number of horizontal scan lines below the horizon*/
  const int numOfHorizontalScanLineBelow;
  /** The growth factor of the distance between two scan lines*/
  const double horizontalOffsetModifier;
  /** The maximum distance between the merged runs*/
  const int clusteringDistanceTolerance;
  /** The minimum length of a pink run*/
  const int minPinkRunLength;
  /** The maximum vertical distance of scanlines to be clustered, over the 
  * estimated merged horizontal distance (half beacon aspect ratio should be 1:1) */
  const double clusteringAspectRatio;
  /** The minimum value for the ratio of the 2 dimensions of the pink part of
  * the beacon, to be used for projection on the unknown colored part*/
  const double projectionAspectRatio;
  /** The maximum number of pixels which are examined after an edge response has been found
  * for determining the color beyond the edge*/
  const int edgeScanDepth;
  /** The minimum U channel value of a pink pixel*/
  unsigned char minPinkUValue;
  /** A confidence threshold used to determine if a target beacon, whose type is
  not completely clear, can be accepted or not */
  const double minFlagConfidence;
  /** A flag specialist*/
  SLAMFlagSpecialist flagSpecialist;
  /** The color corrector*/
  const ColorCorrector& colorCorrector;

  /** Adds a pink run to the list of candidates
  * @param pinkRun A pink run
  * @return true, if the run has been appended
  */
  bool addCandidate(const Run& pinkRun);

  /** Scans along a line*/ 
  void scanForPink(const Vector2<int>& start, const Vector2<int>& end);

  /** Scans for one other beacon part*/
  bool scanForBeaconPart(const Vector2<int>& start, const Vector2<int>& end,
                         Vector2<int>& position, Vector2<int>& edge, colorClass& color); 

  /** Clusters the pink elements to find the pink beacon parts*/
  void clusterPinkBeaconParts();

  /** Tries to detect a beacon near a pink part*/
  void analyzeBeacon(const Vector2<int>& center, const double pinkRunWidth);
  
  
  /** Looks for the vertical edges and the color type of a beacon */
  int scanForBeaconEdges(const Vector2<int>& position, const double pinkRunWidth,  
        Flag::FlagType& flagType, Vector2<int>& topEdge, Vector2<int>& bottomEdge);

  enum {lowReliability = 1, mediumReliability = 4, highReliability = 6};
};

#endif //SLAMBeaconDetector_h_

/*
* $Log: SLAMBeaconDetector.h,v $
* Revision 1.1  2004/07/02 10:11:47  nistico
* Cloned main image processor and created
* SpecialLandmarks specialist for SLAM challenge
*
*
*/
