/**
* @file SLAMBallSpecialist.h
* 
* This file contains a class for Image Processing.
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>
* @author Max Risler
* @author Ronnie Brunn
* @author Michael Kunz
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/

#ifndef __SLAMBallSpecialist_h_
#define __SLAMBallSpecialist_h_

#include "Tools/Math/Geometry.h"
#include "Tools/Debugging/DebugImages.h"
#include "Representations/Perception/ColorTable.h"
#include "Representations/Perception/Image.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/BallPercept.h"
#include "Modules/ImageProcessor/ImageProcessorTools/ColorCorrector.h"
#include "Modules/ImageProcessor/ImageProcessorTools/BresenhamLineScan.h"
#include "Tools/Math/Common.h"

/**
The BallSpecialist finds a single ball in an image.
*/
class SLAMBallSpecialist  
{
public:
  SLAMBallSpecialist(const ColorCorrector& colorCorrector);
  
  /** 
  * Searches for the ball in the image, starting from the secified point. 
  */
  void searchBall
  (
    const Image& image,
    const ColorTable& colorTable,
    const CameraMatrix& cameraMatrix,
    const CameraMatrix& prevCameraMatrix,
    int x, int y,
    BallPercept& ballPercept
  );
  
  /**
  * Returns the Similarity of the given color to orange.
  */
  unsigned char getSimilarityToOrange (unsigned char y, unsigned char u, unsigned char v)
  {
/*      double r = y + 1.140 * (u - 128);
      double g = y - 0.394 * (v - 128) - 0.581 * (u - 128);
      double b = y + 2.028 * (v - 128);
      if(r < 0) r = 0; if(r > 255) r = 255;
      if(g < 0) g = 0; if(g > 255) g = 255;
      if(b < 0) b = 0; if(b > 255) b = 255;*/
    unsigned char r, g, b;
    Image::convertFromYCbCrToRGB(y, u, v, r, g, b);

//	  return (unsigned char) min(max(r-2*b-abs(2*g-r),0),255);
	  return (unsigned char) min(max(r - 2*b - 3*max((g-r/2),0) - (3*max((r/2-g),0))/2 ,0),255);
  }

private:
  /**
  * The class represents a ball point.
  */
  class BallPoint : public Vector2<int>
  {
  public:
    BallPoint() : atBorder(false), greenIsClose(true), yellowIsClose(false), hardEdge(true){};
    bool greenIsClose;
    bool yellowIsClose;
    bool hardEdge;
    bool atBorder;

    BallPoint& operator=(const Vector2<int>& other)
    {
      x = other.x;
      y = other.y;

      return *this;
    }
  };

  class BallPointList
  {
  public:
    BallPointList() : number(0) {};

    enum {maxNumberOfPoints = 400}; /**< The maximum number of ball points. */
    BallPoint ballPoints[maxNumberOfPoints]; /**< The ball points. */
    int number; /**< The actual number of ball points. */

    void add(const BallPoint& ballPoint);

    BallPoint& operator[] (int i) { return ballPoints[i]; }
    const BallPoint& operator[] (int i) const { return ballPoints[i]; }
  };

  /** Scan for the ball starting at a given trigger point */
  void scanForBallPoints
  (
   const Image& image,
   const CameraInfo& bwCameraInfo,
   const ColorTable& colorTable,
   int x, int y,
   BallPointList& ballPoints,
   int& countAmbiguous,
   int& countOrange,
   int& maxOrangePerLine,
   int& countPixel
  );

  /** Finds the end of the ball */
  bool findEndOfBall(
    const Image& image,
    const CameraInfo& bwCameraInfo,
    const ColorTable& colorTable,
    const BallPoint& start,
    const Vector2<int>& step,
    BallPoint& destination,
    int& countAmbiguous,
    int& countOrange,
    int& maxOrangePerLine,
    int& countPixel
    );

  /**
  * The function tries to calculate the ball percept by using the Levenberg-Marquardt
  * algorithm. The function fails if less than 3 points are available.
  * @return true if ball percept was created
  */
  bool createBallPerceptLevenbergMarquardt
  (
    const BallPointList& ballPoints,
    Vector2<int>& center,
    double& radius
  );

  /*
  * Check if a list of points is inside a given circle
  */
  bool checkIfPointsAreInsideBall(const BallPointList& ballPoints, Vector2<int>& center, double radius);

  /**
  * The function checks whether a ball percept is plausible and will add it if so.
  * @param centerX The x-coordinate of the center of the ball.
  * @param centerY The y-coordinate of the center of the ball.
  * @param radius The radius of the ball.
  */
  void addBallPercept
  (
    const Image& image,
    const CameraInfo& bwCameraInfo,
    const ColorTable& colorTable,
    const CameraMatrix& cameraMatrix,
    const CameraMatrix& prevCameraMatrix,
    const Vector2<int>& center,
    double radius,
    BallPercept& ballPercept
  );  

  const ColorCorrector& colorCorrector;
};

#endif// __BallSpecialist_h_

/*
* $Log: SLAMBallSpecialist.h,v $
* Revision 1.1  2004/07/02 10:11:47  nistico
* Cloned main image processor and created
* SpecialLandmarks specialist for SLAM challenge
*
*
*/
