/**
* @file RegionGrowingImageProcessor.h
*
* Definition of class RegionGrowingImageProcessor
*
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/

#ifndef RegionGrowingImageProcessor_h_
#define RegionGrowingImageProcessor_h_

#include "Modules/ImageProcessor/ImageProcessor.h"
#include "HSI.h"
#include "Tools/Debugging/DebugImages.h"
#include "Tools/Math/Common.h"
/**
* @class RegionGrowingImageProcessor
*
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/
class RegionGrowingImageProcessor : public ImageProcessor
{
public:
  /** 
  * Constructor.
  * @param interfaces The parameters of the RegionGrowingImageProcessor module.
  */
  RegionGrowingImageProcessor(const ImageProcessorInterfaces& interfaces);

  /** Executes the module */
  virtual void execute();

  /** Handles a message, used for colortable changes*/
  bool handleMessage(InMessage& message);

private:
  struct Coord
  {
    unsigned char x, /**< The x coordinate of a pixel. */
                  y; /**< The y coordinate of a pixel. */
  };

  /**
   * This structure represents an entry in the region table for a single pixel.
   * All list-like structures use array indices, because it turned out that this
   * is faster than using pointer (maybe because indices only require 2 bytes,
   * and thereby the overall structure is shorter).
   */
  struct Region
  {
    unsigned short region, /**< If this index points to this entry, this entry represents a region. Otherwise, the follow the link. */
                   count; /**< The number of pixels in the region represented by this entry. */
    union
    {
      struct
      {
        unsigned short horizontal, /**< Link to the next horizontal neighborhood relation with the same contrast. */
                       vertical; /**< Link to the next vertical neighborhood relation with the same contrast. */
        int ySum, /**< The sum of the y channel of all pixels in this region. */
            uSum, /**< The sum of the u channel of all pixels in this region. */
            vSum; /**< The sum of the v channel of all pixels in this region. */
      } mergeInfo; /**< 16 bytes used during region growing. */
      struct 
      {
        unsigned short info; /**< Points to additional information about this region. Invalid if equal to EOL. */
        Coord north, /**< The highest pixel in this region. */
              northWest, /**< The highest/leftest pixel in this region. */
              southWest, /**< The lowest/leftest pixel in this region . */
              south, /**< The lowest pixel in this region. */
              southEast, /**< The lowest/rightest pixel in this region. */
              northEast; /**< The highest/rightest pixel in this region. */
      } regionInfo; /**< 14 bytes used during region calculation. */
    };
  };

  struct Info
  {
    Coord* boundary; /**< Points to the six boundary points in the regions table. */
    unsigned short count; /**< The number of pixels in the region. If 0, the region is invalid. */
    unsigned char y, /**< The average of the y channel of all pixels in this blob. */
                  u, /**< The average of the u channel of all pixels in this blob. */
                  v; /**< The average of the v channel of all pixels in this blob. */
    HSIColor hsi;
    double colorProbability[numOfColors]; /**< The similarity of the color to certain color classes. */
    Vector2<double> upperLeft, /**< Horizon aligned upper left corner of the bounding rectangle. */ 
                    lowerLeft, /**< Horizon aligned lower left corner of the bounding rectangle. */
                    lowerRight, /**< Horizon aligned lower right corner of the bounding rectangle. */
                    upperRight; /**< Horizon aligned upper right corner of the bounding rectangle. */
    double xCenter, /**< The center of the bounding rectangle in x direction. */
           width, /**< The width of the bounding rectangle. */
           height, /**< The height of the bounding rectangle. */
           yMin,
           yMax;
    bool show;
  };

  enum 
  {
    sizeOfNeighborTable = 3 * 255 + 1, /**< The number of possible contrasts between two neighboring pixels. */
    maxNumberOfRegions = 200 /**< The maximum number of region stored. */
  }; 

  Region regions[cameraResolutionHeight_ERS7 * cameraResolutionWidth_ERS7 / 2]; /**< The region table. */
  unsigned short horizontalNeighbors[sizeOfNeighborTable], /**< The beginnings of the horizontal neighborhood lists. */
                 verticalNeighbors[sizeOfNeighborTable], /**< The beginnings of the vertical neighborhood lists. */
                 firstRegion; /**< The beginning of the region list. */
  Geometry::Line horizon; /**< The horizon in image coordinates. */
  Geometry::Line vertical; /**< The perpendicular to the horizon. */
  Info infos[maxNumberOfRegions]; /**< Further information about the regions found. */
  int numberOfRegions; /**< The number of regions found. */
  HSIColorClasses colorClasses;

  DECLARE_DEBUG_IMAGE(imageProcessorGeneral);
  DECLARE_DEBUG_COLOR_CLASS_IMAGE(segmentedImage1);

  /**
   * The function creates the lists of all neighborhood relations and it initilizes the region table.
   * It is a little bit lengthy, because it tries to handle all special cases in different loops to
   * speed up processing. It uses different resolutions for different firstRegion of the image.
   */
  void createNeighborhoodTable();
  
  /**
   * The function merges all pixels to regions.
   */
  void mergeAll();

  /**
   * The function follows a linked list and compresses it by one step.
   * Note: a full compression turned out to be slower.
   * @param r The index of the entry to start with.
   * @return The index of the last entry in the list. This entry represents
   *         the entier region.
   */
  unsigned short compress(unsigned short r)
  {
    while(regions[r].region != r)
    {
      unsigned short next = regions[r].region;
      regions[r].region = regions[next].region;
      r = next;
    }
    return r;
  }

  /**
   * The function merges two neighboring firstRegion if their colors do not differ too much.
   * @param r The index of the first region.
   * @param offset The offset of the index of the second region relative to the first one.
   */
  void mergeNeighbors(unsigned short r, const int offset)
  {
    unsigned short r1 = compress(r),
                   rp = r + offset,
                   r2 = compress(rp);
    
    if(r1 != r2)
    {
      int c = (regions[r1].count * regions[r2].count) << 4;
      if(abs(regions[r1].mergeInfo.vSum * regions[r2].count - regions[r2].mergeInfo.vSum * regions[r1].count) <= c &&
         abs(regions[r1].mergeInfo.uSum * regions[r2].count - regions[r2].mergeInfo.uSum * regions[r1].count) <= c &&
         abs(regions[r1].mergeInfo.ySum * regions[r2].count - regions[r2].mergeInfo.ySum * regions[r1].count) <= (c << 2))
      {
        regions[r1].mergeInfo.ySum += regions[r2].mergeInfo.ySum;
        regions[r1].mergeInfo.uSum += regions[r2].mergeInfo.uSum;
        regions[r1].mergeInfo.vSum += regions[r2].mergeInfo.vSum;
        regions[r1].count += regions[r2].count;
        regions[r2].region = r1;
        regions[rp].region = r1;
      }
    }
  }

  /**
   * The function calculates the average color and the boundary of each region and inserts them into a list.
   */
  void calculateRegions();

  /**
   * The function updates the regions for a single row.
   * @param y The row in the image.
   * @param xStep The step size in x-direction in the row.
   * @param ySteo The step size in y-direction in the row.
   */
  void calculateRegionsForRow(int y, int xStep, int yStep);

  /**
   * The function calculates the bounding rectangle for each region.
   * It also removes some regions that are too small.
   */
  void calculateRegionInfo();

  /**
   * The function projects a point to the horizon and the perpendicular.
   * @param point The point to be projected.
   * @return A point in the system of coordinates of the horizon.
   */
  Vector2<double> project(const Vector2<double>& point) const;

  /**
   * The function searches for landmarks.
   */
  void findLandmarks();

  /**
   * The function searches for a goal.
   * Note: two goals cannot be visible at the same time.
   */
  void findGoal();

  /**
   * The function scans along a line a searches for neighbors in a given distance.
   * @param p1 The start of the line.
   * @param p2 The end of the line.
   * @param offset The relative offset that is added to points on the line to find neighbors.
   * @param probes The number of points searched for neighbors.
   * @param neighbors The set of neighbors found. This array is filled with indices from the infos array.
   */
  int findNeighbors(Vector2<double> p1, const Vector2<double>& p2, 
                    const Vector2<double>& offset, int probes, unsigned short* neighbors);

  static double calcSizeProbability(double size1, double size2)
  {
    return gauss(1 - min(size1, size2) / max(size1, size2), 0.4);
  }

  static double calcMinSizeProbability(double size1, double size2)
  {
    if(size1 < size2)
      return 1;
    else
      return gauss(1 - size2 / size1, 0.4);
  }

  static double calcCenterProbability(double center1, double size1, double center2, double size2)
  {
    return gauss((center1 - center2) / max(size1, size2), 0.2);
  }

  static double calcRangeAroundZeroProbability(double min, double max, double size)
  {
    if(min > 0)
      return gauss(min / size, 1);
    else if(max < 0)
      return gauss(max / size, 1);
    else
      return 1;
  }
};

#endif// RegionGrowingImageProcessor_h_

/*
* $Log: RegionGrowingImageProcessor.h,v $
* Revision 1.1  2004/06/17 11:24:04  roefer
* Added RGIP and GT2004SL
*
*/
