/**
* @file RegionGrowingImageProcessor.cpp
*
* Implementation of class RegionGrowingImageProcessor
*
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/

#include "RegionGrowingImageProcessor.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Modules/ImageProcessor/ImageProcessorTools/ColorCorrector.h"
#include "Tools/Streams/InOut.h"

const int Y = 0, /**< Relative offset of Y component. */
          U = cameraResolutionWidth_ERS7, /**< Relative offset of U component. */
          V = 2 * cameraResolutionWidth_ERS7, /**< Relative offset of V component. */
          Yh = Y - 1,
          Uh = U - 1,
          Vh = V - 1,
          Yh2 = Y - 2,
          Uh2 = U - 2,
          Vh2 = V - 2,
          Yv2 = Y - 12 * cameraResolutionWidth_ERS7,
          Uv2 = U - 12 * cameraResolutionWidth_ERS7,
          Vv2 = V - 12 * cameraResolutionWidth_ERS7,
          Yh4 = Y - 4,
          Uh4 = U - 4,
          Vh4 = V - 4,
          Yv4 = Y - 24 * cameraResolutionWidth_ERS7,
          Uv4 = U - 24 * cameraResolutionWidth_ERS7,
          Vv4 = V - 24 * cameraResolutionWidth_ERS7;

const unsigned short EOL = 0xffff; /**< Marks the end of a horizontal or vertical list. */

static const colorClass classes[] = 
{
  orange,
  yellow,
  skyblue,
  pink,
  green,
  white
};

static const struct
{
  Flag::FlagType type;
  colorClass upperColor,
             lowerColor;
} flags[] =
{
  {Flag::pinkAboveYellow, pink, yellow},
  {Flag::pinkAboveSkyblue, pink, skyblue},
  {Flag::yellowAbovePink, yellow, pink},
  {Flag::skyblueAbovePink, skyblue, pink}
};

RegionGrowingImageProcessor::RegionGrowingImageProcessor(const ImageProcessorInterfaces& interfaces)
: ImageProcessor(interfaces)
{
  InTextFile stream("hsiTable.cfg");
  stream >> colorClasses;
}

void RegionGrowingImageProcessor::execute()
{
  ColorCorrector::load();
  landmarksPercept.reset(image.frameNumber);
  landmarksPercept.cameraOffset = cameraMatrix.translation;

  createNeighborhoodTable();
  mergeAll();
  calculateRegions();
  horizon = Geometry::calculateHorizon(cameraMatrix, image.cameraInfo);
  vertical.base = horizon.base;
  vertical.direction.x = -horizon.direction.y;
  vertical.direction.y = horizon.direction.x;
  calculateRegionInfo();
  findLandmarks();
  landmarksPercept.estimateOffsetForFlags(Vector2<double>(cameraMatrix.translation.x, cameraMatrix.translation.y));
  //findGoal();
  landmarksPercept.estimateOffsetForGoals(Vector2<double>(cameraMatrix.translation.x, cameraMatrix.translation.y));

  // debug output
  COMPLEX_DRAWING(imageProcessor_horizon,
    Vector2<int> horizonStart;
    Vector2<int> horizonEnd;
    Geometry::getIntersectionPointsOfLineAndRectangle(
      Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 1),
      horizon, horizonStart, horizonEnd);
    LINE(imageProcessor_horizon, horizonStart.x, horizonStart.y, horizonEnd.x, horizonEnd.y,
         1, Drawings::ps_solid, Drawings::white);
    DEBUG_DRAWING_FINISHED(imageProcessor_horizon);
  )

  COMPLEX_DRAWING(imageProcessor_general,
    for(unsigned short i = 0; i < numberOfRegions; ++i)
      if(infos[i].count)
      {
        const Info& r = infos[i];
        int m = 0;
        for(int k = 1; k < int(sizeof(classes) / sizeof(classes[0])); ++k)
          if(r.colorProbability[classes[k]] > r.colorProbability[classes[m]])
            m = k;
        if(r.show)
        {
          Drawings::Color col = ColorClasses::colorClassToDrawingsColor(classes[m]);
          Coord* c = r.boundary;
          int j;
          /*for(j = 0; j < 6; ++j)
          {
            LINE(imageProcessor_general, c[j].x, c[j].y, c[(j + 1) % 6].x, c[(j + 1) % 6].y,
                 0, Drawings::ps_solid, col);
          }*/

          for(j = 0; j < 4; ++j)
          {
            Vector2<double>* c = &infos[i].upperLeft;
            LINE(imageProcessor_general, c[j].x, c[j].y, c[(j + 1) % 4].x, c[(j + 1) % 4].y,
              0, Drawings::ps_solid, col);
          }
        }
      }
    DEBUG_DRAWING_FINISHED(imageProcessor_general);
  )

  GENERATE_DEBUG_IMAGE(imageProcessorGeneral,
    imageProcessorGeneralImage.cameraInfo = image.cameraInfo;
    for(int y = 0; y < image.cameraInfo.resolutionHeight; ++y)
      for(int x = 0; x < image.cameraInfo.resolutionWidth; ++x)
      {
        int x2 = x;
        int y2 = y >> 1;
        if(y >= image.cameraInfo.resolutionHeight / 2)
        {
          x2 = ((x2 >> 2) << 2) + 3;
          y2 = (((y2 + 1) >> 1) << 1) - 1;
        }
        else if(y >= image.cameraInfo.resolutionHeight / 4)
          x2 = ((x2 >> 1) << 1) + 1;
        unsigned short i = regions[compress(y2 * cameraResolutionWidth_ERS7 + x2)].regionInfo.info;
        if(i == EOL)
        {
          DEBUG_IMAGE_SET_PIXEL_BLACK(imageProcessorGeneral, x, y);
        }
        else
        {
          DEBUG_IMAGE_SET_PIXEL_YUV(imageProcessorGeneral, x, y, infos[i].y, infos[i].u, infos[i].v);
        }
      }
    SEND_DEBUG_IMAGE(imageProcessorGeneral);
  )
/*
  GENERATE_DEBUG_IMAGE(segmentedImage1,
    segmentedImage1ColorClassImage.width = image.cameraInfo.resolutionWidth;
    segmentedImage1ColorClassImage.height = image.cameraInfo.resolutionHeight;
    HSIColor color;
    for(int y = 0; y < image.cameraInfo.resolutionHeight; ++y)
      for(int x = 0; x < image.cameraInfo.resolutionWidth; ++x)
      {
        const Region* r = &regions[y][x];
        while(r->region != r)
          r = r->region;
        color.fromYCbCr((unsigned char) r->ySum,
                        (unsigned char) r->uSum,
                        (unsigned char) r->vSum);
        segmentedImage1ColorClassImage.image[y][x] = hsiColorTable.getColorClass(color);
      }
    SEND_DEBUG_COLOR_CLASS_IMAGE(segmentedImage1);
  )*/
}

bool RegionGrowingImageProcessor::handleMessage(InMessage& message)
{
  return false;
}

void RegionGrowingImageProcessor::createNeighborhoodTable()
{
  // reset lists
  for(int i = 0; i < sizeOfNeighborTable; ++i)
  {
    horizontalNeighbors[i] = EOL;
    verticalNeighbors[i] = EOL;
  }

  // add first upper left corner without neighbors
  const unsigned char* p = &image.image[1][0][0];
  Region* r = &regions[0];
  r->region = 0;
  unsigned char yy = p[Y],
                uu = p[U],
                vv = p[V];
  ColorCorrector::correct(0, 1, yy, uu, vv);
  r->mergeInfo.ySum = yy;
  r->mergeInfo.uSum = uu;
  r->mergeInfo.vSum = vv;
  r->count = 1;

  // add first row without upper neighbors
  for(int x = 1; x < image.cameraInfo.resolutionWidth; ++x)
  {
    ++r;
    ++p;
    int offset = r - &regions[0];
    r->region = offset;
    unsigned char yy = p[Y],
                  uu = p[U],
                  vv = p[V];
    int index = abs(int(yy) - p[Yh]) +
                abs(int(uu) - p[Uh]) +
                abs(int(vv) - p[Vh]);
    r->mergeInfo.horizontal = horizontalNeighbors[index];
    horizontalNeighbors[index] = offset;
    ColorCorrector::correct(x, 1, yy, uu, vv);
    r->mergeInfo.ySum = yy;
    r->mergeInfo.uSum = uu;
    r->mergeInfo.vSum = vv;
    r->count = 1;
  }

  int y;
  for(y = 1; y < image.cameraInfo.resolutionHeight / 8; ++y)
  {
    // add left pixel in row without left neighbor
    const unsigned char* p = &image.image[y << 1][0][0];
    Region* r = &regions[y * cameraResolutionWidth_ERS7];
    int offset = r - &regions[0];
    r->region = offset;
    unsigned char yy = p[Y],
                  uu = p[U],
                  vv = p[V];
    int index = abs(int(yy) - p[Yv2]) +
                abs(int(uu) - p[Uv2]) +
                abs(int(vv) - p[Vv2]);
    r->mergeInfo.vertical = verticalNeighbors[index];
    verticalNeighbors[index] = offset;
    ColorCorrector::correct(0, y << 1, yy, uu, vv);
    r->mergeInfo.ySum = yy;
    r->mergeInfo.uSum = uu;
    r->mergeInfo.vSum = vv;
    r->count = 1;

    // add all other pixels with left and upper neighbors
    for(int x = 1; x < image.cameraInfo.resolutionWidth; ++x)
    {
      ++p;
      ++r;
      int offset = r - &regions[0];
      r->region = offset;
      unsigned char yy = p[Y],
                    uu = p[U],
                    vv = p[V];
      int index = abs(int(yy) - p[Yh]) +
                  abs(int(uu) - p[Uh]) +
                  abs(int(vv) - p[Vh]);
      r->mergeInfo.horizontal = horizontalNeighbors[index];
      horizontalNeighbors[index] = offset;
      index = abs(int(yy) - p[Yv2]) +
              abs(int(uu) - p[Uv2]) +
              abs(int(vv) - p[Vv2]);
      r->mergeInfo.vertical = verticalNeighbors[index];
      verticalNeighbors[index] = offset;
      ColorCorrector::correct(x, y << 1, yy, uu, vv);
      r->mergeInfo.ySum = yy;
      r->mergeInfo.uSum = uu;
      r->mergeInfo.vSum = vv;
      r->count = 1;
    }
  }

  for(; y < image.cameraInfo.resolutionHeight / 4; ++y)
  {
    // add left pixel in row without left neighbor
    const unsigned char* p = &image.image[y << 1][0][1];
    Region* r = &regions[1 + y * cameraResolutionWidth_ERS7];
    int offset = r - &regions[0];
    r->region = offset;
    unsigned char yy = p[Y],
                  uu = p[U],
                  vv = p[V];
    int index = abs(int(yy) - p[Yv2]) +
                abs(int(uu) - p[Uv2]) +
                abs(int(vv) - p[Vv2]);
    r->mergeInfo.vertical = verticalNeighbors[index];
    verticalNeighbors[index] = offset;
    ColorCorrector::correct(1, y << 1, yy, uu, vv);
    r->mergeInfo.ySum = yy;
    r->mergeInfo.uSum = uu;
    r->mergeInfo.vSum = vv;
    r->count = 1;

    // add all other pixels with left and upper neighbors
    for(int x = 3; x < image.cameraInfo.resolutionWidth; x += 2)
    {
      p += 2;
      r += 2;
      int offset = r - &regions[0];
      r->region = offset;
      r[-1].region = offset - 2;
      unsigned char yy = p[Y],
                    uu = p[U],
                    vv = p[V];
      int index = abs(int(yy) - p[Yh2]) +
                  abs(int(uu) - p[Uh2]) +
                  abs(int(vv) - p[Vh2]);
      r->mergeInfo.horizontal = horizontalNeighbors[index];
      horizontalNeighbors[index] = offset;
      index = abs(int(yy) - p[Yv2]) +
              abs(int(uu) - p[Uv2]) +
              abs(int(vv) - p[Vv2]);
      r->mergeInfo.vertical = verticalNeighbors[index];
      verticalNeighbors[index] = offset;
      ColorCorrector::correct(x, y << 1, yy, uu, vv);
      r->mergeInfo.ySum = yy;
      r->mergeInfo.uSum = uu;
      r->mergeInfo.vSum = vv;
      r->count = 1;
    }
  }

  for(++y; y < image.cameraInfo.resolutionHeight / 2; y += 2)
  {
    // add left pixel in row without left neighbor
    const unsigned char* p = &image.image[y << 1][0][3];
    Region* r = &regions[3 + y * cameraResolutionWidth_ERS7];
    int offset = r - &regions[0];
    r->region = offset;
    r[-cameraResolutionWidth_ERS7].region = (unsigned short) (offset - (cameraResolutionWidth_ERS7 << 1));
    unsigned char yy = p[Y],
                  uu = p[U],
                  vv = p[V];
    int index = abs(int(yy) - p[Yv4]) +
                abs(int(uu) - p[Uv4]) +
                abs(int(vv) - p[Vv4]);
    r->mergeInfo.vertical = verticalNeighbors[index];
    verticalNeighbors[index] = offset;
    ColorCorrector::correct(1, y << 1, yy, uu, vv);
    r->mergeInfo.ySum = yy;
    r->mergeInfo.uSum = uu;
    r->mergeInfo.vSum = vv;
    r->count = 1;

    // add all other pixels with left and upper neighbors
    for(int x = 7; x < image.cameraInfo.resolutionWidth; x += 4)
    {
      p += 4;
      r += 4;
      int offset = r - &regions[0];
      r->region = offset;
      r[-1].region = offset - 4;
      r[-cameraResolutionWidth_ERS7].region = (unsigned short) (offset - (cameraResolutionWidth_ERS7 << 1));
      unsigned char yy = p[Y],
                    uu = p[U],
                    vv = p[V];
      int index = abs(int(yy) - p[Yh4]) +
                  abs(int(uu) - p[Uh4]) +
                  abs(int(vv) - p[Vh4]);
      r->mergeInfo.horizontal = horizontalNeighbors[index];
      horizontalNeighbors[index] = offset;
      index = abs(int(yy) - p[Yv4]) +
              abs(int(uu) - p[Uv4]) +
              abs(int(vv) - p[Vv4]);
      r->mergeInfo.vertical = verticalNeighbors[index];
      verticalNeighbors[index] = offset;
      ColorCorrector::correct(x, y << 1, yy, uu, vv);
      r->mergeInfo.ySum = yy;
      r->mergeInfo.uSum = uu;
      r->mergeInfo.vSum = vv;
      r->count = 1;
    }
  }
}

void RegionGrowingImageProcessor::mergeAll()
{
  for(int i = 0; i < sizeOfNeighborTable; ++i)
  {
    unsigned short r = horizontalNeighbors[i];
    while(r != EOL)
    {
      mergeNeighbors(r, -1);
      r = regions[r].mergeInfo.horizontal;
    }
    r = verticalNeighbors[i];
    while(r != EOL)
    {
      mergeNeighbors(r, -cameraResolutionWidth_ERS7);
      r = regions[r].mergeInfo.vertical;
    }
  }
}

void RegionGrowingImageProcessor::calculateRegions()
{
  numberOfRegions = 0;
  firstRegion = EOL;
  int y;
  for(y = 0; y < image.cameraInfo.resolutionHeight / 4; y += 2)
    calculateRegionsForRow(y, 1, 2);
  for(; y < image.cameraInfo.resolutionHeight / 2; y += 2)
    calculateRegionsForRow(y, 2, 2);
  for(y += 2; y < image.cameraInfo.resolutionHeight; y += 4)
    calculateRegionsForRow(y, 4, 4);
}

void RegionGrowingImageProcessor::calculateRegionsForRow(int y, int xStep, int yStep)
{
  int y2 = y + yStep;
  unsigned short r0 = (y >> 1) * cameraResolutionWidth_ERS7 + xStep - 1,
                 r = compress(r0);
  for(int x = 0; x < image.cameraInfo.resolutionWidth;)
  {
    int x2 = x;
    unsigned short r2 = r;
    while(r == r2 && (x2 += xStep) < image.cameraInfo.resolutionWidth)
    {
      r0 += xStep;
      r2 = compress(r0);
    }

    if(regions[r].count)
    {
      if(regions[r].count < 3)
      {
        // region can only be one pixel wide or high -> mark it as irrelevant
        regions[r].regionInfo.info = EOL;
        // mark region as processed
        regions[r].count = 0;
      }
      else
      {
        Info& i = infos[numberOfRegions];

        // copy count, because it will be set to 0 in region
        i.count = regions[r].count;

        // compress color
        i.y = (unsigned char) (regions[r].mergeInfo.ySum / regions[r].count);
        i.u = (unsigned char) (regions[r].mergeInfo.uSum / regions[r].count);
        i.v = (unsigned char) (regions[r].mergeInfo.vSum / regions[r].count);

        // link region and info
        regions[r].regionInfo.info = numberOfRegions;
        i.boundary = &regions[r].regionInfo.north;
        
        // init boundary
        regions[r].regionInfo.north.x = regions[r].regionInfo.northWest.x = regions[r].regionInfo.southWest.x = (unsigned char) x;
        regions[r].regionInfo.south.x = regions[r].regionInfo.southEast.x = regions[r].regionInfo.northEast.x = (unsigned char) x2;
        regions[r].regionInfo.north.y = regions[r].regionInfo.northWest.y = regions[r].regionInfo.northEast.y = (unsigned char) y;
        regions[r].regionInfo.southWest.y = regions[r].regionInfo.south.y = regions[r].regionInfo.southEast.y = (unsigned char) y2;
        
        // mark region as processed
        regions[r].count = 0;

        if(numberOfRegions < maxNumberOfRegions - 1)
          ++numberOfRegions;
      }
    }
    else // update boundary
    {
      if(x + y < regions[r].regionInfo.northWest.x + regions[r].regionInfo.northWest.y)
      {
        regions[r].regionInfo.northWest.x = (unsigned char) x;
        regions[r].regionInfo.northWest.y = (unsigned char) y;
      }
      if(x - y2 < regions[r].regionInfo.southWest.x - regions[r].regionInfo.southWest.y)
      {
        regions[r].regionInfo.southWest.x = (unsigned char) x;
        regions[r].regionInfo.southWest.y = (unsigned char) y2;
      }
      if(x2 + y2 > regions[r].regionInfo.southEast.x + regions[r].regionInfo.southEast.y)
      {
        regions[r].regionInfo.southEast.x = (unsigned char) x2;
        regions[r].regionInfo.southEast.y = (unsigned char) y2;
      }
      if(x2 - y > regions[r].regionInfo.northEast.x - regions[r].regionInfo.northEast.y)
      {
        regions[r].regionInfo.northEast.x = (unsigned char) x2;
        regions[r].regionInfo.northEast.y = (unsigned char) y;
      }
      regions[r].regionInfo.south.x = (unsigned char) x2;
      regions[r].regionInfo.south.y = (unsigned char) y2;
    }
    x = x2;
    r = r2;
  }
}

void RegionGrowingImageProcessor::calculateRegionInfo()
{
  for(int i = 0; i < numberOfRegions; ++i)
  {
    Info& r = infos[i];
    r.show = false;

    // check whether height or width is only one pixel
    // first calculate minimum and maximum
    Coord min = r.boundary[0],
          max = min;
    for(int j = 1; j < 6; ++j)
    {
      const Coord& point = r.boundary[j];
      if(point.x < min.x)
        min.x = point.x;
      else if(point.x > max.x)
        max.x = point.x;
      if(point.y < min.y)
        min.y = point.y;
      if(point.y > max.y)
        max.y = point.y;
    }

    // "one pixel" depends on the sampling resolution
    int xLimit = 1,
        yLimit = 2;
    if(min.y >= image.cameraInfo.resolutionHeight / 2)
    {
      xLimit = 4;
      yLimit = 4;
    }
    else if(min.y >= image.cameraInfo.resolutionHeight / 4)
      xLimit = 2;

    // calculate horizon aligned boundary only if large enough
    if(max.x - min.x > xLimit && max.y - min.y > yLimit)
    {
      Vector2<double> min = project(Vector2<double>(r.boundary[0].x, r.boundary[0].y)),
                      max = min;
      int j;
      for(j = 1; j < 6; ++j)
      {
        Vector2<double> point = project(Vector2<double>(r.boundary[j].x, r.boundary[j].y));
        if(point.x < min.x)
          min.x = point.x;
        else if(point.x > max.x)
          max.x = point.x;
        if(point.y < min.y)
          min.y = point.y;
        if(point.y > max.y)
          max.y = point.y;
      }
      r.upperLeft = horizon.base + horizon.direction * min.x + vertical.direction * min.y;
      r.upperRight = horizon.base + horizon.direction * max.x + vertical.direction * min.y;
      r.lowerLeft = horizon.base + horizon.direction * min.x + vertical.direction * max.y;
      r.lowerRight = horizon.base + horizon.direction * max.x + vertical.direction * max.y;
      r.hsi.fromYCbCr(r.y, r.u, r.v);
      r.xCenter = (max.x + min.x) / 2;
      r.width = max.x - min.x;
      r.height = max.y - min.y;
      r.yMin = min.y;
      r.yMax = max.y;
      for(j = 0; j < int(sizeof(classes) / sizeof(classes[0])); ++j)
        r.colorProbability[classes[j]] = colorClasses.calcSimilarity(r.hsi, classes[j]);
    }
    else // otherwise mark region as non-existent
      r.count = 0;
  }
}

Vector2<double> RegionGrowingImageProcessor::project(const Vector2<double>& point) const
{
  Vector2<double> offset = point - horizon.base;
  return Vector2<double> (offset * horizon.direction, offset * vertical.direction);
}

void RegionGrowingImageProcessor::findLandmarks()
{
  double maxProbability[4] = {0, 0, 0, 0};
  int maxUpper[4] = {0, 0, 0, 0},
      maxLower[4] = {0, 0, 0, 0};
  double pp[4][8];
  for(int i = 0; i < numberOfRegions; ++i)
    if(infos[i].count)
    {
      double p[8];
      Info& r = infos[i];
      unsigned short upperNeighbors[4],
                     lowerNeighbors[4];
      int numberOfUpperNeighbors = findNeighbors(r.upperLeft, r.upperRight, vertical.direction * -4, 4, upperNeighbors),
          numberOfLowerNeighbors = findNeighbors(r.lowerLeft, r.lowerRight, vertical.direction * 4, 4, lowerNeighbors);
      if(numberOfUpperNeighbors && numberOfLowerNeighbors)
      {
        double maxWhiteness = 0;
        int j;
        for(j = 0; j < numberOfLowerNeighbors; ++j)
        {
          const Info& lower = infos[lowerNeighbors[j]];
          p[0] = lower.colorProbability[white];
          p[1] = calcMinSizeProbability(r.width, lower.width);
          p[2] = calcRangeAroundZeroProbability(lower.yMin, lower.yMax, r.height);
          double whiteness = p[0] * p[1] * p[2];
          if(whiteness > maxWhiteness)
            maxWhiteness = whiteness;
        }

        for(j = 0; j < numberOfUpperNeighbors; ++j)
        {
          const Info& upper = infos[upperNeighbors[j]];
          p[3] = calcSizeProbability(r.width, upper.width);
          p[4] = calcCenterProbability(r.xCenter, r.width, upper.xCenter, upper.width);
          double f = maxWhiteness * p[3] * p[4];
          if(upper.upperLeft.y > 0 || upper.upperRight.y > 0) // full height is visible
          {
            p[5] = calcMinSizeProbability(upper.height, r.height);
            f *= p[5];
          }
          else
            p[5] = 1;

          for(int k = 0; k < 4; ++k)
          {
            p[6] = upper.colorProbability[flags[k].upperColor];
            p[7] = r.colorProbability[flags[k].lowerColor];
            double probability = f * p[6] * p[7];
            if(probability > maxProbability[k])
            {
              maxProbability[k] = probability;
              maxUpper[k] = upperNeighbors[j];
              maxLower[k] = i;
              for(int l = 0; l < 8; ++l)
                pp[k][l] = p[l];
            }
          }
        }
      }
    }
  for(int j = 0; j < 4; ++j)
  {
    // exclude impossible constellations
    // yellow and skyblue landmarks cannot be in the same image
    // two landmarks of the same color can only be in the same image if the are far away from each other
    // always remove the less probable landmark
    if(maxProbability[j] > 0.2 && 
       maxProbability[j] > maxProbability[(j ^ 1) & 1] &&
       maxProbability[j] > maxProbability[(j ^ 1) | 2] &&
       (maxProbability[j] > maxProbability[j ^ 2] || fabs(infos[maxLower[j]].xCenter - infos[maxLower[j ^ 2]].xCenter) > 100))
    {
      Info& upper = infos[maxUpper[j]],
          & lower = infos[maxLower[j]];
      Vector2<double> upperLeft,
                      upperRight,
                      lowerLeft,
                      lowerRight;
      Geometry::calculateAnglesForPoint(Vector2<int>(int(upper.upperLeft.x), int(upper.upperLeft.y)), 
                                        cameraMatrix, image.cameraInfo, upperLeft);
      Geometry::calculateAnglesForPoint(Vector2<int>(int(upper.upperRight.x), int(upper.upperRight.y)), 
                                        cameraMatrix, image.cameraInfo, upperRight);
      Geometry::calculateAnglesForPoint(Vector2<int>(int(lower.lowerLeft.x), int(lower.lowerLeft.y)), 
                                        cameraMatrix, image.cameraInfo, lowerLeft);
      Geometry::calculateAnglesForPoint(Vector2<int>(int(lower.lowerRight.x), int(lower.lowerRight.y)), 
                                        cameraMatrix, image.cameraInfo, lowerRight);
      ConditionalBoundary boundary;
      boundary.addX(upperLeft.x, upper.upperLeft.x < 1 && lower.lowerLeft.x < 1);
      boundary.addY(upperLeft.y, upper.upperLeft.y < 1 && upper.upperRight.y < 1);
      boundary.addX(lowerRight.x, upper.upperRight.x > image.cameraInfo.resolutionWidth - 2 &&
                                  lower.lowerRight.x > image.cameraInfo.resolutionWidth - 2);
      boundary.addY(lowerRight.y, lower.lowerLeft.y > image.cameraInfo.resolutionHeight - 2 &&
                                  lower.lowerRight.y > image.cameraInfo.resolutionHeight - 2);
      landmarksPercept.addFlag(flags[j].type, getPlayer().getTeamColor() == Player::blue, boundary);
    }

    /*OUTPUT(idText, text, j << ": " 
                         << int(maxProbability[j] * 100) << ", "
                         << int(pp[j][0] * 100) << ", "
                         << int(pp[j][1] * 100) << ", "
                         << int(pp[j][2] * 100) << ", "
                         << int(pp[j][3] * 100) << ", "
                         << int(pp[j][4] * 100) << ", "
                         << int(pp[j][5] * 100) << ", "
                         << int(pp[j][6] * 100) << ", "
                         << int(pp[j][7] * 100));*/
  }
}

void RegionGrowingImageProcessor::findGoal()
{
  double maxProbability = 0;
  int maxGoal = 0;
  double pp[8];
  for(int i = 0; i < numberOfRegions; ++i)
    if(infos[i].count)
    {
      double p[8];
      Info& r = infos[i];
      unsigned short lowerNeighbors[10],
                     sideNeighbors[10];
      int numberOfLowerNeighbors = findNeighbors(r.lowerLeft, r.lowerRight, vertical.direction * 4, 10, lowerNeighbors),
          numberOfSideNeighbors = findNeighbors(r.upperLeft, r.lowerLeft, horizon.direction * -4, 5, sideNeighbors);
      numberOfSideNeighbors += findNeighbors(r.upperRight, r.lowerRight, horizon.direction * 4, 5, sideNeighbors + numberOfSideNeighbors);
      double maxWhiteness = 0;
      int j;
      for(j = 0; j < numberOfSideNeighbors; ++j)
      {
        const Info& side = infos[sideNeighbors[j]];
        double p0 = side.colorProbability[white];
        double p1 = calcRangeAroundZeroProbability(side.yMin, side.yMax, r.height);
        double whiteness = p0 * p1;
        if(whiteness > maxWhiteness)
        {
          maxWhiteness = whiteness;
          p[0] = p0;
          p[1] = p1;
        }
      }

      double maxGreenness = 0;
      for(j = 0; j < numberOfLowerNeighbors; ++j)
      {
        const Info& lower = infos[lowerNeighbors[j]];
        double p2 = lower.colorProbability[green];
        double p3 = calcMinSizeProbability(r.width, lower.width);
        double greenness = p2 * p3;
        if(greenness > maxGreenness)
        {
          maxGreenness = greenness;
          p[2] = p2;
          p[3] = p3;
        }
      }

      p[4] = max(r.colorProbability[yellow], r.colorProbability[skyblue]);
      p[5] = calcRangeAroundZeroProbability(r.yMin, r.yMax, r.height);
      double probability = maxWhiteness * maxGreenness * p[4] * p[5];
      if(probability > maxProbability)
      {
        maxProbability = probability;
        maxGoal = i;
        for(int l = 0; l < 6; ++l)
          pp[l] = p[l];
      }
    }
  //if(maxProbability > 0.2)
//    infos[maxGoal].show = true;
  
  OUTPUT(idText, text, int(maxProbability * 100) << ", "
                       << int(pp[0] * 100) << ", "
                       << int(pp[1] * 100) << ", "
                       << int(pp[2] * 100) << ", "
                       << int(pp[3] * 100) << ", "
                       << int(pp[4] * 100) << ", "
                       << int(pp[5] * 100));
}

int RegionGrowingImageProcessor::findNeighbors(Vector2<double> p1, const Vector2<double>& p2, 
                                               const Vector2<double>& offset, int probes, unsigned short* neighbors)
{
  int numberOfNeighbors = 0;
  Vector2<double> step;
  if(probes == 1)
    p1 += (p2 - p1) * 0.5;
  else
    step = (p2 - p1) / (probes - 1);
  for(int i = 0; i < probes; ++i)
  {
    Vector2<double> p = p1 + offset;
    int x = (int) p.x,
        y = (int) (p.y * 0.5);
    if(x >= 0 && x < image.cameraInfo.resolutionWidth &&
       y >= 0 && y < image.cameraInfo.resolutionHeight / 2)
    {
      if(y >= image.cameraInfo.resolutionHeight / 4)
      {
        x = ((x >> 2) << 2) + 3;
        y = (((y + 1) >> 1) << 1) - 1;
      }
      else if(y >= image.cameraInfo.resolutionHeight / 8)
        x = ((x >> 1) << 1) + 1;

      unsigned short neighbor = regions[compress(x + y * cameraResolutionWidth_ERS7)].regionInfo.info;
      ASSERT(neighbor == EOL || neighbor < numberOfRegions);
      if(neighbor != EOL && infos[neighbor].count)
      {
        int j;
        for(j = 0; j < numberOfNeighbors && neighbors[j] != neighbor; ++j)
          ;
        if(j == numberOfNeighbors)
          neighbors[numberOfNeighbors++] = neighbor;
      }
    }
    p1 += step;
  }
  return numberOfNeighbors;
}

/*
* $Log: RegionGrowingImageProcessor.cpp,v $
* Revision 1.2  2004/06/17 16:06:20  roefer
* Warnings removed
*
* Revision 1.1  2004/06/17 11:24:04  roefer
* Added RGIP and GT2004SL
*
*/
