/**
* @file HSI.h
*
* Definition of classes that represents HSI colors and classes.
*
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/

#ifndef HSI_h_
#define HSI_h_

#include "Representations/Perception/Image.h"
#include "Platform/GTAssert.h"
/**
 * The function calculates the Gaussian normal distribution.
 * @param x The value for which the value is calculated.
 * @param sigma The standard deviation.
 * @return The value of the function at x.
 */
inline double gauss(double x, double sigma)
{
  return exp(-x * x / (2 * sigma * sigma));
}

class HSIColor
{
public:
  unsigned char h,
                s,
                i;
  HSIColor() {h = s = i = 0;}
  void fromYCbCr(unsigned char y,
                 unsigned char cb,
                 unsigned char cr)
  {
    Image::convertFromYCbCrToHSI(y, cb, cr, h, s, i);
  }
};

Out& operator<<(Out& stream, const HSIColor& color);
In& operator>>(In& stream, HSIColor& color);


class HSIColorClass;
Out& operator<<(Out& stream, const HSIColorClass& cc);
In& operator>>(In& stream, HSIColorClass& cc);

class HSIColorClass
{
private:
  enum
  {
    hSigma = 10,
    sSigma = 20,
    iSigma = 5
  };

  colorClass id;
  HSIColor min,
           max;

public:
  HSIColorClass() {}

  HSIColorClass(colorClass id, const HSIColor& min, const HSIColor& max) 
  {
    this->id = id;
    this->min = min;
    this->max = max;
  }

  bool isInside(const HSIColor& color) const
  {
    return color.s >= min.s && color.s <= max.s &&
           color.i >= min.i && color.i <= max.i &&
           (min.h <= max.h && color.h >= min.h && color.h <= max.h ||
            min.h > max.h && (color.h >= min.h || color.h <= max.h));
  }

  colorClass getId() const {return id;}

  double calcSimilarity(const HSIColor& color) const;

  friend Out& operator<<(Out& stream, const HSIColorClass& cc);
  friend In& operator>>(In& stream, HSIColorClass& cc);
};

class HSIColorClasses;
Out& operator<<(Out& stream, const HSIColorClasses& cc);
In& operator>>(In& stream, HSIColorClasses& cc);

class HSIColorClasses
{
private:
  enum {numOfColorClasses = numOfColors - 3}; // no noColor, black, yellowOrange
  HSIColorClass classes[numOfColorClasses];

  static int compare(const HSIColorClass*, const HSIColorClass*);

public:
  HSIColorClasses() {}

  colorClass getColorClass(const HSIColor& color) const
  {
    for(int i = 0; i < numOfColorClasses; ++i)
      if(classes[i].isInside(color))
        return classes[i].getId();
    return noColor;
  }

  double calcSimilarity(const HSIColor& color, colorClass c) const
  {
    ASSERT(classes[c - 1].getId() == c);
    return classes[c - 1].calcSimilarity(color);
  }

  friend Out& operator<<(Out& stream, const HSIColorClasses& cc);
  friend In& operator>>(In& stream, HSIColorClasses& cc);
};

#endif// HSI_h_

/*
* $Log: HSI.h,v $
* Revision 1.1  2004/06/17 11:24:04  roefer
* Added RGIP and GT2004SL
*
*/
