/**
* @file HSI.cpp
*
* Implementation of classes related to HSI colors and color classes.
*
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/

#include "HSI.h"
#include "Tools/Streams/InOut.h"
#include "Platform/GTAssert.h"

Out& operator<<(Out& stream, const HSIColor& color)
{
  return stream << color.h << color.s << color.i;
}

In& operator>>(In& stream, HSIColor& color)
{
  return stream >> color.h >> color.s >> color.i;
}

double HSIColorClass::calcSimilarity(const HSIColor& color) const
{
  double h = 1,
         s = 1,
         i = 1;

  if(min.h < max.h && (color.h < min.h || color.h > max.h) ||
     min.h >= max.h && (color.h < min.h && color.h > max.h))
  {
    int h1 = abs(color.h - min.h),
        h2 = abs(color.h - max.h);
    if(h1 > 128)
      h1 = 255 - h1;
    if(h2 > 128)
      h2 = 255 - h2;
    if(h1 < h2)
      h = gauss(h1, hSigma);
    else
      h = gauss(h2, hSigma);
  }  

  if(color.s < min.s)
    s = gauss(color.s - min.s, sSigma);
  else if(color.s > max.s)
    s = gauss(color.s - max.s, sSigma);

  if(color.i < min.i)
    i = gauss(color.i - min.i, iSigma);
  else if(color.i > max.i)
    s = gauss(color.i - max.i, iSigma);

  return h * s * i;
}

Out& operator<<(Out& stream, const HSIColorClass& cc)
{
  return stream << ColorClasses::getColorName(cc.id) << cc.min << cc.max << endl;
}

In& operator>>(In& stream, HSIColorClass& cc)
{
  char buf[80];
  stream >> buf;
  int i;
  for(i = 1; i < numOfColors && strcmp(ColorClasses::getColorName(colorClass(i)), buf); ++i)
    ;
  ASSERT(i < numOfColors);
  cc.id = colorClass(i);
  stream >> cc.min >> cc.max >> endl;
  return stream;
}

int HSIColorClasses::compare(const HSIColorClass* c1, const HSIColorClass* c2)
{
  return c1->getId() < c2->getId() ? -1 : c1->getId() > c2->getId();
}

Out& operator<<(Out& stream, const HSIColorClasses& cc)
{
  for(int i = 0; i < HSIColorClasses::numOfColorClasses; ++i)
    stream << cc.classes[i];
  return stream;
}

In& operator>>(In& stream, HSIColorClasses& cc)
{
  for(int i = 0; i < HSIColorClasses::numOfColorClasses; ++i)
    stream >> cc.classes[i];
  qsort(cc.classes, HSIColorClasses::numOfColorClasses, sizeof(HSIColorClass), 
        (int(*)(const void*, const void*)) HSIColorClasses::compare);
  return stream;
}

/*
* $Log: HSI.cpp,v $
* Revision 1.1  2004/06/17 11:24:04  roefer
* Added RGIP and GT2004SL
*
*/
