/** 
* @file SegmentationTools.h
* Declaration of file SegmentationTools.
*
* @author <A href=mailto:damdfree.fr>Damien DEOM</A>
*/

#ifndef _SEGMENTATION_TOOLS_
#define _SEGMENTATION_TOOLS_



#include "Tools/Math/Geometry.h"
#include "Tools/Math/Vector2.h"
#include "Tools/ColorClasses.h"
#include "MSH2004EdgeDetection.h"


typedef Vector2<int> point;

typedef Vector2<double> direction;

inline point mil(point& v1,point& v2) {
	return point ((v1.x+v2.x) / 2,(v1.y+v2.y) / 2);
};


template <class T> class listed {
protected:

	T* next;

public:
	listed(): next(0) {}
	//~listed() {if(next)delete next;}

	T* getNext(){return next;}

	void setNext(T* f) {next = f;}

	void insert(T* i) {
		if (!i) return;
		T* tmp = next;
		next = i;
		i->setNext(tmp);
	}

	void swapNext(T* toSwapWith) {
		
		if (!(next && toSwapWith)) return;
		T* tmp = this->next;
		T* tmp2 = toSwapWith->next->next;

		this->next = toSwapWith->next;
		toSwapWith->next->next = tmp;
		toSwapWith->next = tmp2;

	}

};


struct figure : public listed<figure> {

	figure() : listed<figure>() {}

	virtual unsigned short getId() = 0;
	virtual point toConsider() = 0;
	virtual double size() {return 0;}
};


class edge : public figure {
private:
	point pt;
	unsigned short id;
public:

	edge(point& p, unsigned short i) : figure(), pt(p) , id(i) {}
	edge(int& x, int& y, unsigned short i) :  figure(), id(i) {
	
		pt.x = x; pt.y = y;
	}

	point toConsider() {;return pt;}
	unsigned short getId() {return id;}

};



struct LinePair2 : public figure {
	virtual point pt1() = 0;
	virtual point pt2() = 0;
	virtual direction getDirection() = 0;
};

class horLinePair2 : public LinePair2 {

	point p1;
	int p2x;
	
	point center;
public:
	horLinePair2(point p1, int p2x)  {
		this->p1 = p1;
		this->p2x = p2x;
		center = point((p1.x+p2x)/2, p1.y);
	}

	horLinePair2(point p1, point p2)  {
		this->p1 = p1;
		this->p2x = p2.x;
		center = point((p1.x+p2.x)/2, p1.y);
	}

	unsigned short getId() {return (unsigned short)p1.y;}
	point toConsider() {return center;}
	point pt1() {return p1;}
	point pt2() {return point(p2x, p1.y);}
	double size() {return (double) (p2x - p1.x);}
	direction getDirection() {return direction(1,0);}


};

struct extLinePair2 : public LinePair2 {

	point p1, p2;
	point center;
	unsigned short id;

	extLinePair2(point& p1, point& p2, unsigned short id)  {
		this->p1 = p1;
		this->p2 = p2;
		this->id = id;
		center = mil(p1, p2);
	}

	point pt1()  {return p1;}
	point pt2()  {return p2;}
	double size() {return (p2 - p1).abs();}
	point toConsider() {return center;}
	unsigned short getId() {return id;}
	direction getDirection() {
		return direction((double)(p2.x - p1.x),(double)(p2.y - p1.y));
	}


};

struct coloredLinePair2 : public extLinePair2 {
	colorClass color;

	coloredLinePair2(point& v1,point& v2,unsigned short id, colorClass c) :
		extLinePair2(v1, v2, id), color(c) {}

};



struct coloredHorLinePair2 : public horLinePair2 {

	colorClass color;

	coloredHorLinePair2(point& v1,point& v2, colorClass c) :
		horLinePair2(v1, v2), color(c) {}

};


template <class T> class slist {

private:
	unsigned int size;
	T* first, *last;

public:

	class iterator {

	private:

		T* elt;

	public:
	
		iterator(T* l = 0) : elt(l) {}
		//~iterator() {delete elt;}

		T* get() const {return elt;}
		T* getNext() const {return elt == 0 ? 0 : elt->getNext();}
		bool end() const {return  elt == 0;}

		T* operator++() {

			if (end()) return 0;
			elt= elt->getNext();
			return elt;
		}

		T* operator++(int) {
			if (end()) return 0;
			T* old = elt;
			elt = elt->getNext();
			return old;
		}

		T*operator+=(const int n) {
			T* tmp = elt;
			int i;
			for (i = 0; i<n;++i) 
				if (tmp) tmp = tmp->getNext();
				else return 0;

			elt = tmp;
			return elt;
		}

	};


	T* front()  {return first;}
	T* back()  {return last;}
	void setLast(T* l) {last = l;}
	
	void cutFront() { first = last = 0;size = 0;}

	slist() : size(0), first(0), last(0){}
	~slist() {delete first, delete last;}
	
	void push_front(T* lp) {
		if (lp==0) return;
		if (!last) last = lp; 

		lp->setNext(first);
		first = lp;
		++size;
	}

	void push_back(slist<T>& lst) {
		if (last) {
			last->setNext(lst.front());
			last = lst.back();
		} else {
			first = lst.front();
			last = lst.back();
		}
		size+= lst.getSize();
		lst.cutFront();
	}

	unsigned getSize() const {return size;}
	bool empty() {return size==0;}
	void incSize(unsigned int n) {if (n>0) size+=n;}

	void clear() {

		while (first) {
			T* tmp = first->getNext();
			delete first;
			first = tmp;
		}

		last = 0;
		size = 0;

	}


	T* pop_front() {

		if (!first) return 0;
		T* tmp = first->getNext();
		delete first;
		first = tmp;
		if (size==1) last = first;
		--size;
		return first;
	}

	void insert(T*  pos, T* i) {
		if (!pos) return;
		T* tmp = pos->getNext();

		if (!tmp) {
			pos->setNext(i);
			i->setNext(0);
			last = i;
		} else {
			pos->setNext(i);
			i->setNext(tmp);
		}

		if (pos==last) last = i;
	}

	void erase(T* e) {
		
		if (e->getNext()==last) last = e;

		T* n = e->getNext()->getNext();
		delete e->getNext();
		e->setNext(n);
		--size;

	}

};


struct lineOnField : public listed<lineOnField> {

	point p1, p2;

	lineOnField(point& p1, point& p2) : listed<lineOnField>() {
		this->p1 = p1;
		this->p2 = p2;
	}

	point pt1()  {return p1;}
	point pt2()  {return p2;}
	double size() {return (p2 - p1).abs();}

	Geometry::Line getLine() {

		Geometry::Line l;

		l.base = Vector2<double> ((double)p1.x, (double)p1.y);
		l.direction = Vector2<double> ((double)p2.x, (double)p2.y) - l.base;

		return l;

	}

};



/* adds a new scan line between two line pairs */


inline bool DoubleScanLinesNumber(slist<figure>& lst, MSH2004EdgeDetection& edgeScanner) {

	if (lst.getSize()<2) return false;

	edgeScanner.threshold = 12;

	int stepY;

	slist<figure>::iterator it(lst.front());

	unsigned int c = 0;

	do {

		LinePair2 *lp = (LinePair2 *)it.get();
		LinePair2 *next = (LinePair2 *)lp->getNext();
		int f1x = lp->pt1().x,
			f2x = lp->pt2().x;
		int n1x = next->pt1().x,
			n2x = next->pt2().x;
		int n1y = next->pt1().y,
			f1y = lp->pt1().y;

		stepY = (int)(n1y - f1y)/2;

		//	OUTPUT(idText,text, "step " << stepY);

		if (n1x<f2x && n2x>f1x /*&& n2x-n1x <20 */&& n1y!=f1y && stepY>0) {

			int leftCorner = (f1x<n1x) ? f1x : n1x, 
				rightCorner = (f2x>n2x) ? f2x : n2x;

			point center ((int)(leftCorner+rightCorner)/2, f1y+stepY);

			DOT(imageProcessor_ground, center.x,center.y,Drawings::red, Drawings::white);


			point left = center, right = center;

			edgeScanner.scanWest(left.x,left.y);
			edgeScanner.scanEast(right.x,right.y);
		
			extLinePair2* lp = new extLinePair2(left,right,0);

			LINE(imageProcessor_ground,
					lp->pt1().x,lp->pt1().y, lp->pt2().x,lp->pt2().y, 
					0.5, Drawings::ps_solid, Drawings::red);

			if ((lp->pt2().x - lp->pt1().x)<=3*(f2x-f1x)) {
				it.get()->insert(lp);++c;
			}

		}



	} while (++it && !it.end());

	lst.incSize(c);
	return true;

};


inline figure* getMin (figure * lp) {
	
	figure * min = lp, *cur;
	slist<figure>::iterator it(lp);

	double d_min = 10000.0;
	unsigned short cur_id;

	do {

		cur = it.get();

		if (!cur->getNext()) break;

		double d =Geometry::distance(lp->toConsider(), cur->getNext()->toConsider());
		cur_id = cur->getId();

		if (d<d_min) { d_min = d; min = cur;}

		//OUTPUT(idText,text, "id = " << cur_id);

	} while (++it /*&& it.get()->getId()==cur_id*/);

	return min;
}


/* join the nearest LinePairs in order to make the interpretation of the shape possible */

inline void createLinearSegment(slist<figure>& lst) {

	if (lst.getSize()<3)return;

	slist<figure>::iterator start(lst.front());

	do {
		//if (start.get()->getId() == start.getNext()->getId()) {
			figure* min = getMin(start.get());
			if (min != start.get()) start.get()->swapNext(min);
		//}

			//OUTPUT(idText,text, "min : " << min->toConsider().x << " " <<min->toConsider().y);
		//}
	
	} while (++start && start.getNext());

	//lst.setLast(start.get());
}

/*
inline double AngleBetweenLines(lineOnField* l1, lineOnField* l2)  {

	Vector2<double> pt1();
	Geometry::getIntersectionOfLines(l1->line,l2->line,pt1());
	Vector2<double> v2(l1->line.direction.x,l1->line.direction.y);
	double d =  Geometry::getDistanceToLine(l2->line,v2);

	return asin(d/(l1->line.direction-l1->line.base).abs());
}

*/



/** Gives a number between 0 and 360 with the same order as v.angle()*/
inline double theta2(point v1, point v2){
	double t = 0;
	int dx = v1.x - v2.x;
	int dy = v1.y - v2.y;
	int ax = abs(dx);
	int ay = abs(dy);
	if (dx == 0 && dy == 0) t = 0;
	else t = (double)dy /(double)(ax + ay);
	if (dx < 0) t = 2 - t;
	else if (dy < 0) t = 4 + t;
	return t*90; // multiplication with 90 is not really needed for correct ordering
}

inline double theta2(lineOnField* l1, lineOnField* l2) {

	double angle1 = theta2(l1->pt1(),l1->pt2());
	double angle = theta2(l2->pt1(), l2->pt2()) - angle1;

	while (angle<0) angle+=180;

	return angle;

}

inline double AngleRelativeToHorizontal(const point & pt1, const point& pt2) {
	double d1 =  abs(pt2.x - pt1.x);
	double d2 =  (pt2 - pt1).abs();
	return acos(d1/d2);
}




#endif

