/**
* @file Modules/ImageProcessor/ImageProcessorTools/MSH2004ColorCorrector.h
* 
* This file contains an utility-class for detecting edges in digital images. 
*
* @author <A href="mailto:sadprofessor@web.de">Bernd Schmidt</A>
*/

#ifndef __MSH2004EdgeDetection_h_
#define __MSH2004EdgeDetection_h_

#include "Representations/Perception/Image.h"
#include "Modules/ImageProcessor/RasterImageProcessor/RasterImageProcessor.h"


/**
* @class MSH2004EdgeDetection
* Utility-class for edge-detection.
*/
class MSH2004EdgeDetection
{  

public:

  /**
  * Default constructor.
  */

	MSH2004EdgeDetection(RasterImageProcessor& processor);
	
	~MSH2004EdgeDetection();

	int threshold;

	int tempX;

	int tempY;

	bool scanEast(int& x, int& y);

	bool scanWest(int& x, int& y);
	
	bool scanNorth(int& x, int& y);

	bool scanSouth(int& x, int& y);

	bool ballScanEast(int& x, int& y);

	bool ballScanWest(int& x, int& y);

	/** Follows the given direction,
		and determines edges by the sum of all three channels.
		The position of the edge is stored in x and y.
		@param x The starting x-value.
		@param y The starting y-value.
		@return edge was found*/
	bool scan(int& x,int& y,Vector2<double> direction);

	/** Follows the given direction,
		and determines edges by the sum of all three channels.
		The position of the edge is stored in x and y.
		Fills the colorBuffer while scanning, so the client
		can determine with the colorBuffer-Array what kind of
		object was found.

		@param x The starting x-value.
		@param y The starting y-value.
		@return edge was found*/
	bool bufferedScan(int& x,int& y,Vector2<double> direction);

	/** Follows the last given direction. Skips 3 pixels at the beginning
	of each scan, while filling the buffer.*/ 
	bool bufferedScan(int& x,int& y);

		/** Follows the given direction,and searches for field-edge-points. 
		Determines edges by the sum of the first two channels.
		The position of the edge is stored in x and y.
		@param x The starting x-value.
		@param y The starting y-value.
		@return edge was found*/
	bool scanField(int& x,int& y,Vector2<double> direction);

	/** Follows the last given direction,and searches for field-edge-points. 
		Determines edges by the sum of the first two channels.
		The position of the edge is stored in x and y.
		@param x The starting x-value.
		@param y The starting y-value.
		@return edge was found*/
	bool scanField(int& x,int& y);

	/** Follows the last given direction,
		and determines edges by the sum of all three channels */
	bool scan(int& x,int& y);

	bool colorScan(int& x,int& y,Vector2<double> direction,colorClass& lastColor);

	bool colorScan(int& x,int& y,colorClass& lastColor);
	
	bool susanScan(int& x,int& y,Vector2<double> direction);
	
	bool scan(int& x,int& y,int x1,int y1);

	bool colorScan(int& x,int& y,int x1,int y1,colorClass& lastColor);

	inline void setDirection(Vector2<double> dir){
		cx = (dir.x<0)? -1 : 1;
		cy = (dir.y<0)? -1 : 1;
		dx = abs((int)(dir.x * 500.0f));
	  dy = abs((int)(dir.y * 500.0f));
		if (dx>dy){
			e = dx - 2*dy;
			f = 2*(dx - dy);
			g = -2*dy;
		}
		else{
			e = dy - 2*dx;
			f = 2*(dy - dx);
			g = -2*dx;
		}
		direction = dir;
	}

	inline colorClass getColor(int index){
		return colorBuffer[index];
	}

	inline int getRange(int index){
		return ranges[index];
	}

	inline bool isHorizontalEdge(int x, int y){
		return horizontalEdgeVote(x,y) > threshold;
	}

	inline int horizontalEdgeVote(int x,int y){
		if (x < 1 || x > ip.image.cameraInfo.resolutionWidth-2) return -1;
		int cx1 = x-1;
		int cx2 = x+1;
		int a1 = ip.image.image[y][0][cx1];
		int b1 = ip.image.image[y][1][cx1];
		int c1 = ip.image.image[y][2][cx1];
		int a2 = ip.image.image[y][0][cx2];
		int b2 = ip.image.image[y][1][cx2];
		int c2 = ip.image.image[y][2][cx2];

		return abs(a1 - a2) + abs(b1 - b2) + abs(c1 - c2);
	}

	inline bool isVerticalEdge(int x, int y){
		return verticalEdgeVote(x,y) > threshold;
	}

	inline int verticalEdgeVote(int x,int y){
		if (y < 1 || y > ip.image.cameraInfo.resolutionHeight-2) return -1;
		int cy1 = y-1;
		int cy2 = y+1;
		int a1 = ip.image.image[cy1][0][x];
		int b1 = ip.image.image[cy1][1][x];
		int c1 = ip.image.image[cy1][2][x];
		int a2 = ip.image.image[cy2][0][x];
		int b2 = ip.image.image[cy2][1][x];
		int c2 = ip.image.image[cy2][2][x];

		return abs(a1 - a2) + abs(b1 - b2) + abs(c1 - c2);
	}

	inline int ballEdgeVote(int x,int y){
		//could be different code as in crossEdgeVote()
		if (x < 1 || x > ip.image.cameraInfo.resolutionWidth-2) return -1;
		if (y < 1 || y > ip.image.cameraInfo.resolutionHeight-2) return -1;
		int vertical = 0;
		int horizontal = 0;
		int p1 = x-1;
		int p2 = x+1;
		int a1 = ip.image.image[y][0][p1];
		int b1 = ip.image.image[y][1][p1];
		int c1 = ip.image.image[y][2][p1];
		int a2 = ip.image.image[y][0][p2];
		int b2 = ip.image.image[y][1][p2];
		int c2 = ip.image.image[y][2][p2];
		
		horizontal = abs(a1 - a2) + abs(b1 - b2) + abs(c1 - c2);
		
		p1 = y-1;
		p2 = y+1;
		a1 = ip.image.image[p1][0][x];
		b1 = ip.image.image[p1][1][x];
		c1 = ip.image.image[p1][2][x];
		a2 = ip.image.image[p2][0][x];
		b2 = ip.image.image[p2][1][x];
		c2 = ip.image.image[p2][2][x];
		
		vertical = abs(a1 - a2) + abs(b1 - b2) + abs(c1 - c2);
		
		return (vertical > horizontal) ? vertical : horizontal; 
	}

	inline int crossEdgeVote(int x,int y){
		if (x < 1 || x > ip.image.cameraInfo.resolutionWidth-2) return -1;
		if (y < 1 || y > ip.image.cameraInfo.resolutionHeight-2) return -1;
		int vertical = 0;
		int horizontal = 0;
		int p1 = x-1;
		int p2 = x+1;
		int a1 = ip.image.image[y][0][p1];
		int b1 = ip.image.image[y][1][p1];
		int c1 = ip.image.image[y][2][p1];
		int a2 = ip.image.image[y][0][p2];
		int b2 = ip.image.image[y][1][p2];
		int c2 = ip.image.image[y][2][p2];
		
		horizontal = abs(a1 - a2) + abs(b1 - b2) + abs(c1 - c2);
		
		p1 = y-1;
		p2 = y+1;
		a1 = ip.image.image[p1][0][x];
		b1 = ip.image.image[p1][1][x];
		c1 = ip.image.image[p1][2][x];
		a2 = ip.image.image[p2][0][x];
		b2 = ip.image.image[p2][1][x];
		c2 = ip.image.image[p2][2][x];
		
		vertical = abs(a1 - a2) + abs(b1 - b2) + abs(c1 - c2);
		
		return (vertical > horizontal) ? vertical : horizontal; 
	}
	inline int fieldEdgeVote(int x,int y){
		if (x < 1 || x > ip.image.cameraInfo.resolutionWidth-2) return -1;
		if (y < 1 || y > ip.image.cameraInfo.resolutionHeight-2) return -1;
		int vertical = 0;
		int horizontal = 0;
		int p1 = x-1;
		int p2 = x+1;
		int a1 = ip.image.image[y][0][p1];
		int b1 = ip.image.image[y][1][p1];
		int a2 = ip.image.image[y][0][p2];
		int b2 = ip.image.image[y][1][p2];

		
		horizontal = abs(a1 - a2) + abs(b1 - b2);
		
		p1 = y-1;
		p2 = y+1;
		a1 = ip.image.image[p1][0][x];
		b1 = ip.image.image[p1][1][x];
		a2 = ip.image.image[p2][0][x];
		b2 = ip.image.image[p2][1][x];
		
		vertical = abs(a1 - a2) + abs(b1 - b2);
		
		return (vertical > horizontal) ? vertical : horizontal; 
	}
	/* implemented with colorCorrection*/
	inline colorClass getColor(int x,int y){
		if (x < 1 || x > ip.image.cameraInfo.resolutionWidth-2) return (colorClass)-1;
		if (y < 1 || y > ip.image.cameraInfo.resolutionHeight-2) return (colorClass)-1;

		//TODO remove hack for color correction
		unsigned char cy = ip.image.image[y][0][x];
		unsigned char cu = ip.image.image[y][1][x];
		unsigned char cv = ip.image.image[y][2][x];
		ip.corrector.getCorrectedPixel(x,y,cy,cu,cv);
		return ip.colorTable.getColorClass(cy,cu,cv);	
	}

	/** Gives the next pixel of last scan. This is only for the scans that
		use Bresenham's algorithm.*/
	inline void skip(int& x,int& y){
		if (dx>dy){
			x+=cx;
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
		}
		else{
			y+=cy;
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
		}
	}

	inline bool findStart(Vector2<int>& start,const Vector2<double>& dir){
		int dx = (int)(dir.x*300 + 0.5);
	    int dy = (int)(dir.y*300 + 0.5);

		Vector2<int> rightBorder(start.x + dx,start.y + dy);

		return Geometry::clipLineWithRectangleCohenSutherland(
			Vector2<int>(1,1),
			Vector2<int>(ip.image.cameraInfo.resolutionWidth -2,ip.image.cameraInfo.resolutionHeight -2),
			start,
			rightBorder
			);
	}

	inline int getBufferSize(){
		return bufferSize;
	}

//Bresenham
/*inline void draw_line(int x0, int y0, int x1, int y1){
	int x,y=y0,dx = 2*(x1-x0),dy = 2*(y1-y0);
	int dydx = dy - dx, d = dy - dx/2;
	for(x=x0;x<=x1;x++){
		DOT(imageProcessor_general,x,y,
			Drawings::white, Drawings::red);
		if(d<0) {d+= dy;}
		else {y+=1; d += dydx;}
	}
}
*/


private:

	RasterImageProcessor& ip;
	
	SUSANEdgeDetectionLite& susanDetector;
	Vector2<double> direction;

	int e;

	int f;

	int g;

	int dx;

	int dy;

	int cx;

	int cy;

	int bufferSize;

	colorClass currentColor;

	colorClass lastColor;
	
	int colorRange;

	/** This buffer is used to store the colors 
	* passed by a buffered scan-method during the last scan
	*/
	colorClass colorBuffer[20];

	/** This buffer is used to store the ranges of the colors */
	int ranges[20];

	inline int fastCrossEdgeVote(int x,int y){
		int vertical = 0;
		int horizontal = 0;
		int p1 = x-1;
		int p2 = x+1;
		int a1 = ip.image.image[y][0][p1];
		int b1 = ip.image.image[y][1][p1];
		int c1 = ip.image.image[y][2][p1];
		int a2 = ip.image.image[y][0][p2];
		int b2 = ip.image.image[y][1][p2];
		int c2 = ip.image.image[y][2][p2];
		
		horizontal = abs(a1 - a2) + abs(b1 - b2) + abs(c1 - c2);
		
		p1 = y-1;
		p2 = y+1;
		a1 = ip.image.image[p1][0][x];
		b1 = ip.image.image[p1][1][x];
		c1 = ip.image.image[p1][2][x];
		a2 = ip.image.image[p2][0][x];
		b2 = ip.image.image[p2][1][x];
		c2 = ip.image.image[p2][2][x];
		
		vertical = abs(a1 - a2) + abs(b1 - b2) + abs(c1 - c2);
		
		return (vertical > horizontal) ? vertical : horizontal; 
	}

	inline void addColor(){
		if (lastColor != -1){
			colorBuffer[bufferSize] = lastColor;
			ranges[bufferSize] = colorRange;
			bufferSize++;
			colorRange = 0;
		}
		else{
			colorRange = 0;
		}
	}

	inline bool insideImage(int x,int y){
		if (x < 1 || x > ip.image.cameraInfo.resolutionWidth-2) return false;
		if (y < 1 || y > ip.image.cameraInfo.resolutionHeight-2) return false;
		return true;
	}

	inline int susanVote(int x,int y){
		if (x < 1 || x > ip.image.cameraInfo.resolutionWidth-2) return -1;
		if (y < 1 || y > ip.image.cameraInfo.resolutionHeight-2) return -1;
		int i = 0;
		if (susanDetector.isEdgePoint(ip.image,x,y,
			SUSANEdgeDetectionLite::componentA)) ++i;
		//if (susanDetector.isEdgePoint(ip.image,x,y,
		//	SUSANEdgeDetectionLite::componentB)) ++i;
		//if (susanDetector.isEdgePoint(ip.image,x,y,
		//	SUSANEdgeDetectionLite::componentC)) ++i;
		return i;
	}


	
};

#endif// __MSH2004EdgeDetection_h_

/*
* Change log :
* 
* $Log: MSH2004EdgeDetection.h,v $
* Revision 1.1.1.1  2004/05/22 17:19:53  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/04/08 15:33:06  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.3  2004/03/25 15:10:03  pg_besc
* made some changes
*
* Revision 1.2  2004/03/17 20:58:47  schmidtb
* added new scan methods and buffering.
*
* Revision 1.1  2004/03/11 20:19:43  schmidtb
* Established MSH2004EdgeDetection.
*
*/
