/**
* @file Modules/ImageProcessor/ImageProcessorTools/MSH2004ColorCorrector.h
* 
* This file contains a class that represents a table used for color correction.
*
* @author <A href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</A>
*/

#ifndef __MSH2004ColorCorrector_h_
#define __MSH2004ColorCorrector_h_

#include "Representations/Perception/Image.h"


const double centralPointX = 107.5;
const double centralPointY = 79.5;
/**
* @class MSH2004ColorCorrector
* The class represents a table for color correction.
*/
class MSH2004ColorCorrector
{
private:

  /** 
  * radialOrder is the polinomial power (+1) of the radial factor of the color correction.
  * colorOrder is the polinomial power (+1) of the brighness factor of the color correction.
  */
  enum {maxRadialOrder=10, maxColorOrder=10, max_radius=140};
  int radialOrder;
  int colorOrder;

  unsigned char radiusLUT[cameraResolutionHeight_ERS7][cameraResolutionWidth_ERS7]; /** Radius look up table, used to index the correction table. */
  unsigned char colorCorrectionLUT[max_radius][256][3]; /** Correction table. */
  bool active; /** Flag which keeps the corrector status. */

  double radialP[3*maxRadialOrder];
  double colorP[3*maxColorOrder];

  /** The following functions are called by the constructor to build the LookUpTables*/
  void setupRadiusLUT();
  void setupCorrectionLUT();
  
  /** Corrects the radiometric distortion of a single channel of a given pixel, based on the distance from the image center.
  * @param radius_i The distance of the given pixel from the image center.
  * @param color The brightness value of the given channel
  * @param channel Selects the channel to which the color distortion is to be applied (Y=0, U=1, V=2).
  * @return The corrected value.
  */
  unsigned char colorDistortionCorrection(const unsigned char radius_i, 
                        const unsigned char color, const unsigned char channel) const;
  
public:
  /**
  * Default constructor.
  */
  MSH2004ColorCorrector();

  /**
  * This functions returns a radiometric distortion corrected pixel.
  * @param x The x coordinate of the pixel.
  * @param y The y coordinate of the pixel.
  * @param Y The Y (brightness) color channel (to be) corrected.
  * @param U The U (crominance) color channel (to be) corrected.
  * @param V The V (crominance) color channel (to be) corrected.
  */
  inline void getCorrectedPixel(const int x, const int y, unsigned char& Y, unsigned char& U, unsigned char& V) const
  {
    unsigned char radius = radiusLUT[y][x];
    Y = colorCorrectionLUT[radius][Y][0];
    U = colorCorrectionLUT[radius][U][1];
    V = colorCorrectionLUT[radius][V][2];
  }

  inline void getCorrectedColor(const int x, const int y, unsigned char& color, unsigned char spectrum) const
  {
    unsigned char radius = radiusLUT[y][x];
    color = colorCorrectionLUT[radius][color][spectrum];
  }

  /**
  * This functions corercts the radiometric distortion of a chosen pixel in an image.
  * @param source The source image
  * @param x The x coordinate of the pixel.
  * @param y The y coordinate of the pixel.
  */
  inline void correctPixel(Image& source, const int x, const int y) const
  {
    unsigned char radius = radiusLUT[y][x];
    unsigned char Y, U, V;
    Y = source.image[y][0][x];
    U = source.image[y][1][x];
    V = source.image[y][2][x];
    source.image[y][0][x] = colorCorrectionLUT[radius][Y][0];
    source.image[y][1][x] = colorCorrectionLUT[radius][U][1];
    source.image[y][2][x] = colorCorrectionLUT[radius][V][2];
  }
  
  /**
  * The function returns whether the corrector is active.
  * @return Is it active?
  */
  inline bool isActive() const {return active;}
};

#endif// __MSH2004ColorCorrector_h_

/*
* Change log :
* 
* $Log: MSH2004ColorCorrector.h,v $
* Revision 1.1.1.1  2004/05/22 17:19:52  cvsadm
* created new repository GT2004_WM
*
* Revision 1.4  2004/04/19 21:36:21  nistico
* Parameter file format more flexible
*
* Revision 1.3  2004/04/08 15:33:06  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.3  2004/03/28 18:35:19  nistico
* Temporarily created MSH2004ImageProcessor2, it has some important advancements compared to
* the MSH2004ImageProcessor1 but it is untested on real robots yet, hence the creation of the
* new module.
* It will be removed soon, don't use it!
*
* Revision 1.2  2004/03/19 11:04:58  nistico
* Some corrections and restructuring
*
* Revision 1.1  2004/03/01 12:54:09  nistico
* -Added MSH2004ColorCorrector
* -Integrated MSH2004ColorCorrector into ColorTable32KImageProcessor (through a hack! :-) this has to be improved...
*
*
*/
