
#ifndef __ImageProcessorUtilityClasses_h_
#define __ImageProcessorUtilityClasses_h_


#include "Tools/Math/Vector2.h"
#include "Tools/Math/Matrix2x2.h"

/**
* @class ImageInfo
*
* Additional information about the current image,
* computed by the ImageProcessor
*/
class ImageInfo
{
public:
  /** The horizon*/
  Geometry::Line horizon;
  /** A line perpendicular to the horizon*/
  Geometry::Line vertLine;
  /** Flag, indicates whether the horizon is in the image or not*/
  bool horizonInImage;
  /** The starting point of the horizon*/
  Vector2<int> horizonStart;
  /** The end point of the horizon*/
  Vector2<int> horizonEnd;
  /** The bottom right corner of the image*/
  Vector2<int> maxImageCoordinates;
};


/**
* @class Run
*
* Describes a sequence of pixels of the same colour
*/
class Run
{
public:
  /** Constructor*/
  Run():length(0) {}

  /** The first point*/
  Vector2<int> start;
  /** The last point*/
  Vector2<int> end;
  /** The first point of the corresponding scan line*/
  Vector2<int> scanLineStart;
  /** The length of the run*/
  int length;
  /** The color*/
  colorClass color;
};


/**
* @class TransformedRun
*
* Special class for comparing and clustering runs
*/
class TransformedRun
{
public:
  /** Computes values
  * @param run An original run
  * @param rotMat The rotation matrix
  * @param numOfRun The number of the transformed run
  */
  void transform(const Run& run, const Matrix2x2<double>& rotMat, int numOfRun)
  {
    start.x = (double)(run.start.x - run.scanLineStart.x);
    start.y = (double)(run.start.y - run.scanLineStart.y);
    end.x = (double)(run.end.x - run.scanLineStart.x);
    end.y = (double)(run.end.y - run.scanLineStart.y);
    start = (rotMat*start);
    end = (rotMat*end);
    start.x += (double)run.scanLineStart.x;
    start.y += (double)run.scanLineStart.y;
    end.x += (double)run.scanLineStart.x;
    end.y += (double)run.scanLineStart.y;
    this->numOfRun = numOfRun;
  }

  /** The first point*/
  Vector2<double> start;
  /** The last point*/
  Vector2<double> end;
  /** Number of corresponding real run*/
  int numOfRun;
};


#endif
