/**
* @file GoalRecognizer.cpp
*
* Implementation of class GoalRecognizer
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#include "GoalRecognizer.h"
#include "Tools/Player.h"
#include "Tools/FieldDimensions.h"
#include "Tools/RangeArray.h"
#include "Representations/Perception/ColorTable64.h"

#include "ColorTableReferenceColor.h"

//color table 64
//~ #define COLOR_CLASS(y,u,v) (colorClass)((const ColorTable64&) colorTable).colorClasses[(y) >> 2][(u) >> 2][(v) >> 2]
#define COLOR_CLASS(y,u,v) colorTable.getColorClass(y,u,v)

GoalRecognizer::GoalRecognizer
(
 const Image& image, 
 const CameraMatrix& cameraMatrix,
 const ColorTable& colorTable,
 int goalIndicationAboveHorizon,
 int goalIndicationBelowHorizon,
 ObstaclesPercept& obstaclesPercept,
 LandmarksPercept& landmarksPercept
 )
 :
image(image),
cameraMatrix(cameraMatrix),
colorTable(colorTable),
goalIndicationAboveHorizon(goalIndicationAboveHorizon),
goalIndicationBelowHorizon(goalIndicationBelowHorizon),
obstaclesPercept(obstaclesPercept),
landmarksPercept(landmarksPercept)
{
  useFixedScanLines = false;
}

GoalRecognizer::GoalRecognizer
(
 const Image& image, 
 const CameraMatrix& cameraMatrix,
 const ColorTable& colorTable,
 ObstaclesPercept& obstaclesPercept,
 LandmarksPercept& landmarksPercept
 )
 :
image(image),
cameraMatrix(cameraMatrix),
colorTable(colorTable),
obstaclesPercept(obstaclesPercept),
landmarksPercept(landmarksPercept)
{
  useFixedScanLines = true;
}


GoalRecognizer::~GoalRecognizer()
{
}

void GoalRecognizer::getGoalPercept(LandmarksPercept& landmarksPercept)
{
  if(getPlayer().getTeamColor() == Player::blue) 
  {
    colorOfOpponentGoal = yellow;
    colorOfOwnGoal = skyblue;
  }
  else 
  {
    colorOfOpponentGoal = skyblue;
    colorOfOwnGoal = yellow;
  }

  INIT_DEBUG_IMAGE(imageProcessorGoals, image);
  
  horizonLine = Geometry::calculateHorizon(cameraMatrix, image.cameraInfo);
  verticalLine.base = horizonLine.base;
  verticalLine.direction.x = - horizonLine.direction.y;
  verticalLine.direction.y = horizonLine.direction.x;

  if(useFixedScanLines)
    calculateScanLinesParallelToHorizon();
  else
    calculateScanLinesParallelToHorizon(goalIndicationAboveHorizon, goalIndicationBelowHorizon, 7);

  scanHorizontalForGoals();
  calculateVerticalGoalScanLines();
  scanLinesForGoals();

  SEND_DEBUG_IMAGE(imageProcessorGoals);
}

void GoalRecognizer::calculateScanLinesParallelToHorizon()
{
  numberOfHorizontalScanLines = 0;
  
  Geometry::Line lineAboveHorizon;
  lineAboveHorizon.direction = horizonLine.direction;
  
  int i;
  
  Vector2<int> bottomLeft(0,0);
  Vector2<int> topRight(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3);
  
  for(i = 14; i >= 0; i--)
  {
    lineAboveHorizon.base = horizonLine.base + verticalLine.direction * ( (i * 4.5) - 25 );
    
    if(Geometry::getIntersectionPointsOfLineAndRectangle(
      bottomLeft, topRight, lineAboveHorizon,
      leftPoint[numberOfHorizontalScanLines], 
      rightPoint[numberOfHorizontalScanLines])
      )
    {
      numberOfHorizontalScanLines++;
    }
  }
}

void GoalRecognizer::calculateScanLinesParallelToHorizon
(
 int distanceAboveHorizon,
 int distanceBelowHorizon,
 int numberOfScanLines
 )
{
  numberOfHorizontalScanLines = 0;
  
  Geometry::Line scanLine;
  scanLine.direction = horizonLine.direction;
  
  int i;
  
  Vector2<int> bottomLeft(0,0);
  Vector2<int> topRight(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3);
  
  for(i = 0; i < numberOfScanLines; i++)
  {
    scanLine.base = 
      horizonLine.base + 
      verticalLine.direction * -(distanceBelowHorizon + (distanceAboveHorizon - distanceBelowHorizon) * i / (numberOfScanLines - 1));
//      verticalLine.direction * distanceBelowHorizon;// - (-distanceAboveHorizon + distanceBelowHorizon) * i / numberOfScanLines);
    
    if(Geometry::getIntersectionPointsOfLineAndRectangle(
      bottomLeft, topRight, scanLine,
      leftPoint[numberOfHorizontalScanLines], 
      rightPoint[numberOfHorizontalScanLines])
      )
    {
      numberOfHorizontalScanLines++;
    }
  }
}

void GoalRecognizer::scanHorizontalForGoals()
{
  enum {yellowGoal = 1, skyblueGoal = 0, numberOfGoalColors = 2};
  int colorToCheck;
  colorClass colorClassToCheck;

  RangeArray<double> goalClusters[numberOfGoalColors];
  numberOfGoalIndications = 0;

  // variables needed to determine free part of goal
  RangeArray<double> goalClustersForFreePart[numberOfGoalColors];
  int numberOfLinesWithLargeParts[numberOfGoalColors] = {0, 0};

  int i;
  for(i = 0; i < numberOfHorizontalScanLines; i++)
  {
    Geometry::PixeledLine line(leftPoint[i], rightPoint[i]);
    
    // state machine
    // states
    enum { noGoalColorFound, 
      foundBeginOfGoalColor, 
      foundEndOfGoalColor
    } goalColorState[numberOfGoalColors] = {noGoalColorFound,noGoalColorFound};
    // counter
    int numberOfPixelsSinceLastOccurrenceOfGoalColor[numberOfGoalColors];
    int numberOfPixelsSinceFirstOccurrenceOfGoalColor[numberOfGoalColors] = {0,0};

    bool colorStartsAtBeginOfImage[numberOfGoalColors] = {false, false};
    ColoredPartsCheck goalCheck[numberOfGoalColors];

    int x = 0, y = 0;
    colorClass color;

    if(line.getNumberOfPixels() > 3)
    {
      for(int z = 2; z < line.getNumberOfPixels(); z++)
      {
        x = line.getPixelX(z);
        y = line.getPixelY(z);
        
        color = COLOR_CLASS(image.image[y][0][x], image.image[y][1][x], image.image[y][2][x]);

        for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
        {
          if(colorToCheck == yellowGoal) colorClassToCheck = yellow;
          else colorClassToCheck = skyblue;

          if(goalColorState[colorToCheck] == noGoalColorFound)
          {
            DEBUG_IMAGE_SET_PIXEL_BLACK(imageProcessorGoals, x, y + colorToCheck);
            if(color == colorClassToCheck)
            {
              goalCheck[colorToCheck].numberOfColoredPixels = 0;
              goalCheck[colorToCheck].rangeOfColor = 0;
              if(numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] == 0)
              {
                goalCheck[colorToCheck].firstPoint.x = x;
                goalCheck[colorToCheck].firstPoint.y = y;
              }
              numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck]++;
              if(numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] > 2)
              {
                goalColorState[colorToCheck] = foundBeginOfGoalColor;
                if(z < 5) 
                {
                  DOT(imageProcessor_general, x, y, Drawings::green, Drawings::green);
                  colorStartsAtBeginOfImage[colorToCheck] = true;
                }
              }
            }
            else
            {
              numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] = 0;
            }
          }
          else if(goalColorState[colorToCheck] == foundBeginOfGoalColor)
          {
            DEBUG_IMAGE_SET_PIXEL_GRAY(imageProcessorGoals, x, y + colorToCheck);
            if(color == colorClassToCheck) goalCheck[colorToCheck].numberOfColoredPixels++;
            else
            {
              goalCheck[colorToCheck].lastPoint.x = x;
              goalCheck[colorToCheck].lastPoint.y = y;
              goalColorState[colorToCheck] = foundEndOfGoalColor;
              numberOfPixelsSinceLastOccurrenceOfGoalColor[colorToCheck] = 0;
              numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] = 0;
            }
            goalCheck[colorToCheck].rangeOfColor++;
          }
          else if(goalColorState[colorToCheck] == foundEndOfGoalColor)
          {
            DEBUG_IMAGE_SET_PIXEL_WHITE(imageProcessorGoals, x, y + colorToCheck);
            numberOfPixelsSinceLastOccurrenceOfGoalColor[colorToCheck]++;
            if(color == colorClassToCheck)
            {
              numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck]++;
              if(numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] > 2)
              {
                goalColorState[colorToCheck] = foundBeginOfGoalColor;
              }
            }
            else if(
              numberOfPixelsSinceLastOccurrenceOfGoalColor[colorToCheck] > 5
              ||
              color == white
              )
            {
              goalColorState[colorToCheck] = noGoalColorFound;
              goalCheck[colorToCheck].determineLargePart(10, colorStartsAtBeginOfImage[colorToCheck], false, cameraMatrix, image.cameraInfo);
              DOT(imageProcessor_general, x, y, Drawings::green, Drawings::white);
              colorStartsAtBeginOfImage[colorToCheck] = false;
            }
            else
            {
              numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] = 0;  
            }
            goalCheck[colorToCheck].rangeOfColor++;
          }
        }//for(int colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
      }//for(int z = 2; z < line.getNumberOfPixels(); z++)

      for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
      {
        if(goalColorState[colorToCheck] == foundBeginOfGoalColor || goalColorState[colorToCheck] == foundEndOfGoalColor)
        {
          if(goalColorState[colorToCheck] == foundBeginOfGoalColor)
          {
            goalCheck[colorToCheck].lastPoint.x = x;
            goalCheck[colorToCheck].lastPoint.y = y;
          }
          goalCheck[colorToCheck].determineLargePart(10, colorStartsAtBeginOfImage[colorToCheck], true, cameraMatrix, image.cameraInfo);
        }//if(goalColorState == foundBeginOfGoalColor || goalColorState == foundEndOfGoalColor)
      }//for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
    }//if(line.getNumberOfPixels() > 3)

    for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
    {
      // Add all large parts found on that line to the structure that determines clusters.
      if(goalCheck[colorToCheck].numberOfLargeParts > 0) numberOfLinesWithLargeParts[colorToCheck]++;
      int index;
      for(index = 0; index < goalCheck[colorToCheck].numberOfLargeParts; index++)
      {
        goalClusters[colorToCheck].add(
          Range<double>(goalCheck[colorToCheck].largePartEndAngles[index].x, goalCheck[colorToCheck].largePartBeginAngles[index].x), 
          goalCheck[colorToCheck].largePartBeginIsOnBorder[index],
          goalCheck[colorToCheck].largePartEndIsOnBorder[index]
          );
        if(numberOfLinesWithLargeParts[colorToCheck] > 2 && numberOfLinesWithLargeParts[colorToCheck] < 5)
        {
          goalClustersForFreePart[colorToCheck].add(
            Range<double>(goalCheck[colorToCheck].largePartEndAngles[index].x, goalCheck[colorToCheck].largePartBeginAngles[index].x), 
            goalCheck[colorToCheck].largePartBeginIsOnBorder[index],
            goalCheck[colorToCheck].largePartEndIsOnBorder[index]
            );
        }
      }
    }
  }// for(int i = 0; i < numberOfHorizontalScanLines; i++)

  for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
  {
    if(colorToCheck == yellowGoal) colorClassToCheck = yellow;
    else colorClassToCheck = skyblue;

    // Add one goal indication for each cluster
    int index;
    for(index = 0; index < goalClusters[colorToCheck].getNumberOfClusters(); index++)
    {
      Range<double> currentRange = goalClusters[colorToCheck].getCluster(index);
      Vector2<int> leftPoint, rightPoint;
      Geometry::calculatePointByAngles(Vector2<double>(currentRange.max,0.02 * index), cameraMatrix, image.cameraInfo, leftPoint);
      Geometry::calculatePointByAngles(Vector2<double>(currentRange.min,0.02 * index), cameraMatrix, image.cameraInfo, rightPoint);
      LINE(imageProcessor_flagsAndGoals, leftPoint.x, leftPoint.y, rightPoint.x, rightPoint.y, 2, Drawings::ps_solid, Drawings::blue);
      goalIndicationLeft[numberOfGoalIndications]   = leftPoint;
      goalIndicationRight[numberOfGoalIndications]  = rightPoint;
      goalIndicationCenter[numberOfGoalIndications] = (leftPoint + rightPoint) / 2;
      leftOfGoalIndicationIsOnBorder[numberOfGoalIndications] = goalClusters[colorToCheck].isLeftOnBorder(index);
      rightOfGoalIndicationIsOnBorder[numberOfGoalIndications] = goalClusters[colorToCheck].isRightOnBorder(index);
      colorOfGoalIndication[numberOfGoalIndications] = colorClassToCheck;
      numberOfGoalIndications++;
    }

    // Determine the free part of the goal
    for(index = 0; index < goalClustersForFreePart[colorToCheck].getNumberOfClusters(); index++)
    {
      Range<double> currentRange = goalClustersForFreePart[colorToCheck].getCluster(index);
      Vector2<int> leftPoint, rightPoint;
      Geometry::calculatePointByAngles(Vector2<double>(currentRange.min,-0.1 + 0.02 * index), cameraMatrix, image.cameraInfo, leftPoint);
      Geometry::calculatePointByAngles(Vector2<double>(currentRange.max,-0.1 + 0.02 * index), cameraMatrix, image.cameraInfo, rightPoint);
      LINE(imageProcessor_flagsAndGoals, leftPoint.x, leftPoint.y, rightPoint.x, rightPoint.y, 2, Drawings::ps_solid, Drawings::green);
    }

    double center, angle;
    bool determinedFreePartOfGoal = false;
    if(goalClustersForFreePart[colorToCheck].getNumberOfClusters() == 1)
    {
      bool leftOfPartIsOnBorder = goalClustersForFreePart[colorToCheck].isLeftOnBorder(0);
      bool rightOfPartIsOnBorder = goalClustersForFreePart[colorToCheck].isRightOnBorder(0);
      Range<double> freePartRange = goalClustersForFreePart[colorToCheck].getCluster(0);
      if(!leftOfPartIsOnBorder && !rightOfPartIsOnBorder || freePartRange.getSize() > 0.7 * image.cameraInfo.openingAngleWidth)
      {
        determinedFreePartOfGoal = true;
        center = (freePartRange.min + freePartRange.max) / 2.0;
        angle = freePartRange.max - freePartRange.min;
      }
    }
    else if(goalClustersForFreePart[colorToCheck].getNumberOfClusters() == 2)
    {
      Range<double> freePartRange[2];
      freePartRange[0] = goalClustersForFreePart[colorToCheck].getCluster(0);
      freePartRange[1] = goalClustersForFreePart[colorToCheck].getCluster(1);

      bool freePartIsSeenCompletely[2];

      freePartIsSeenCompletely[0] =
        !goalClustersForFreePart[colorToCheck].isLeftOnBorder(0) && 
        !goalClustersForFreePart[colorToCheck].isRightOnBorder(0);

      freePartIsSeenCompletely[1] =
        !goalClustersForFreePart[colorToCheck].isLeftOnBorder(1) && 
        !goalClustersForFreePart[colorToCheck].isRightOnBorder(1);

      int largePart, smallPart;
      if(freePartRange[0].getSize() < freePartRange[1].getSize()) 
      {
        largePart = 1;
        smallPart = 0;
      }
      else 
      {
        largePart = 0;
        smallPart = 1;
      }

      if(
        freePartIsSeenCompletely[largePart] && freePartIsSeenCompletely[smallPart] ||
        !freePartIsSeenCompletely[largePart] && freePartIsSeenCompletely[smallPart]
        )
      {
        determinedFreePartOfGoal = true;
        center = (freePartRange[largePart].min + freePartRange[largePart].max) / 2.0;
        angle = freePartRange[largePart].max - freePartRange[largePart].min;
      }
    }

    if(determinedFreePartOfGoal)
    {
      if(colorClassToCheck == colorOfOpponentGoal)
      {
        obstaclesPercept.angleToFreePartOfGoal[ObstaclesPercept::opponentGoal] = center;
        obstaclesPercept.widthOfFreePartOfGoal[ObstaclesPercept::opponentGoal] = angle;
        obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::opponentGoal] = true;
      }
      else
      {
        obstaclesPercept.angleToFreePartOfGoal[ObstaclesPercept::ownGoal] = center;
        obstaclesPercept.widthOfFreePartOfGoal[ObstaclesPercept::ownGoal] = angle;
        obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::ownGoal] = true;
      }
    }

  }//for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
}//void GridImageProcessor2::scanHorizontalForFlagsAndGoals()

void GoalRecognizer::calculateVerticalGoalScanLines()
{
  numberOfGoalScanLines = 0;
  int i;
  Vector2<double> anglesForGoalScanLine;
  for(i = 0; i < numberOfGoalIndications; i++)
  { 
    // Line through center of indication
    Geometry::calculateAnglesForPoint(goalIndicationCenter[i], cameraMatrix, image.cameraInfo, anglesForGoalScanLine);
    anglesForGoalScanLine.y = -1.0;
    Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
    anglesForGoalScanLine.y =  0.3;
    Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, topGoalPoint[numberOfGoalScanLines]);

    if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3),
      topGoalPoint[numberOfGoalScanLines], bottomGoalPoint[numberOfGoalScanLines]))
    {
      colorOfGoalScanLine[numberOfGoalScanLines] = colorOfGoalIndication[i];
      indexOfGoalIndication[numberOfGoalScanLines] = i;
      scanLineToCheckBestAngle[numberOfGoalScanLines] = false;
      numberOfGoalScanLines++;
    }

    // Line at the left side
    Geometry::calculateAnglesForPoint((goalIndicationLeft[i] + goalIndicationCenter[i]) / 2, cameraMatrix, image.cameraInfo, anglesForGoalScanLine);
    anglesForGoalScanLine.y = -1.0;
    Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
    anglesForGoalScanLine.y =  0.3;
    Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, topGoalPoint[numberOfGoalScanLines]);

    if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3),
      topGoalPoint[numberOfGoalScanLines], bottomGoalPoint[numberOfGoalScanLines]))
    {
      colorOfGoalScanLine[numberOfGoalScanLines] = colorOfGoalIndication[i];
      indexOfGoalIndication[numberOfGoalScanLines] = i;
      scanLineToCheckBestAngle[numberOfGoalScanLines] = false;
      numberOfGoalScanLines++;
    }

    // Line at the right side
    Geometry::calculateAnglesForPoint((goalIndicationRight[i] + goalIndicationCenter[i]) / 2, cameraMatrix, image.cameraInfo, anglesForGoalScanLine);
    anglesForGoalScanLine.y = -1.0;
    Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
    anglesForGoalScanLine.y =  0.3;
    Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, topGoalPoint[numberOfGoalScanLines]);

    if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3),
      topGoalPoint[numberOfGoalScanLines], bottomGoalPoint[numberOfGoalScanLines]))
    {
      colorOfGoalScanLine[numberOfGoalScanLines] = colorOfGoalIndication[i];
      indexOfGoalIndication[numberOfGoalScanLines] = i;
      scanLineToCheckBestAngle[numberOfGoalScanLines] = false;
      numberOfGoalScanLines++;
    }

  }

  // line for the free part of the goal
  for(i = 0; i < 2; i++)
  {
    if(obstaclesPercept.angleToFreePartOfGoalWasDetermined[i] == true)
    {
      anglesForGoalScanLine.x = obstaclesPercept.angleToFreePartOfGoal[i];
      anglesForGoalScanLine.y = -1.0;
      Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
      anglesForGoalScanLine.y =  0.3;
      Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, topGoalPoint[numberOfGoalScanLines]);

      if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3),
        topGoalPoint[numberOfGoalScanLines], bottomGoalPoint[numberOfGoalScanLines]))
      {
        if(i==1) colorOfGoalScanLine[numberOfGoalScanLines] = colorOfOpponentGoal;
        else colorOfGoalScanLine[numberOfGoalScanLines] = colorOfOwnGoal;
        scanLineToCheckBestAngle[numberOfGoalScanLines] = true;
        numberOfGoalScanLines++;
      }
    }
  }
}// calculateVerticalGoalScanLines()

void GoalRecognizer::scanLinesForGoals()
{
  ConditionalBoundary yellowGoalBoundary, skyblueGoalBoundary;
  ConditionalBoundary boundaryOfIndication[maxNumberOfGoalScanLines];
  bool useBoundaryOfIndication[maxNumberOfGoalScanLines];
  int i;
  for(i = 0; i < numberOfGoalIndications; i++) useBoundaryOfIndication[i] = true;

  bool foundYellowGoal = false;
  bool foundSkyblueGoal = false;
  bool pixelHasGoalColor;

  for(i = 0; i < numberOfGoalScanLines; i++)
  {
    Geometry::PixeledLine line(topGoalPoint[i], bottomGoalPoint[i]);

    // state machine
    // states
    enum { noGoalColorFound, 
      foundBeginOfGoalColor, 
      foundEndOfGoalColor,
      determinedGoal
    } goalColorState = noGoalColorFound;
    // counter
    int numberOfPixelsSinceLastOccurrenceOfGoalColor = 0;
    int numberOfPixelsSinceFirstOccurrenceOfGoalColor = 0;
    int whiteAndPinkCounter = 0;

    bool colorStartsAtBeginOfImage = false;

    ColoredPartsCheck goalCheck;

    int x = 0, y = 0;
    colorClass color;

    if(line.getNumberOfPixels() > 3)
    {
      bool scanLineIsPartOfGoal = true;
      for(int z = 2; z < line.getNumberOfPixels(); z++) 
      {
        pixelHasGoalColor = false;
        x = line.getPixelX(z);
        y = line.getPixelY(z);

        if(scanLineToCheckBestAngle[i]){
          DEBUG_IMAGE_SET_PIXEL_GRAY(imageProcessorGoals, x, y);
        }
        else {DEBUG_IMAGE_SET_PIXEL_WHITE(imageProcessorGoals, x, y);}

        color = COLOR_CLASS(image.image[y][0][x], image.image[y][1][x], image.image[y][2][x]);
        
        if(colorOfGoalScanLine[i] == skyblue && color == skyblue) pixelHasGoalColor = true;
        if(colorOfGoalScanLine[i] == yellow && color == yellow) pixelHasGoalColor = true;
        
        if(goalColorState == noGoalColorFound)
        {
          if(pixelHasGoalColor)
          {
            DEBUG_IMAGE_SET_PIXEL_BLACK(imageProcessorGoals, x, y);
            goalCheck.numberOfColoredPixels = 0;
            goalCheck.rangeOfColor = 0;
            if(numberOfPixelsSinceFirstOccurrenceOfGoalColor == 0)
            {
              goalCheck.firstPoint.x = x;
              goalCheck.firstPoint.y = y;
            }
            numberOfPixelsSinceFirstOccurrenceOfGoalColor++;
            if(numberOfPixelsSinceFirstOccurrenceOfGoalColor > 2)
            {
              goalColorState = foundBeginOfGoalColor;
              if(z < 5) colorStartsAtBeginOfImage = true;
            }
          }
          else
          {
            numberOfPixelsSinceFirstOccurrenceOfGoalColor = 0;
          }
        }
        else if(goalColorState == foundBeginOfGoalColor)
        {
          DEBUG_IMAGE_SET_PIXEL_GREEN(imageProcessorGoals, x, y);
          if(pixelHasGoalColor) goalCheck.numberOfColoredPixels++;
          else
          {
            goalCheck.lastPoint.x = x;
            goalCheck.lastPoint.y = y;
            goalColorState = foundEndOfGoalColor;
            numberOfPixelsSinceLastOccurrenceOfGoalColor = 0;
            numberOfPixelsSinceFirstOccurrenceOfGoalColor = 0;
          }
          goalCheck.rangeOfColor++;
        }
        else if(goalColorState == foundEndOfGoalColor)
        {
          numberOfPixelsSinceLastOccurrenceOfGoalColor++;
          DEBUG_IMAGE_SET_PIXEL_RED(imageProcessorGoals, x, y);
          if(pixelHasGoalColor)
          {
            numberOfPixelsSinceFirstOccurrenceOfGoalColor++;
            if(numberOfPixelsSinceFirstOccurrenceOfGoalColor > 2)
            {
              goalColorState = foundBeginOfGoalColor;
            }
          }
          else if(numberOfPixelsSinceLastOccurrenceOfGoalColor > 20)
          {
            DEBUG_IMAGE_SET_PIXEL_BLUE(imageProcessorGoals, x, y);
//            goalColorState = noGoalColorFound;
            goalColorState = determinedGoal;
            if(goalCheck.determineLargePart(7, colorStartsAtBeginOfImage, false, cameraMatrix, image.cameraInfo))
            {
              if(whiteAndPinkCounter > 5)
              {
                scanLineIsPartOfGoal = false;
                DOT(imageProcessor_general, x, y, Drawings::blue, Drawings::blue);
              }
            }
            whiteAndPinkCounter = 0;
            colorStartsAtBeginOfImage = false;
          }
          else if(color == white || color == pink) 
          {
            DEBUG_IMAGE_SET_PIXEL_PINK(imageProcessorGoals, x, y);
            whiteAndPinkCounter++;
          }
          else
          {
            numberOfPixelsSinceFirstOccurrenceOfGoalColor = 0;  
          }
          goalCheck.rangeOfColor++;
        }
      }//for(int z = 0; z < line.getNumberOfPixels(); z++) 
      if(goalColorState == foundBeginOfGoalColor || goalColorState == foundEndOfGoalColor)
      {
        if(goalColorState == foundBeginOfGoalColor)
        {
          goalCheck.lastPoint.x = x;
          goalCheck.lastPoint.y = y;
        }
        goalCheck.determineLargePart(7, colorStartsAtBeginOfImage, true, cameraMatrix,image.cameraInfo);
      }//if(goalColorState == foundBeginOfGoalColor || goalColorState == foundEndOfGoalColor)
      
      if(!scanLineIsPartOfGoal && !scanLineToCheckBestAngle[i]) 
        useBoundaryOfIndication[indexOfGoalIndication[i]] = false;

      if(goalCheck.numberOfLargeParts > 0)
      {
        Vector2<double> topAngles, bottomAngles, leftAngles, rightAngles;
        Geometry::calculateAnglesForPoint(goalCheck.largePartBegin[0], cameraMatrix, image.cameraInfo, topAngles);
        Geometry::calculateAnglesForPoint(goalCheck.largePartEnd[goalCheck.numberOfLargeParts - 1], cameraMatrix, image.cameraInfo, bottomAngles);
        
        if(scanLineToCheckBestAngle[i])
        {
          if(bottomAngles.y > 0 || !scanLineIsPartOfGoal) 
          {
            if(colorOfOpponentGoal == colorOfGoalScanLine[i])
              obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::opponentGoal] = false;
            else
              obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::ownGoal] = false;
          }
          else
          {
            //calculate distance
            int distance = Geometry::getDistanceBySize(
              image.cameraInfo,
              goalHeight, 
              (goalCheck.largePartBegin[0] - goalCheck.largePartEnd[goalCheck.numberOfLargeParts - 1]).abs() 
            );
            //~ int distance = Geometry::getDistanceBySize(
              //~ goalHeight, 
              //~ (goalCheck.largePartBegin[0] - goalCheck.largePartEnd[goalCheck.numberOfLargeParts - 1]).abs(), 
              //~ image.cameraInfo.resolutionWidth, image.cameraInfo.openingAngleWidth);

            if(colorOfOpponentGoal == colorOfGoalScanLine[i])
              obstaclesPercept.distanceToFreePartOfGoal[ObstaclesPercept::opponentGoal] = distance;
            else
              obstaclesPercept.distanceToFreePartOfGoal[ObstaclesPercept::ownGoal] = distance;
          }
        }
        else
        {
          Geometry::calculateAnglesForPoint(goalIndicationLeft[indexOfGoalIndication[i]], cameraMatrix, image.cameraInfo, leftAngles);
          Geometry::calculateAnglesForPoint(goalIndicationRight[indexOfGoalIndication[i]], cameraMatrix, image.cameraInfo, rightAngles);
          
          // if(bottomAngles.y < 0)
          //if(topAngles.y > 0)
          {
            boundaryOfIndication[indexOfGoalIndication[i]].addY(topAngles.y, goalCheck.largePartBeginIsOnBorder[0]);
            boundaryOfIndication[indexOfGoalIndication[i]].addY(bottomAngles.y, goalCheck.largePartEndIsOnBorder[goalCheck.numberOfLargeParts - 1]);
            boundaryOfIndication[indexOfGoalIndication[i]].addX(leftAngles.x, leftOfGoalIndicationIsOnBorder[indexOfGoalIndication[i]]);
            boundaryOfIndication[indexOfGoalIndication[i]].addX(rightAngles.x, rightOfGoalIndicationIsOnBorder[indexOfGoalIndication[i]]);
          }//
        }//!scanLineToCheckBestAngle[i]
      }//if(goalCheck.numberOfLargeParts > 0)
      else
      {
        if(scanLineToCheckBestAngle[i])
        {
          if(colorOfOpponentGoal == colorOfGoalScanLine[i])
            obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::opponentGoal] = false;
          else
            obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::ownGoal] = false;
        }
      }
    }//if(line.getNumberOfPixels() > 3)
  }//for(int i = 0; i < numberOfGoalScanLines; i++)


  Vector2<int> corner[4];
  int penWidth[4] = {3,3,3,3};

  for(i = 0; i < numberOfGoalIndications; i++)
  {
    Drawings::Color color;
    color = Drawings::dark_gray;

    if(useBoundaryOfIndication[i])
    {
      color = Drawings::light_gray;
      if(colorOfGoalIndication[i] == yellow) 
      {
        yellowGoalBoundary.add(boundaryOfIndication[i]);
        foundYellowGoal = true;
      }
      if(colorOfGoalIndication[i] == skyblue) 
      {
        skyblueGoalBoundary.add(boundaryOfIndication[i]);
        foundSkyblueGoal = true;
      }
    }

    (boundaryOfIndication[i].isOnBorder(boundaryOfIndication[i].y.max))? penWidth[0] = 1 : penWidth[0] = 3;
    (boundaryOfIndication[i].isOnBorder(boundaryOfIndication[i].x.min))? penWidth[1] = 1 : penWidth[1] = 3;
    (boundaryOfIndication[i].isOnBorder(boundaryOfIndication[i].y.min))? penWidth[2] = 1 : penWidth[2] = 3;
    (boundaryOfIndication[i].isOnBorder(boundaryOfIndication[i].x.max))? penWidth[3] = 1 : penWidth[3] = 3;

    Geometry::calculatePointByAngles(
      Vector2<double>(boundaryOfIndication[i].x.max,boundaryOfIndication[i].y.max),
      cameraMatrix, image.cameraInfo, 
      corner[0]
      );
    
    Geometry::calculatePointByAngles(
      Vector2<double>(boundaryOfIndication[i].x.min,boundaryOfIndication[i].y.max),
      cameraMatrix, image.cameraInfo, 
      corner[1]
      );
    
    Geometry::calculatePointByAngles(
      Vector2<double>(boundaryOfIndication[i].x.min,boundaryOfIndication[i].y.min),
      cameraMatrix, image.cameraInfo, 
      corner[2]
      );
    
    Geometry::calculatePointByAngles(
      Vector2<double>(boundaryOfIndication[i].x.max,boundaryOfIndication[i].y.min),
      cameraMatrix, image.cameraInfo, 
      corner[3]
      );

    LINE(imageProcessor_flagsAndGoals,
      (int)corner[0].x, (int)corner[0].y, 
      (int)corner[1].x, (int)corner[1].y,
      penWidth[0], Drawings::ps_solid, color);
    LINE(imageProcessor_flagsAndGoals,
      (int)corner[1].x, (int)corner[1].y, 
      (int)corner[2].x, (int)corner[2].y,
      penWidth[1], Drawings::ps_solid, color);
    LINE(imageProcessor_flagsAndGoals,
      (int)corner[2].x, (int)corner[2].y, 
      (int)corner[3].x, (int)corner[3].y,
      penWidth[2], Drawings::ps_solid, color);
    LINE(imageProcessor_flagsAndGoals,
      (int)corner[3].x, (int)corner[3].y, 
      (int)corner[0].x, (int)corner[0].y,
      penWidth[3], Drawings::ps_solid, color);
  }

  bool ownTeamColorIsBlue = getPlayer().getTeamColor() == Player::blue;
  if(
    foundYellowGoal && 
    yellowGoalBoundary.y.min < fromDegrees(0) &&
    (
    yellowGoalBoundary.x.getSize() > 1.5 * yellowGoalBoundary.y.getSize() ||
      yellowGoalBoundary.isOnBorder(yellowGoalBoundary.x.min) ||
      yellowGoalBoundary.isOnBorder(yellowGoalBoundary.x.max)
    )
    )
  {
    landmarksPercept.addGoal(yellow, ownTeamColorIsBlue, yellowGoalBoundary);
    Vector2<double> cameraOffset(cameraMatrix.translation.x, cameraMatrix.translation.y);
    landmarksPercept.estimateOffsetForGoals(cameraOffset); 
  }
  if(
    foundSkyblueGoal && 
    skyblueGoalBoundary.y.min < fromDegrees(0) &&
    (
    skyblueGoalBoundary.x.getSize() > 1.5 * skyblueGoalBoundary.y.getSize() ||
      skyblueGoalBoundary.isOnBorder(skyblueGoalBoundary.x.min) ||
      skyblueGoalBoundary.isOnBorder(skyblueGoalBoundary.x.max)
    )
    )
  {
    landmarksPercept.addGoal(skyblue, ownTeamColorIsBlue, skyblueGoalBoundary);
    Vector2<double> cameraOffset(cameraMatrix.translation.x, cameraMatrix.translation.y);
    landmarksPercept.estimateOffsetForGoals(cameraOffset); 
  }
}//scanLinesForGoals()

/*
* Change log :
* 
* $Log: GoalRecognizer.cpp,v $
* Revision 1.1.1.1  2004/05/22 17:19:49  cvsadm
* created new repository GT2004_WM
*
* Revision 1.7  2004/05/07 15:16:24  nistico
* All geometry calculations are making use of intrinsic functions.
* I updated most of the images processors to make correct use of this.
* Please test if there are any problems, because i'm going to remove the
* old code soon.
*
* Revision 1.6  2004/02/03 15:29:00  nistico
* GoalRecognizer is now ColorTable agnostic, no relevant adverse performance impact found
*
* Revision 1.5  2003/12/31 23:50:37  roefer
* Removed inclusion of RobotDimensions.h because it is not used
*
* Revision 1.4  2003/12/15 11:47:00  juengel
* Introduced CameraInfo
*
* Revision 1.3  2003/11/26 11:43:00  juengel
* new scan line calculation method added
*
* Revision 1.2  2003/10/23 07:24:20  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.8  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.7  2003/09/26 11:38:51  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.6  2003/09/01 10:16:17  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.5  2003/08/30 10:19:53  juengel
* DebugDrawings clean-up 1
*
* Revision 1.4  2003/08/29 13:15:05  juengel
* auto calibrating version uses level 2 colors.
*
* Revision 1.3  2003/08/18 12:06:59  juengel
* Changed usage of numberOfPixelsSinceLastOccurrenceOfGoalColor.
*
* Revision 1.2  2003/07/29 12:46:28  juengel
* Moved calculateHorizon to Geometry
*
* Revision 1.1  2003/07/21 13:40:24  juengel
* Moved ColorTableReferenceColor and GoalRecognizer to ImageProcessorTools.
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.2  2003/06/24 08:15:12  dueffert
* calculation optimized
*
* Revision 1.1  2003/06/12 16:49:50  juengel
* Added GoalRecognizer.
*
*
*/
